@extends('admin.layouts.admin')

@section('title', 'Order Details - ' . $order->order_number)

@section('content')
<div class="container-fluid">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1>Order #{{ $order->order_number }}</h1>
        <div>
            <a href="{{ route('admin.orders.invoice', $order) }}" class="btn btn-outline-primary" target="_blank">
                <i class="fas fa-print"></i> Print Invoice
            </a>
            <a href="{{ route('admin.orders.index') }}" class="btn btn-secondary">
                <i class="fas fa-arrow-left"></i> Back to Orders
            </a>
        </div>
    </div>

    {{-- Added: session alerts (from updated file) --}}
    @if(session('success'))
    <div class="alert alert-success alert-dismissible fade show" role="alert">
        {{ session('success') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    @endif

    @if(session('error'))
    <div class="alert alert-danger alert-dismissible fade show" role="alert">
        {{ session('error') }}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    </div>
    @endif
    {{-- end alerts --}}

    <div class="row">
        <!-- Order Info & Status -->
        <div class="col-md-8">
            <!-- Order Status Card -->
            <div class="card mb-4">
                <div class="card-header bg-primary text-white">
                    <h5 class="mb-0">Order Status & Information</h5>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-6">
                            <p><strong>Order Date:</strong> {{ $order->created_at->format('F d, Y h:i A') }}</p>
                            <p><strong>Payment Method:</strong> {{ ucfirst($order->payment_method ?? 'N/A') }}</p>
                            <p><strong>Payment Status:</strong> 
                                <span class="badge bg-{{ $order->payment_status_badge }}">
                                    {{ ucfirst($order->payment_status) }}
                                </span>
                            </p>
                            @if($order->transaction_id)
                            <p><strong>Transaction ID:</strong> {{ $order->transaction_id }}</p>
                            @endif
                        </div>
                        <div class="col-md-6">
                            <form action="{{ route('admin.orders.update-status', $order) }}" method="POST">
                                @csrf
                                @method('PATCH')
                                <div class="mb-3">
                                    <label class="form-label"><strong>Order Status:</strong></label>
                                    <select name="status" class="form-select" onchange="this.form.submit()">
                                        <option value="pending" {{ $order->status == 'pending' ? 'selected' : '' }}>Pending</option>
                                        <option value="processing" {{ $order->status == 'processing' ? 'selected' : '' }}>Processing</option>
                                        <option value="shipped" {{ $order->status == 'shipped' ? 'selected' : '' }}>Shipped</option>
                                        <option value="out_for_delivery" {{ $order->status == 'out_for_delivery' ? 'selected' : '' }}>Out for Delivery</option>
                                        <option value="delivered" {{ $order->status == 'delivered' ? 'selected' : '' }}>Delivered</option>
                                        <option value="cancelled" {{ $order->status == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                                        <option value="refunded" {{ $order->status == 'refunded' ? 'selected' : '' }}>Refunded</option>
                                    </select>
                                </div>
                            </form>
                            @if($order->shipped_at)
                            <p><strong>Shipped Date:</strong> {{ $order->shipped_at->format('F d, Y') }}</p>
                            @endif
                            @if($order->delivered_at)
                            <p><strong>Delivered Date:</strong> {{ $order->delivered_at->format('F d, Y') }}</p>
                            @endif
                        </div>
                    </div>
                    @if($order->notes)
                    <hr>
                    <p><strong>Customer Notes:</strong></p>
                    <p class="text-muted">{{ $order->notes }}</p>
                    @endif
                </div>
            </div>

            <!-- Tracking Information -->
            <div class="card mb-4">
                <div class="card-header bg-info text-white">
                    <h5 class="mb-0">Tracking Information</h5>
                </div>
                <div class="card-body">
                    {{-- Keep existing tracking form (old) to avoid duplicate routes --}}
                    <form action="{{ route('admin.orders.tracking', $order) }}" method="POST">
                        @csrf
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Tracking Number</label>
                                    <input type="text" name="tracking_number" class="form-control" 
                                           value="{{ $order->tracking_number }}" placeholder="Enter tracking number">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Carrier</label>
                                    <select name="carrier" class="form-select">
                                        <option value="">Select Carrier</option>
                                        <option value="fedex" {{ $order->carrier == 'fedex' ? 'selected' : '' }}>FedEx</option>
                                        <option value="ups" {{ $order->carrier == 'ups' ? 'selected' : '' }}>UPS</option>
                                        <option value="usps" {{ $order->carrier == 'usps' ? 'selected' : '' }}>USPS</option>
                                        <option value="dhl" {{ $order->carrier == 'dhl' ? 'selected' : '' }}>DHL</option>
                                        <option value="other" {{ $order->carrier == 'other' ? 'selected' : '' }}>Other</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label class="form-label">Estimated Delivery</label>
                                    <input type="date" name="estimated_delivery" class="form-control" 
                                           value="{{ $order->estimated_delivery ? $order->estimated_delivery->format('Y-m-d') : '' }}">
                                </div>
                            </div>
                            <div class="col-md-12">
                                <div class="mb-3">
                                    <label class="form-label">Tracking Update Message *</label>
                                    <textarea name="update_message" class="form-control" rows="2" required 
                                              placeholder="e.g., Package picked up by carrier, In transit to destination..."></textarea>
                                </div>
                            </div>
                            <div class="col-md-12">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save"></i> Update Tracking
                                </button>
                            </div>
                        </div>
                    </form>

                    @if($order->tracking_updates && count($order->tracking_updates) > 0)
                    <hr>
                    <h6>Tracking History</h6>
                    <div class="timeline">
                        @foreach(array_reverse($order->tracking_updates) as $update)
                        <div class="timeline-item">
                            {{-- marker and improved timestamp parsing (accepts timestamp or date) --}}
                            <div class="timeline-marker"></div>
                            <div class="timeline-content">
                                <small class="text-muted">
                                    {{ \Carbon\Carbon::parse($update['timestamp'] ?? $update['date'] ?? now())->format('M d, Y h:i A') }}
                                    @if(isset($update['user']))
                                        - by {{ $update['user'] }}
                                    @endif
                                </small>
                                <p class="mb-0">{{ $update['message'] ?? '' }}</p>
                                @if(isset($update['status']))
                                    <div class="mt-1"><small class="text-muted">Status: {{ ucfirst($update['status']) }}</small></div>
                                @endif
                            </div>
                        </div>
                        @endforeach
                    </div>
                    @endif
                </div>
            </div>

            <!-- Order Items -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0">Order Items</h5>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Product</th>
                                    <th>SKU</th>
                                    <th>Price</th>
                                    <th>Quantity</th>
                                    <th>Total</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($order->items as $item)
                                <tr>
                                    <td>
                                        <div class="d-flex align-items-center">
                                            {{-- Added: product image (from updated file) but keep link behavior --}}
                                            @if(isset($item->product_image) && $item->product_image)
                                                <div class="me-2">
                                                    @if($item->product && isset($item->product->slug))
                                                        <a href="{{ route('shop.product', $item->product->slug) }}" target="_blank">
                                                            <img src="{{ Storage::url($item->product_image) }}" alt="{{ $item->product_name }}" style="width:50px;height:50px;object-fit:cover;">
                                                        </a>
                                                    @else
                                                        <img src="{{ Storage::url($item->product_image) }}" alt="{{ $item->product_name }}" style="width:50px;height:50px;object-fit:cover;">
                                                    @endif
                                                </div>
                                            @endif

                                            <div>
                                                @if($item->product)
                                                    <a href="{{ route('shop.product', $item->product->slug) }}" target="_blank">
                                                        {{ $item->product_name }}
                                                    </a>
                                                @else
                                                    {{ $item->product_name }}
                                                @endif

                                                @if($item->attributes)
                                                <br>
                                                <small class="text-muted">
                                                    @foreach($item->attributes as $key => $value)
                                                        {{ ucfirst($key) }}: {{ $value }}@if(!$loop->last), @endif
                                                    @endforeach
                                                </small>
                                                @endif
                                            </div>
                                        </div>
                                    </td>
                                    <td>{{ $item->product_sku }}</td>
                                    <td>${{ number_format($item->price, 2) }}</td>
                                    <td>{{ $item->quantity }}</td>
                                    <td>${{ number_format($item->total, 2) }}</td>
                                </tr>
                                @endforeach
                            </tbody>
                            <tfoot>
                                <tr>
                                    <td colspan="4" class="text-end"><strong>Subtotal:</strong></td>
                                    <td><strong>${{ number_format($order->subtotal, 2) }}</strong></td>
                                </tr>
                                @if($order->discount_amount > 0)
                                <tr>
                                    <td colspan="4" class="text-end">Discount:</td>
                                    <td>-${{ number_format($order->discount_amount, 2) }}</td>
                                </tr>
                                @endif
                                <tr>
                                    <td colspan="4" class="text-end">Tax:</td>
                                    <td>${{ number_format($order->tax_amount, 2) }}</td>
                                </tr>
                                <tr>
                                    <td colspan="4" class="text-end">Shipping:</td>
                                    <td>${{ number_format($order->shipping_amount, 2) }}</td>
                                </tr>
                                <tr class="table-active">
                                    <td colspan="4" class="text-end"><strong>Total:</strong></td>
                                    <td><strong>${{ number_format($order->total_amount, 2) }}</strong></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        </div>

        <!-- Customer & Shipping Info -->
        <div class="col-md-4">
            <!-- Customer Information -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0">Customer Information</h5>
                </div>
                <div class="card-body">
                    <p><strong>Name:</strong> {{ $order->customer_name }}</p>
                    <p><strong>Email:</strong> {{ $order->customer_email }}</p>
                    <p><strong>Phone:</strong> {{ $order->customer_phone ?? 'Not provided' }}</p>
                    @if($order->user)
                    <p><strong>Registered User:</strong> Yes</p>
                    <a href="{{ route('admin.customers.show', $order->user) }}" class="btn btn-sm btn-outline-primary">
                        View Customer Profile
                    </a>
                    @else
                    <p><strong>Registered User:</strong> No (Guest Checkout)</p>
                    @endif
                </div>
            </div>

            <!-- Shipping Address -->
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0">Shipping Address</h5>
                </div>
                <div class="card-body">
                    @if($order->shippingAddress)
                    <address>
                        {{ $order->shippingAddress->first_name }} {{ $order->shippingAddress->last_name }}<br>
                        @if($order->shippingAddress->company)
                            {{ $order->shippingAddress->company }}<br>
                        @endif
                        {{ $order->shippingAddress->address_line_1 }}<br>
                        @if($order->shippingAddress->address_line_2)
                            {{ $order->shippingAddress->address_line_2 }}<br>
                        @endif
                        {{ $order->shippingAddress->city }}, {{ $order->shippingAddress->state }} 
                        {{ $order->shippingAddress->postal_code ?? $order->shippingAddress->postcode }}<br>
                        {{ $order->shippingAddress->country }}
                    </address>
                    @else
                    <p>No shipping address provided</p>
                    @endif
                </div>
            </div>

            <!-- Quick Actions -->
            <div class="card">
                <div class="card-header">
                    <h5 class="mb-0">Quick Actions</h5>
                </div>
                <div class="card-body">
                    @if($order->canBeCancelled())
                    <form action="{{ route('admin.orders.cancel', $order) }}" method="POST" class="d-inline">
                        @csrf
                        <button type="submit" class="btn btn-warning w-100 mb-2" 
                                onclick="return confirm('Are you sure you want to cancel this order?')">
                            <i class="fas fa-times"></i> Cancel Order
                        </button>
                    </form>
                    @endif

                    @if($order->canBeRefunded())
                    <button type="button" class="btn btn-danger w-100 mb-2" onclick="processRefund()">
                        <i class="fas fa-undo"></i> Process Refund
                    </button>
                    @endif

                    <button type="button" class="btn btn-info w-100" onclick="sendEmail()">
                        <i class="fas fa-envelope"></i> Send Email to Customer
                    </button>
                </div>
            </div>

            {{-- Notes form: use admin.orders.updateNotes if present, otherwise fall back to admin.orders.update-status --}}
            <div class="card mt-3">
                <div class="card-header">
                    <h5>Order Notes</h5>
                </div>
                <div class="card-body">
                    <form action="{{ Route::has('admin.orders.updateNotes') ? route('admin.orders.updateNotes', $order) : route('admin.orders.update-status', $order) }}" method="POST">
                        @csrf
                        @unless(Route::has('admin.orders.updateNotes'))
                            {{-- fallback uses the existing update-status route which expects PATCH --}}
                            @method('PATCH')
                        @endunless

                        <div class="mb-3">
                            <textarea name="notes" class="form-control" rows="4" 
                                      placeholder="Add internal notes about this order...">{{ $order->notes }}</textarea>
                        </div>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Save Notes
                        </button>
                    </form>
                </div>
            </div>
            {{-- end Notes --}}
        </div>
    </div>
</div>

<style>
/* Merged timeline styling - kept original structure and added marker styling (no duplicate selectors) */
.timeline {
    position: relative;
    padding-left: 30px;
}
.timeline-item {
    position: relative;
    padding-bottom: 20px;
}
.timeline-item:before {
    content: '';
    position: absolute;
    left: -21px;
    top: 20px;
    height: 100%;
    width: 2px;
    background: #dee2e6;
}
.timeline-item:last-child:before {
    display: none;
}
/* marker style (from updated file) */
.timeline-marker {
    position: absolute;
    left: -25px;
    top: 5px;
    width: 10px;
    height: 10px;
    border-radius: 50%;
    background: #007bff;
    border: 2px solid #fff;
    box-shadow: 0 0 0 2px #dee2e6;
}
</style>

@endsection

@push('scripts')
<script>
function processRefund() {
    if (confirm('Are you sure you want to refund this order? This action cannot be undone.')) {
        // Implement refund logic
        alert('Refund functionality to be implemented');
    }
}

function sendEmail() {
    // Implement email functionality
    alert('Email functionality to be implemented');
}
</script>
@endpush