@extends('admin.layouts.admin')

@section('title', 'Create Style Prediction')

@section('content')
<div class="container-fluid">
    <!-- Page Header -->
    <div class="d-flex justify-content-between align-items-center mb-4">
        <div>
            <h1 class="h3">Create New Style Prediction</h1>
            <p class="text-muted">Add a new fashion trend prediction for future forecasting</p>
        </div>
        <a href="{{ route('admin.trends.predictions') }}" class="btn btn-secondary">
            <i class="fas fa-arrow-left me-2"></i>Back to Predictions
        </a>
    </div>

    <!-- Form -->
    <div class="row">
        <div class="col-lg-8">
            <form action="{{ route('admin.trends.store-predictions') }}" method="POST" enctype="multipart/form-data">
                @csrf
                
                <div class="card">
                    <div class="card-header">
                        <h5 class="mb-0">Prediction Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Title -->
                        <div class="mb-3">
                            <label for="title" class="form-label">Prediction Title *</label>
                            <input type="text" class="form-control @error('title') is-invalid @enderror" 
                                   id="title" name="title" value="{{ old('title') }}" 
                                   placeholder="e.g., Smart Fabric Integration Revolution">
                            <div class="form-text">Create a compelling title that captures the essence of your prediction</div>
                            @error('title')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Description -->
                        <div class="mb-3">
                            <label for="description" class="form-label">Description *</label>
                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                      id="description" name="description" rows="4" 
                                      placeholder="Describe the prediction and its potential impact on the fashion industry...">{{ old('description') }}</textarea>
                            <div class="form-text">Explain what the prediction entails and why it matters</div>
                            @error('description')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Image Upload -->
                        <div class="mb-3">
                            <label for="image" class="form-label">Prediction Image *</label>
                            <input type="file" class="form-control @error('image') is-invalid @enderror" 
                                   id="image" name="image" accept="image/*">
                            <div class="form-text">
                                Upload a high-quality image representing the prediction (JPEG, PNG, JPG, GIF). Recommended size: 800x600px
                            </div>
                            @error('image')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                            
                            <!-- Image Preview -->
                            <div id="imagePreview" class="mt-3" style="display: none;">
                                <img id="previewImg" src="" class="img-thumbnail" style="max-width: 300px;">
                            </div>
                        </div>

                        <div class="row">
                            <!-- Category -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="category" class="form-label">Category *</label>
                                    <select class="form-select @error('category') is-invalid @enderror" 
                                            id="category" name="category">
                                        <option value="">Select Category</option>
                                        <option value="Technology" {{ old('category') == 'Technology' ? 'selected' : '' }}>
                                            💻 Technology
                                        </option>
                                        <option value="Sustainability" {{ old('category') == 'Sustainability' ? 'selected' : '' }}>
                                            🌱 Sustainability
                                        </option>
                                        <option value="Materials" {{ old('category') == 'Materials' ? 'selected' : '' }}>
                                            🧬 Materials
                                        </option>
                                        <option value="Design Trends" {{ old('category') == 'Design Trends' ? 'selected' : '' }}>
                                            🎨 Design Trends
                                        </option>
                                    </select>
                                    @error('category')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Year -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="year" class="form-label">Target Year *</label>
                                    <input type="number" class="form-control @error('year') is-invalid @enderror" 
                                           id="year" name="year" value="{{ old('year', date('Y') + 1) }}" 
                                           min="2024" max="2030">
                                    <div class="form-text">When do you predict this trend will become mainstream?</div>
                                    @error('year')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <!-- Confidence Level -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="confidence" class="form-label">Confidence Level *</label>
                                    <div class="row">
                                        <div class="col-8">
                                            <input type="range" class="form-range" id="confidenceRange" 
                                                   min="0" max="100" value="{{ old('confidence', 75) }}" 
                                                   oninput="updateConfidenceValue(this.value)">
                                        </div>
                                        <div class="col-4">
                                            <div class="input-group">
                                                <input type="number" class="form-control @error('confidence') is-invalid @enderror" 
                                                       id="confidence" name="confidence" min="0" max="100" 
                                                       value="{{ old('confidence', 75) }}" 
                                                       oninput="updateConfidenceRange(this.value)">
                                                <span class="input-group-text">%</span>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="form-text">How confident are you in this prediction?</div>
                                    @error('confidence')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <!-- Impact Level -->
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="impact_level" class="form-label">Impact Level *</label>
                                    <select class="form-select @error('impact_level') is-invalid @enderror" 
                                            id="impact_level" name="impact_level">
                                        <option value="">Select Impact Level</option>
                                        <option value="Revolutionary" {{ old('impact_level') == 'Revolutionary' ? 'selected' : '' }}>
                                            🚀 Revolutionary - Game changing impact
                                        </option>
                                        <option value="Significant" {{ old('impact_level') == 'Significant' ? 'selected' : '' }}>
                                            ⭐ Significant - Notable industry impact
                                        </option>
                                        <option value="Moderate" {{ old('impact_level') == 'Moderate' ? 'selected' : '' }}>
                                            📊 Moderate - Gradual influence
                                        </option>
                                    </select>
                                    @error('impact_level')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Timeline -->
                        <div class="mb-3">
                            <label for="timeline" class="form-label">Timeline *</label>
                            <input type="text" class="form-control @error('timeline') is-invalid @enderror" 
                                   id="timeline" name="timeline" value="{{ old('timeline') }}" 
                                   placeholder="e.g., Next 12-18 Months, Within 2 Years, Next 5 Years">
                            <div class="form-text">When do you expect this prediction to materialize?</div>
                            @error('timeline')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Impact Areas -->
                        <div class="mb-3">
                            <label class="form-label">Impact Areas *</label>
                            <div id="impactAreasContainer">
                                <div class="input-group mb-2">
                                    <input type="text" class="form-control" name="impact_areas[]" 
                                           placeholder="e.g., Manufacturing Processes" value="{{ old('impact_areas.0') }}">
                                    <button class="btn btn-outline-danger" type="button" onclick="removeImpactArea(this)">
                                        <i class="fas fa-times"></i>
                                    </button>
                                </div>
                                @if(old('impact_areas'))
                                    @foreach(array_slice(old('impact_areas'), 1) as $area)
                                    <div class="input-group mb-2">
                                        <input type="text" class="form-control" name="impact_areas[]" 
                                               placeholder="e.g., Consumer Behavior" value="{{ $area }}">
                                        <button class="btn btn-outline-danger" type="button" onclick="removeImpactArea(this)">
                                            <i class="fas fa-times"></i>
                                        </button>
                                    </div>
                                    @endforeach
                                @endif
                            </div>
                            <button type="button" class="btn btn-sm btn-outline-primary" onclick="addImpactArea()">
                                <i class="fas fa-plus me-1"></i>Add Another Area
                            </button>
                            <div class="form-text">Areas of the fashion industry that will be impacted</div>
                            @error('impact_areas')
                                <div class="text-danger small mt-1">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>

                <!-- Action Buttons -->
                <div class="card mt-3">
                    <div class="card-body">
                        <div class="d-flex gap-2">
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-2"></i>Create Prediction
                            </button>
                            <button type="button" class="btn btn-secondary" onclick="saveDraft()">
                                <i class="fas fa-file-alt me-2"></i>Save Draft
                            </button>
                            <a href="{{ route('admin.trends.predictions') }}" class="btn btn-outline-secondary">
                                Cancel
                            </a>
                        </div>
                    </div>
                </div>
            </form>
        </div>

        <!-- Preview Card -->
        <div class="col-lg-4">
            <div class="card sticky-top" style="top: 20px;">
                <div class="card-header">
                    <h6 class="mb-0">Live Preview</h6>
                </div>
                <div class="card-body">
                    <div id="previewCard">
                        <div class="preview-image mb-3" style="height: 200px; background: #f8f9fa; border-radius: 8px; display: flex; align-items: center; justify-content: center; color: #6c757d;">
                            <i class="fas fa-crystal-ball fa-3x"></i>
                        </div>
                        <div class="preview-badges mb-2">
                            <span class="badge bg-secondary" id="previewCategory">Select Category</span>
                            <span class="badge bg-info" id="previewYear">{{ date('Y') + 1 }}</span>
                            <span class="badge bg-success" id="previewConfidence">75%</span>
                        </div>
                        <h6 id="previewTitle" class="text-muted">Enter prediction title...</h6>
                        <p id="previewDescription" class="text-muted small">Enter prediction description...</p>
                        
                        <div class="mt-3">
                            <div class="d-flex justify-content-between align-items-center mb-2">
                                <small class="fw-bold">Impact Level:</small>
                                <span class="badge bg-secondary" id="previewImpact">Select Level</span>
                            </div>
                            <div class="d-flex justify-content-between align-items-center">
                                <small class="fw-bold">Timeline:</small>
                                <small class="text-muted" id="previewTimeline">Enter timeline...</small>
                            </div>
                        </div>

                        <div id="previewImpactAreas" class="mt-3">
                            <h6 class="small fw-bold">Impact Areas:</h6>
                            <div class="preview-areas"></div>
                        </div>
                    </div>

                    <!-- Tips -->
                    <div class="border-top pt-3 mt-3">
                        <h6 class="small fw-bold mb-2">Prediction Tips</h6>
                        <ul class="small text-muted mb-0">
                            <li>Base predictions on current market trends</li>
                            <li>Consider technological advancements</li>
                            <li>Factor in consumer behavior changes</li>
                            <li>Include sustainability considerations</li>
                            <li>Reference industry expert opinions</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
// Image preview functionality
document.getElementById('image').addEventListener('change', function(e) {
    const file = e.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            document.getElementById('previewImg').src = e.target.result;
            document.getElementById('imagePreview').style.display = 'block';
            
            // Update live preview
            const previewImage = document.querySelector('.preview-image');
            previewImage.innerHTML = `<img src="${e.target.result}" style="width: 100%; height: 100%; object-fit: cover; border-radius: 8px;">`;
        };
        reader.readAsDataURL(file);
    }
});

// Confidence synchronization
function updateConfidenceValue(value) {
    document.getElementById('confidence').value = value;
    document.getElementById('previewConfidence').textContent = value + '%';
}

function updateConfidenceRange(value) {
    document.getElementById('confidenceRange').value = value;
    document.getElementById('previewConfidence').textContent = value + '%';
}

// Impact areas management
function addImpactArea() {
    const container = document.getElementById('impactAreasContainer');
    const newDiv = document.createElement('div');
    newDiv.className = 'input-group mb-2';
    newDiv.innerHTML = `
        <input type="text" class="form-control" name="impact_areas[]" placeholder="e.g., Supply Chain Innovation" oninput="updatePreview()">
        <button class="btn btn-outline-danger" type="button" onclick="removeImpactArea(this)">
            <i class="fas fa-times"></i>
        </button>
    `;
    container.appendChild(newDiv);
    updatePreview();
}

function removeImpactArea(button) {
    if (document.querySelectorAll('#impactAreasContainer .input-group').length > 1) {
        button.parentElement.remove();
        updatePreview();
    }
}

// Live preview updates
function updatePreview() {
    // Update title
    const title = document.getElementById('title').value || 'Enter prediction title...';
    document.getElementById('previewTitle').textContent = title;
    
    // Update description
    const description = document.getElementById('description').value || 'Enter prediction description...';
    document.getElementById('previewDescription').textContent = description.substring(0, 100) + (description.length > 100 ? '...' : '');
    
    // Update category
    const category = document.getElementById('category').value || 'Select Category';
    const categoryBadge = document.getElementById('previewCategory');
    categoryBadge.textContent = category;
    categoryBadge.className = `badge ${getCategoryClass(category)}`;
    
    // Update year
    const year = document.getElementById('year').value || '{{ date('Y') + 1 }}';
    document.getElementById('previewYear').textContent = year;
    
    // Update impact level
    const impact = document.getElementById('impact_level').value || 'Select Level';
    const impactBadge = document.getElementById('previewImpact');
    impactBadge.textContent = impact;
    impactBadge.className = `badge ${getImpactClass(impact)}`;
    
    // Update timeline
    const timeline = document.getElementById('timeline').value || 'Enter timeline...';
    document.getElementById('previewTimeline').textContent = timeline;
    
    // Update impact areas
    const impactAreas = Array.from(document.querySelectorAll('input[name="impact_areas[]"]'))
                            .map(input => input.value)
                            .filter(value => value.trim() !== '');
    
    const areasContainer = document.querySelector('.preview-areas');
    areasContainer.innerHTML = impactAreas.map(area => 
        `<span class="badge bg-light text-dark me-1 mb-1">${area}</span>`
    ).join('') || '<span class="text-muted small">No areas defined yet</span>';
}

function getCategoryClass(category) {
    switch(category) {
        case 'Technology': return 'bg-primary';
        case 'Sustainability': return 'bg-success';
        case 'Materials': return 'bg-info';
        case 'Design Trends': return 'bg-warning text-dark';
        default: return 'bg-secondary';
    }
}

function getImpactClass(impact) {
    switch(impact) {
        case 'Revolutionary': return 'bg-danger';
        case 'Significant': return 'bg-warning text-dark';
        case 'Moderate': return 'bg-info';
        default: return 'bg-secondary';
    }
}

// Add event listeners for live preview
document.addEventListener('DOMContentLoaded', function() {
    document.getElementById('title').addEventListener('input', updatePreview);
    document.getElementById('description').addEventListener('input', updatePreview);
    document.getElementById('category').addEventListener('change', updatePreview);
    document.getElementById('year').addEventListener('input', updatePreview);
    document.getElementById('impact_level').addEventListener('change', updatePreview);
    document.getElementById('timeline').addEventListener('input', updatePreview);
    
    // Add listeners to existing impact area inputs
    document.querySelectorAll('input[name="impact_areas[]"]').forEach(input => {
        input.addEventListener('input', updatePreview);
    });
    
    // Initial preview update
    updatePreview();
});

// Save draft functionality
function saveDraft() {
    alert('Draft saved to browser storage!');
}
</script>
@endsection