@extends('layouts.app')

@section('title', 'Complete Payment - Order #' . $order->order_number)

@section('content')
<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-lg-8">
            <div class="card shadow-lg">
                <div class="card-header bg-primary text-white">
                    <h4 class="mb-0">Complete Your Payment</h4>
                </div>
                <div class="card-body p-4">
                    <!-- Order Summary -->
                    <div class="order-summary mb-4">
                        <h5 class="mb-3">Order Summary</h5>
                        <table class="table table-sm">
                            <tr>
                                <td>Order Number:</td>
                                <td class="text-end"><strong>#{{ $order->order_number }}</strong></td>
                            </tr>
                            <tr>
                                <td>Total Amount:</td>
                                <td class="text-end"><strong>${{ number_format($order->total_amount, 2) }}</strong></td>
                            </tr>
                        </table>
                    </div>

                    <!-- Stripe Payment Form -->
                    <form id="payment-form">
                        <div id="payment-element" class="mb-3">
                            <!-- Stripe Elements will be inserted here -->
                        </div>
                        
                        <div id="payment-message" class="alert d-none"></div>
                        
                        <button id="submit" class="btn btn-primary btn-lg w-100">
                            <span id="button-text">
                                <i class="fas fa-lock me-2"></i>Pay ${{ number_format($order->total_amount, 2) }}
                            </span>
                            <span id="spinner" class="d-none">
                                <span class="spinner-border spinner-border-sm me-2" role="status"></span>
                                Processing...
                            </span>
                        </button>
                    </form>

                    <!-- Security Badges -->
                    <div class="text-center mt-4">
                        <p class="text-muted mb-2">
                            <i class="fas fa-shield-alt text-success me-2"></i>
                            Your payment information is secure and encrypted
                        </p>
                        <img src="https://img.shields.io/badge/Secured%20by-Stripe-green" alt="Secured by Stripe">
                    </div>
                </div>
            </div>

            <!-- Additional Information -->
            <div class="card mt-3">
                <div class="card-body">
                    <h6 class="mb-3">What happens next?</h6>
                    <ul class="mb-0">
                        <li>You'll receive an order confirmation email immediately</li>
                        <li>We'll process your order within 1-2 business days</li>
                        <li>You'll receive tracking information once shipped</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script src="https://js.stripe.com/v3/"></script>
<script>
// Initialize Stripe
const stripe = Stripe('{{ $stripeKey }}');

// Stripe Elements
let elements;

initialize();
checkStatus();

document
  .querySelector("#payment-form")
  .addEventListener("submit", handleSubmit);

async function initialize() {
  const response = await fetch("/stripe/create-payment-intent", {
    method: "POST",
    headers: { 
      "Content-Type": "application/json",
      "X-CSRF-TOKEN": "{{ csrf_token() }}"
    },
    body: JSON.stringify({ 
      amount: {{ $order->total_amount * 100 }},
      order_id: "{{ $order->id }}"
    }),
  });
  
  const { clientSecret } = { clientSecret: "{{ $clientSecret }}" };

  const appearance = {
    theme: 'stripe',
    variables: {
      colorPrimary: '#0570de',
      colorBackground: '#ffffff',
      colorText: '#30313d',
      colorDanger: '#df1b41',
      fontFamily: 'system-ui, sans-serif',
      spacingUnit: '3px',
      borderRadius: '4px',
    }
  };
  
  elements = stripe.elements({ appearance, clientSecret });

  const paymentElementOptions = {
    layout: "tabs"
  };

  const paymentElement = elements.create("payment", paymentElementOptions);
  paymentElement.mount("#payment-element");
}

async function handleSubmit(e) {
  e.preventDefault();
  setLoading(true);

  const { error } = await stripe.confirmPayment({
    elements,
    confirmParams: {
      return_url: "{{ route('checkout.confirm-payment', $order->order_number) }}",
      receipt_email: "{{ $order->customer_email }}",
    },
    redirect: "if_required"
  });

  if (error) {
    if (error.type === "card_error" || error.type === "validation_error") {
      showMessage(error.message);
    } else {
      showMessage("An unexpected error occurred.");
    }
    setLoading(false);
  } else {
    // Payment succeeded
    window.location.href = "{{ route('checkout.success', $order->order_number) }}?payment_status=succeeded&payment_intent_id=" + error.payment_intent.id;
  }
}

async function checkStatus() {
  const clientSecret = new URLSearchParams(window.location.search).get(
    "payment_intent_client_secret"
  );

  if (!clientSecret) {
    return;
  }

  const { paymentIntent } = await stripe.retrievePaymentIntent(clientSecret);

  switch (paymentIntent.status) {
    case "succeeded":
      showMessage("Payment succeeded!");
      window.location.href = "{{ route('checkout.success', $order->order_number) }}?payment_status=succeeded";
      break;
    case "processing":
      showMessage("Your payment is processing.");
      break;
    case "requires_payment_method":
      showMessage("Your payment was not successful, please try again.");
      break;
    default:
      showMessage("Something went wrong.");
      break;
  }
}

function showMessage(messageText) {
  const messageContainer = document.querySelector("#payment-message");
  messageContainer.classList.remove("d-none");
  messageContainer.classList.add("alert-danger");
  messageContainer.textContent = messageText;

  setTimeout(function () {
    messageContainer.classList.add("d-none");
    messageContainer.textContent = "";
  }, 4000);
}

function setLoading(isLoading) {
  if (isLoading) {
    document.querySelector("#submit").disabled = true;
    document.querySelector("#spinner").classList.remove("d-none");
    document.querySelector("#button-text").classList.add("d-none");
  } else {
    document.querySelector("#submit").disabled = false;
    document.querySelector("#spinner").classList.add("d-none");
    document.querySelector("#button-text").classList.remove("d-none");
  }
}
</script>
@endpush
@endsection