@extends('layouts.app')

@section('title', 'Order Tracking - ' . $order->order_number)

@section('content')
<div class="tracking-result-premium">
    <!-- Result Header -->
    <section class="result-header-premium">
        <div class="container">
            <div class="tracking-found" data-aos="fade-up">
                <div class="success-icon">
                    <i class="fas fa-check-circle"></i>
                </div>
                <h1>Order Found!</h1>
                <p>Tracking information for order #{{ $order->order_number }}</p>
            </div>
        </div>
    </section>

    <!-- Order Status Timeline -->
    <section class="tracking-timeline-section">
        <div class="container">
            <div class="timeline-card" data-aos="zoom-in">
                <div class="timeline-header">
                    <h3>Order Status: <span class="status-badge {{ $order->status }}">{{ ucfirst(str_replace('_', ' ', $order->status)) }}</span></h3>
                    @if($order->estimated_delivery)
                    <p class="delivery-date">
                        <i class="fas fa-calendar-alt"></i>
                        Estimated Delivery: {{ \Carbon\Carbon::parse($order->estimated_delivery)->format('F d, Y') }}
                    </p>
                    @endif
                </div>

                <div class="timeline-progress">
                    <div class="progress-bar-container">
                        <div class="progress-bar-fill" style="width: {{ $order->status == 'delivered' ? '100' : ($order->status == 'out_for_delivery' ? '75' : ($order->status == 'shipped' ? '50' : ($order->status == 'processing' ? '25' : '10'))) }}%"></div>
                    </div>
                    
                    <div class="timeline-steps">
                        <div class="step {{ in_array($order->status, ['pending', 'processing', 'shipped', 'out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                            <div class="step-icon">
                                <i class="fas fa-clipboard-check"></i>
                            </div>
                            <div class="step-title">Order Placed</div>
                            <div class="step-date">{{ $order->created_at->format('M d, H:i') }}</div>
                        </div>

                        <div class="step {{ in_array($order->status, ['processing', 'shipped', 'out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                            <div class="step-icon">
                                <i class="fas fa-box"></i>
                            </div>
                            <div class="step-title">Processing</div>
                            @if($order->status != 'pending')
                            <div class="step-date">In Progress</div>
                            @endif
                        </div>

                        <div class="step {{ in_array($order->status, ['shipped', 'out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                            <div class="step-icon">
                                <i class="fas fa-truck"></i>
                            </div>
                            <div class="step-title">Shipped</div>
                            @if($order->shipped_at)
                            <div class="step-date">{{ \Carbon\Carbon::parse($order->shipped_at)->format('M d, H:i') }}</div>
                            @endif
                        </div>

                        <div class="step {{ in_array($order->status, ['out_for_delivery', 'delivered']) ? 'completed' : '' }}">
                            <div class="step-icon">
                                <i class="fas fa-shipping-fast"></i>
                            </div>
                            <div class="step-title">Out for Delivery</div>
                        </div>

                        <div class="step {{ $order->status == 'delivered' ? 'completed' : '' }}">
                            <div class="step-icon">
                                <i class="fas fa-home"></i>
                            </div>
                            <div class="step-title">Delivered</div>
                            @if($order->delivered_at)
                            <div class="step-date">{{ \Carbon\Carbon::parse($order->delivered_at)->format('M d, H:i') }}</div>
                            @endif
                        </div>
                    </div>
                </div>

                @if($order->tracking_number)
                <div class="tracking-number-display">
                    <i class="fas fa-barcode"></i>
                    <span>Tracking Number:</span>
                    <strong>{{ $order->tracking_number }}</strong>
                    @if($order->carrier)
                    <span class="carrier-badge">{{ $order->carrier }}</span>
                    @endif
                </div>
                @endif
            </div>
        </div>
    </section>

    <!-- Tracking Updates -->
    @if($trackingUpdates && count($trackingUpdates) > 0)
    <section class="tracking-updates-section">
        <div class="container">
            <div class="updates-card" data-aos="fade-up">
                <h3 class="updates-title">
                    <i class="fas fa-history"></i>
                    Tracking History
                </h3>
                <div class="updates-timeline">
                    @foreach($trackingUpdates as $update)
                    <div class="update-item">
                        <div class="update-marker"></div>
                        <div class="update-content">
                            <div class="update-time">
                                {{ \Carbon\Carbon::parse($update['timestamp'] ?? $update['created_at'])->format('M d, Y H:i') }}
                            </div>
                            <div class="update-status">{{ $update['status'] ?? $update['message'] }}</div>
                            @if(isset($update['location']))
                            <div class="update-location">
                                <i class="fas fa-map-marker-alt"></i>
                                {{ $update['location'] }}
                            </div>
                            @endif
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
    </section>
    @endif

    <!-- Order Details -->
    <section class="order-details-section">
        <div class="container">
            <div class="row">
                <!-- Order Items -->
                <div class="col-lg-8">
                    <div class="details-card" data-aos="fade-right">
                        <h3 class="card-title">
                            <i class="fas fa-shopping-bag"></i>
                            Order Items
                        </h3>
                        <div class="items-list">
                            @foreach($order->items as $item)
                            <div class="item-row">
                                <div class="item-image">
                                    @if($item->product && $item->product->featured_image)
                                    <img src="{{ \App\Helpers\ImageHelper::getImageUrl($item->product->featured_image) }}" 
                                         alt="{{ $item->product->name }}">
                                    @else
                                    <div class="placeholder-image">
                                        <i class="fas fa-image"></i>
                                    </div>
                                    @endif
                                </div>
                                <div class="item-details">
                                    <h4>{{ $item->product ? $item->product->name : 'Product' }}</h4>
                                    <div class="item-meta">
                                        <span>Qty: {{ $item->quantity }}</span>
                                        <span>Price: ${{ number_format($item->price, 2) }}</span>
                                    </div>
                                </div>
                                <div class="item-total">
                                    ${{ number_format($item->quantity * $item->price, 2) }}
                                </div>
                            </div>
                            @endforeach
                        </div>
                    </div>
                </div>

                <!-- Delivery Information -->
                <div class="col-lg-4">
                    <div class="details-card" data-aos="fade-left">
                        <h3 class="card-title">
                            <i class="fas fa-map-marked-alt"></i>
                            Delivery Information
                        </h3>
                        @if($order->shippingAddress)
                        <div class="address-info">
                            <p><strong>{{ $order->shippingAddress->name }}</strong></p>
                            <p>{{ $order->shippingAddress->address_line_1 }}</p>
                            @if($order->shippingAddress->address_line_2)
                            <p>{{ $order->shippingAddress->address_line_2 }}</p>
                            @endif
                            <p>{{ $order->shippingAddress->city }}, {{ $order->shippingAddress->state }}</p>
                            <p>{{ $order->shippingAddress->postal_code }}</p>
                            <p>{{ $order->shippingAddress->country }}</p>
                            <p class="mt-3">
                                <i class="fas fa-phone"></i> {{ $order->shippingAddress->phone }}
                            </p>
                        </div>
                        @endif

                        <div class="order-summary">
                            <h4>Order Summary</h4>
                            <div class="summary-row">
                                <span>Subtotal:</span>
                                <span>${{ number_format($order->subtotal, 2) }}</span>
                            </div>
                            @if($order->shipping_amount > 0)
                            <div class="summary-row">
                                <span>Shipping:</span>
                                <span>${{ number_format($order->shipping_amount, 2) }}</span>
                            </div>
                            @endif
                            @if($order->tax_amount > 0)
                            <div class="summary-row">
                                <span>Tax:</span>
                                <span>${{ number_format($order->tax_amount, 2) }}</span>
                            </div>
                            @endif
                            @if($order->discount_amount > 0)
                            <div class="summary-row discount">
                                <span>Discount:</span>
                                <span>-${{ number_format($order->discount_amount, 2) }}</span>
                            </div>
                            @endif
                            <div class="summary-row total">
                                <span>Total:</span>
                                <span>${{ number_format($order->total_amount, 2) }}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>

    <!-- Actions -->
    <section class="tracking-actions">
        <div class="container">
            <div class="actions-wrapper" data-aos="fade-up">
                <a href="{{ route('order.track.index') }}" class="btn-action secondary">
                    <i class="fas fa-search"></i>
                    Track Another Order
                </a>
                @auth
                <a href="{{ route('account.order.detail', $order->order_number) }}" class="btn-action primary">
                    <i class="fas fa-user"></i>
                    View in My Account
                </a>
                @endauth
                <a href="{{ route('contact') }}" class="btn-action outline">
                    <i class="fas fa-headset"></i>
                    Contact Support
                </a>
            </div>
        </div>
    </section>
</div>

@push('styles')
<style>
/* Tracking Result Page Premium Styles */
.tracking-result-premium {
    min-height: 100vh;
    background: #fafafa;
}

/* Result Header */
.result-header-premium {
    background: linear-gradient(135deg, #000 0%, #333 100%);
    padding: 4rem 0;
    text-align: center;
}

.tracking-found {
    color: white;
}

.success-icon {
    font-size: 4rem;
    margin-bottom: 2rem;
    color: #28a745;
    text-shadow: 0 4px 8px rgba(40, 167, 69, 0.3);
}

.tracking-found h1 {
    font-size: 2.5rem;
    font-weight: 900;
    letter-spacing: 0.05em;
    margin-bottom: 1rem;
    font-family: 'Space Grotesk', sans-serif;
    color: white;
}

.tracking-found p {
    font-size: 1.125rem;
    opacity: 0.9;
}

/* Timeline Section */
.tracking-timeline-section {
    padding: 5rem 0;
}

.timeline-card {
    background: white;
    padding: 3rem;
    box-shadow: 0 10px 40px rgba(0,0,0,0.05);
    border-radius: 0;
}

.timeline-header {
    margin-bottom: 3rem;
}

.timeline-header h3 {
    font-size: 1.5rem;
    font-weight: 700;
    margin-bottom: 1rem;
    font-family: 'Space Grotesk', sans-serif;
}

.status-badge {
    padding: 0.5rem 1rem;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-badge.pending { background: #ffc107; color: #000; }
.status-badge.processing { background: #17a2b8; color: #fff; }
.status-badge.shipped { background: #007bff; color: #fff; }
.status-badge.out_for_delivery { background: #6f42c1; color: #fff; }
.status-badge.delivered { background: #28a745; color: #fff; }
.status-badge.cancelled { background: #dc3545; color: #fff; }

.delivery-date {
    color: #666;
    margin-top: 0.5rem;
}

.delivery-date i {
    margin-right: 0.5rem;
}

/* Progress Bar */
.progress-bar-container {
    width: 100%;
    height: 4px;
    background: #e0e0e0;
    margin-bottom: 2rem;
    position: relative;
}

.progress-bar-fill {
    height: 100%;
    background: #000;
    transition: width 0.8s ease;
}

/* Timeline Steps */
.timeline-steps {
    display: grid;
    grid-template-columns: repeat(5, 1fr);
    gap: 1rem;
}

.step {
    text-align: center;
    position: relative;
}

.step-icon {
    width: 60px;
    height: 60px;
    margin: 0 auto 1rem;
    background: #f8f8f8;
    border: 3px solid #e0e0e0;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    color: #999;
    transition: all 0.3s;
}

.step.completed .step-icon {
    background: #000;
    border-color: #000;
    color: white;
}

.step-title {
    font-weight: 700;
    margin-bottom: 0.5rem;
    color: #666;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.step.completed .step-title {
    color: #000;
}

.step-date {
    font-size: 0.75rem;
    color: #999;
}

.step.completed .step-date {
    color: #666;
}

/* Tracking Number Display */
.tracking-number-display {
    margin-top: 3rem;
    padding: 2rem;
    background: #f8f8f8;
    display: flex;
    align-items: center;
    gap: 1rem;
    font-family: 'Space Grotesk', sans-serif;
}

.tracking-number-display i {
    font-size: 1.5rem;
    color: #000;
}

.tracking-number-display strong {
    font-weight: 700;
    letter-spacing: 0.05em;
}

.carrier-badge {
    background: #000;
    color: white;
    padding: 0.25rem 0.75rem;
    font-size: 0.75rem;
    font-weight: 700;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

/* Tracking Updates Section */
.tracking-updates-section {
    padding: 3rem 0;
    background: white;
}

.updates-card {
    background: white;
    padding: 3rem;
    box-shadow: 0 10px 40px rgba(0,0,0,0.05);
}

.updates-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin-bottom: 2rem;
    font-family: 'Space Grotesk', sans-serif;
}

.updates-title i {
    margin-right: 0.5rem;
    color: #000;
}

.updates-timeline {
    position: relative;
}

.updates-timeline::before {
    content: '';
    position: absolute;
    left: 10px;
    top: 0;
    bottom: 0;
    width: 2px;
    background: #e0e0e0;
}

.update-item {
    position: relative;
    margin-bottom: 2rem;
    padding-left: 3rem;
}

.update-marker {
    position: absolute;
    left: 0;
    top: 0.5rem;
    width: 20px;
    height: 20px;
    background: #000;
    border-radius: 50%;
}

.update-time {
    font-size: 0.875rem;
    color: #999;
    margin-bottom: 0.25rem;
    font-weight: 600;
}

.update-status {
    font-weight: 700;
    margin-bottom: 0.25rem;
    color: #000;
}

.update-location {
    font-size: 0.875rem;
    color: #666;
}

.update-location i {
    margin-right: 0.5rem;
}

/* Order Details Section */
.order-details-section {
    padding: 3rem 0;
}

.details-card {
    background: white;
    padding: 3rem;
    box-shadow: 0 10px 40px rgba(0,0,0,0.05);
    margin-bottom: 2rem;
    height: fit-content;
}

.card-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin-bottom: 2rem;
    font-family: 'Space Grotesk', sans-serif;
    border-bottom: 2px solid #e0e0e0;
    padding-bottom: 1rem;
}

.card-title i {
    margin-right: 0.5rem;
    color: #000;
}

/* Items List */
.items-list {
    border-top: 1px solid #e0e0e0;
}

.item-row {
    display: flex;
    align-items: center;
    padding: 2rem 0;
    border-bottom: 1px solid #e0e0e0;
    gap: 1.5rem;
}

.item-image {
    width: 80px;
    height: 80px;
    flex-shrink: 0;
}

.item-image img {
    width: 100%;
    height: 100%;
    object-fit: cover;
}

.placeholder-image {
    width: 100%;
    height: 100%;
    background: #f8f8f8;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #999;
    font-size: 1.5rem;
}

.item-details {
    flex-grow: 1;
}

.item-details h4 {
    font-weight: 700;
    margin-bottom: 0.5rem;
    color: #000;
}

.item-meta {
    display: flex;
    gap: 2rem;
    color: #666;
    font-size: 0.875rem;
}

.item-total {
    font-weight: 700;
    font-size: 1.125rem;
    color: #000;
}

/* Address Info */
.address-info {
    margin-bottom: 2rem;
}

.address-info p {
    margin-bottom: 0.5rem;
    color: #666;
    line-height: 1.5;
}

.address-info p:first-child {
    color: #000;
    font-weight: 700;
}

/* Order Summary */
.order-summary {
    border-top: 2px solid #e0e0e0;
    padding-top: 2rem;
}

.order-summary h4 {
    font-weight: 700;
    margin-bottom: 1.5rem;
    font-family: 'Space Grotesk', sans-serif;
}

.summary-row {
    display: flex;
    justify-content: between;
    align-items: center;
    margin-bottom: 1rem;
    padding: 0.5rem 0;
}

.summary-row span:first-child {
    color: #666;
}

.summary-row span:last-child {
    font-weight: 600;
    color: #000;
}

.summary-row.discount span:last-child {
    color: #dc3545;
}

.summary-row.total {
    border-top: 2px solid #000;
    padding-top: 1rem;
    margin-top: 1rem;
    font-size: 1.125rem;
    font-weight: 700;
}

/* Tracking Actions */
.tracking-actions {
    padding: 3rem 0;
    background: white;
}

.actions-wrapper {
    display: flex;
    gap: 1rem;
    justify-content: center;
    flex-wrap: wrap;
}

.btn-action {
    padding: 1rem 2rem;
    border: 2px solid transparent;
    font-weight: 700;
    letter-spacing: 0.05em;
    text-decoration: none;
    text-transform: uppercase;
    transition: all 0.3s;
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
}

.btn-action.primary {
    background: #000;
    color: white;
    border-color: #000;
}

.btn-action.primary:hover {
    background: #333;
    border-color: #333;
    transform: translateY(-2px);
    color: white;
    text-decoration: none;
}

.btn-action.secondary {
    background: #666;
    color: white;
    border-color: #666;
}

.btn-action.secondary:hover {
    background: #000;
    border-color: #000;
    transform: translateY(-2px);
    color: white;
    text-decoration: none;
}

.btn-action.outline {
    background: transparent;
    color: #000;
    border-color: #000;
}

.btn-action.outline:hover {
    background: #000;
    color: white;
    transform: translateY(-2px);
    text-decoration: none;
}

/* Alert Styles */
.alert-premium {
    padding: 1rem;
    margin-bottom: 2rem;
    display: flex;
    align-items: center;
    gap: 1rem;
    background: #f8f9fa;
    border-left: 4px solid #000;
}

.alert-premium.error {
    background: #f8d7da;
    border-left-color: #dc3545;
    color: #721c24;
}

.alert-close {
    background: none;
    border: none;
    color: inherit;
    cursor: pointer;
    margin-left: auto;
}

/* Responsive Design */
@media (max-width: 768px) {
    .timeline-steps {
        grid-template-columns: repeat(2, 1fr);
        gap: 2rem;
    }
    
    .step-icon {
        width: 50px;
        height: 50px;
        font-size: 1.2rem;
    }
    
    .tracking-found h1 {
        font-size: 1.75rem;
    }
    
    .timeline-card,
    .details-card,
    .updates-card {
        padding: 2rem;
    }
    
    .item-row {
        flex-direction: column;
        text-align: center;
        gap: 1rem;
    }
    
    .item-meta {
        justify-content: center;
    }
    
    .actions-wrapper {
        flex-direction: column;
        align-items: center;
    }
    
    .btn-action {
        width: 100%;
        justify-content: center;
        max-width: 300px;
    }
    
    .tracking-number-display {
        flex-direction: column;
        text-align: center;
        gap: 0.5rem;
    }
}

@media (max-width: 480px) {
    .timeline-steps {
        grid-template-columns: 1fr;
    }
    
    .summary-row {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.25rem;
    }
    
    .summary-row span:last-child {
        align-self: flex-end;
    }
}
</style>
@endpush
@endsection