<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use App\Http\Controllers\HomeController;
use App\Http\Controllers\ShopController;
use App\Http\Controllers\CartController;
use App\Http\Controllers\CheckoutController;
use App\Http\Controllers\AccountController;
use App\Http\Controllers\NewsletterController;
use App\Http\Controllers\TrackingController;
use App\Http\Controllers\InvoiceController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\ProductController;
use App\Http\Controllers\Admin\CategoryController;
use App\Http\Controllers\Admin\OrderController;
use App\Http\Controllers\Admin\CouponController;
use App\Http\Controllers\Admin\CustomerController;
use App\Http\Controllers\Admin\InquiryController;
use App\Http\Controllers\Admin\TestimonialController;
use App\Http\Controllers\Admin\CompanyInfoController;
use App\Http\Controllers\Admin\ReportController;
use App\Http\Controllers\Admin\SettingsController;
use App\Http\Controllers\Admin\DiscountController;
use App\Http\Controllers\Admin\BannerController;
use App\Http\Controllers\Admin\InstagramController;
use App\Http\Controllers\Admin\ShippingZoneController;
use App\Http\Controllers\Admin\TaxRateController;
use App\Http\Controllers\Admin\LegalPageController as AdminLegalPageController;
use App\Http\Controllers\Admin\CommunityController as AdminCommunityController;
use App\Http\Controllers\Admin\CommunityPageController as AdminCommunityPageController;
use App\Http\Controllers\LegalPageController;
use App\Http\Controllers\BlogController;
use App\Http\Controllers\SitemapController;
use App\Http\Controllers\LookbookController;
use App\Http\Controllers\StyleGuideController;
use App\Http\Controllers\CommunityController;
use App\Http\Controllers\TrendsController;
use App\Http\Controllers\SizeGuideController;
use App\Http\Controllers\CityPagesController;

/*
|--------------------------------------------------------------------------
| Image Serving Routes (MUST BE FIRST)
|--------------------------------------------------------------------------
*/

// Direct image serving route - works without symlinks
Route::get('/serve-image/{path}', function ($path) {
    return \App\Helpers\ImageHelper::serveImage($path);
})->where('path', '.*')->name('serve.image');

// Alternative image serving route
Route::get('/storage-image/{path}', function ($path) {
    $path = base64_decode($path);
    $fullPath = storage_path('app/public/' . $path);
    
    if (!file_exists($fullPath)) {
        $fullPath = public_path('images/placeholder-product.jpg');
    }
    
    return response()->file($fullPath);
})->where('path', '.*')->name('storage.image');

// Fallback for storage URLs
Route::get('/storage/{path}', function ($path) {
    // First check if file exists in public/storage (symlink)
    if (file_exists(public_path('storage/' . $path))) {
        return response()->file(public_path('storage/' . $path));
    }
    
    // Otherwise serve from actual storage
    $fullPath = storage_path('app/public/' . $path);
    
    if (file_exists($fullPath)) {
        return response()->file($fullPath, [
            'Content-Type' => mime_content_type($fullPath),
            'Cache-Control' => 'public, max-age=31536000',
        ]);
    }
    
    abort(404);
})->where('path', '.*');

/*
|--------------------------------------------------------------------------
| Web Routes - Edition By Punjab Shopping Website
|--------------------------------------------------------------------------
*/

// Frontend Routes (Public)
Route::get('/', [HomeController::class, 'index'])->name('home');
Route::get('/about', [HomeController::class, 'about'])->name('about');
Route::get('/contact', [HomeController::class, 'contact'])->name('contact');
Route::post('/contact', [HomeController::class, 'submitInquiry'])->name('contact.submit');

// Shop Routes
Route::prefix('shop')->name('shop.')->group(function () {
    Route::get('/', [ShopController::class, 'index'])->name('index');
    Route::get('/product/{slug}', [ShopController::class, 'show'])->name('product');
    Route::get('/category/{slug}', [ShopController::class, 'category'])->name('category');
    Route::get('/quick-view/{id}', [ShopController::class, 'quickView'])->name('quick-view');
    Route::post('/wishlist/toggle', [ShopController::class, 'toggleWishlist'])->name('wishlist.toggle');
    Route::get('/search', [ShopController::class, 'search'])->name('search');

    // Discounted products / Sale page
    // route name: shop.discounted
    Route::get('/sale', [ShopController::class, 'discounted'])->name('discounted');
});

// Cart Routes
Route::prefix('cart')->name('cart.')->group(function () {
    Route::get('/', [CartController::class, 'index'])->name('index');
    Route::post('/add', [CartController::class, 'add'])->name('add');
    Route::patch('/update/{id}', [CartController::class, 'update'])->name('update');
    Route::delete('/remove/{id}', [CartController::class, 'remove'])->name('remove');
    Route::post('/apply-coupon', [CartController::class, 'applyCoupon'])->name('apply-coupon');
    Route::delete('/remove-coupon', [CartController::class, 'removeCoupon'])->name('remove-coupon');
    Route::get('/count', [CartController::class, 'getCartCount'])->name('count');

    // Dynamic shipping update (AJAX)
    Route::post('/update-shipping', [CartController::class, 'updateShipping'])->name('update-shipping');
});

// Checkout Routes
Route::prefix('checkout')->name('checkout.')->group(function () {
    Route::get('/', [CheckoutController::class, 'index'])->name('index');
    Route::post('/process', [CheckoutController::class, 'process'])->name('process');
    Route::get('/payment/{orderNumber}', [CheckoutController::class, 'payment'])->name('payment');

    // confirm-payment route already exists here (keeps consistency with your checkout naming)
    Route::post('/confirm-payment/{orderNumber}', [CheckoutController::class, 'confirmPayment'])->name('confirm-payment');

    Route::get('/success/{orderNumber}', [CheckoutController::class, 'success'])->name('success');
    Route::get('/cancel', [CheckoutController::class, 'cancel'])->name('cancel');
});

// Stripe Payment Routes (webhook)
Route::post('/stripe/webhook', [CheckoutController::class, 'handleWebhook'])
    ->name('stripe.webhook');

// Invoice Routes (protected)
Route::get('/invoice/{orderNumber}/download', [InvoiceController::class, 'download'])
    ->name('order.invoice.download')
    ->middleware('auth');

Route::get('/invoice/{orderNumber}', [InvoiceController::class, 'view'])
    ->name('order.invoice')
    ->middleware('auth');

// Account Routes (Protected)
Route::middleware(['auth', 'verified'])->prefix('account')->name('account.')->group(function () {
    Route::get('/dashboard', [AccountController::class, 'dashboard'])->name('dashboard');
    Route::get('/orders', [AccountController::class, 'orders'])->name('orders');
    Route::get('/orders/{orderNumber}', [AccountController::class, 'orderDetail'])->name('order.detail');
    Route::get('/wishlist', [AccountController::class, 'wishlist'])->name('wishlist');
    Route::delete('/wishlist/{id}', [AccountController::class, 'removeFromWishlist'])->name('wishlist.remove');
    Route::get('/addresses', [AccountController::class, 'addresses'])->name('addresses');
    Route::post('/addresses', [AccountController::class, 'addAddress'])->name('address.add');
    Route::put('/addresses/{id}', [AccountController::class, 'updateAddress'])->name('address.update');
    Route::delete('/addresses/{id}', [AccountController::class, 'deleteAddress'])->name('address.delete');
    Route::get('/profile', [AccountController::class, 'profile'])->name('profile');
    Route::put('/profile', [AccountController::class, 'updateProfile'])->name('profile.update');

    // Account delete route (added)
    Route::delete('/delete', [AccountController::class, 'deleteAccount'])->name('delete');
});

// Newsletter
Route::post('/newsletter/subscribe', [NewsletterController::class, 'subscribe'])->name('newsletter.subscribe');
// Check subscription (AJAX friendly)
Route::get('/newsletter/check', [NewsletterController::class, 'checkSubscription'])->name('newsletter.check');

// Blog Routes
Route::prefix('blog')->name('blog.')->group(function () {
    Route::get('/', [BlogController::class, 'index'])->name('index');
    Route::get('/{blog}', [BlogController::class, 'show'])->name('show');
});

// Alternative blog routes (SEO friendly)
Route::get('/blogs', [BlogController::class, 'index'])->name('blogs.index');
Route::get('/blog/{blog}', [BlogController::class, 'show'])->name('blogs.show');

// Lookbook Routes
Route::prefix('lookbook')->name('lookbook.')->group(function () {
    Route::get('/', [LookbookController::class, 'index'])->name('index');
    Route::get('/{lookbook}', [LookbookController::class, 'show'])->name('show');
});

// Style Guide Routes
Route::prefix('style-guide')->name('style-guide.')->group(function () {
    Route::get('/', [StyleGuideController::class, 'index'])->name('index');
    Route::get('/{city}', [StyleGuideController::class, 'show'])->name('show');
});

// Community Routes
Route::prefix('community')->name('community.')->group(function () {
    Route::get('/', [CommunityController::class, 'index'])->name('index');
    Route::get('/submit', [CommunityController::class, 'submit'])->name('submit');
    Route::post('/submit', [CommunityController::class, 'store'])->name('submit.store');
    Route::get('/gallery', [CommunityController::class, 'gallery'])->name('gallery');
    Route::get('/spotlight', [CommunityController::class, 'spotlight'])->name('spotlight');
});

// Trends Routes
Route::prefix('trends')->name('trends.')->group(function () {
    Route::get('/', [TrendsController::class, 'index'])->name('index');
    Route::get('/weekly', [TrendsController::class, 'weekly'])->name('weekly');
    Route::get('/seasonal', [TrendsController::class, 'seasonal'])->name('seasonal');
    Route::get('/news', [TrendsController::class, 'news'])->name('news');
    Route::get('/news/{slug}', [TrendsController::class, 'newsStory'])->name('news.story');
    Route::get('/predictions', [TrendsController::class, 'predictions'])->name('predictions');
});

// Size Guide Routes
Route::prefix('size-guide')->name('size-guide.')->group(function () {
    Route::get('/', [SizeGuideController::class, 'index'])->name('index');
    Route::get('/finder', [SizeGuideController::class, 'finder'])->name('finder');
    Route::post('/find-size', [SizeGuideController::class, 'findSize'])->name('find-size');
    Route::get('/{category}', [SizeGuideController::class, 'show'])->name('show');
});

// City Pages Routes
Route::prefix('city')->name('city.')->group(function () {
    Route::get('/melbourne-streetwear', [CityPagesController::class, 'melbourne'])->name('melbourne');
    Route::get('/sydney-fashion', [CityPagesController::class, 'sydney'])->name('sydney');
});

// Long-tail Keyword Pages
Route::get('/where-to-buy-streetwear-australia', [CityPagesController::class, 'whereToBuyStreetWear'])->name('where-to-buy-streetwear');
Route::get('/punjabi-fusion-streetwear-online', [CityPagesController::class, 'punjabieFusionStreetWear'])->name('punjabi-fusion-streetwear');

// FAQ Page
Route::get('/faq', [CityPagesController::class, 'faq'])->name('faq');

// Legal Pages (Dynamic from Database)
Route::get('/legal', [LegalPageController::class, 'index'])->name('legal.index');
Route::get('/legal/{slug}', [LegalPageController::class, 'show'])->name('legal.show');

// Admin Routes - Protected by auth and admin middleware
Route::middleware(['auth', 'admin'])->prefix('admin')->name('admin.')->group(function () {
    // Dashboard
    Route::get('/dashboard', [DashboardController::class, 'index'])->name('dashboard');

    // Products Management
    Route::resource('products', ProductController::class);
    Route::post('/products/{product}/toggle-featured', [ProductController::class, 'toggleFeatured'])->name('products.toggle-featured');
    Route::post('/products/{product}/duplicate', [ProductController::class, 'duplicate'])->name('products.duplicate');

    // Categories Management
    Route::resource('categories', CategoryController::class);
    Route::post('/categories/{category}/toggle-active', [CategoryController::class, 'toggleActive'])->name('categories.toggle-active');

    // Orders Management
    Route::get('/orders', [OrderController::class, 'index'])->name('orders.index');
    Route::get('/orders/{order}', [OrderController::class, 'show'])->name('orders.show');
    Route::patch('/orders/{order}/status', [OrderController::class, 'updateStatus'])->name('orders.update-status');
    Route::post('/orders/{order}/ship', [OrderController::class, 'markAsShipped'])->name('orders.ship');
    Route::post('/orders/{order}/cancel', [OrderController::class, 'cancel'])->name('orders.cancel');
    Route::get('/orders/{order}/invoice', [OrderController::class, 'invoice'])->name('orders.invoice');

    // Admin: update tracking for an order
    Route::post('/orders/{order}/tracking', [OrderController::class, 'updateTracking'])->name('orders.tracking');

    // Customers Management
    Route::get('/customers', [CustomerController::class, 'index'])->name('customers.index');
    Route::get('/customers/{user}', [CustomerController::class, 'show'])->name('customers.show');
    Route::post('/customers/{user}/block', [CustomerController::class, 'block'])->name('customers.block');
    Route::post('/customers/{user}/unblock', [CustomerController::class, 'unblock'])->name('customers.unblock');

    // Coupons Management
    Route::resource('coupons', CouponController::class);
    Route::post('/coupons/{coupon}/toggle-active', [CouponController::class, 'toggleActive'])->name('coupons.toggle-active');

    // Discounts Management (Admin)
    // Resourceful CRUD for discounts and a toggle route for active/inactive
    Route::resource('discounts', DiscountController::class);
    Route::post('/discounts/{discount}/toggle', [DiscountController::class, 'toggleActive'])->name('discounts.toggle');

    // Banner Management
    Route::resource('banners', BannerController::class)->except(['show']);
    // Toggle route for banners (added from updated file)
    Route::post('/banners/{banner}/toggle-active', [BannerController::class, 'toggleActive'])->name('banners.toggleActive');

    // Instagram Feed Management
    // Use full resource except show (keeps edit/update available) and add toggle route
    Route::resource('instagram', InstagramController::class)->except(['show']);
    Route::post('/instagram/{instagram}/toggle-active', [InstagramController::class, 'toggleActive'])->name('instagram.toggleActive');

    // Inquiries Management
    Route::get('/inquiries', [InquiryController::class, 'index'])->name('inquiries.index');
    Route::get('/inquiries/{inquiry}', [InquiryController::class, 'show'])->name('inquiries.show');
    Route::patch('/inquiries/{inquiry}/status', [InquiryController::class, 'updateStatus'])->name('inquiries.status');
    Route::delete('/inquiries/{inquiry}', [InquiryController::class, 'destroy'])->name('inquiries.destroy');

    // Testimonials Management
    Route::resource('testimonials', TestimonialController::class);
    Route::post('/testimonials/{testimonial}/toggle-active', [TestimonialController::class, 'toggleActive'])->name('testimonials.toggle-active');

    // Company Info & Settings
    Route::get('/company-info', [CompanyInfoController::class, 'edit'])->name('company.edit');
    Route::put('/company-info', [CompanyInfoController::class, 'update'])->name('company.update');
    Route::get('/settings', [SettingsController::class, 'index'])->name('settings.index');
    Route::put('/settings', [SettingsController::class, 'update'])->name('settings.update');

    // <<< New cache management routes (added) >>>
    Route::post('/settings/clear-cache', [SettingsController::class, 'clearCache'])->name('settings.clear-cache');
    Route::post('/settings/optimize', [SettingsController::class, 'optimize'])->name('settings.optimize');

    // Reports
    Route::get('/reports', [ReportController::class, 'index'])->name('reports.index');
    Route::get('/reports/sales', [ReportController::class, 'sales'])->name('reports.sales');
    Route::get('/reports/products', [ReportController::class, 'products'])->name('reports.products');
    Route::get('/reports/customers', [ReportController::class, 'customers'])->name('reports.customers');
    Route::get('/reports/export/{type}', [ReportController::class, 'export'])->name('reports.export');

    /*
    |----------------------------------------------------------------------
    | Shipping Zones Management (Admin)
    |----------------------------------------------------------------------
    |
    | Resourceful controller for shipping zones and a couple of extra routes
    | for toggling active state and bulk updates.
    |
    */
    Route::resource('shipping-zones', ShippingZoneController::class);
    Route::post('/shipping-zones/{shippingZone}/toggle-active', [ShippingZoneController::class, 'toggleActive'])
        ->name('shipping-zones.toggle-active');
    Route::post('/shipping-zones/bulk-update', [ShippingZoneController::class, 'bulkUpdate'])
        ->name('shipping-zones.bulk-update');

    /*
    |----------------------------------------------------------------------
    | Tax Rates Management (Admin)
    |----------------------------------------------------------------------
    |
    | Resourceful controller for tax rates and an extra toggle route.
    |
    */
    Route::resource('tax-rates', TaxRateController::class);
    Route::post('/tax-rates/{taxRate}/toggle-active', [TaxRateController::class, 'toggleActive'])
        ->name('tax-rates.toggle-active');

    /*
    |----------------------------------------------------------------------
    | Legal Pages Management (Admin)
    |----------------------------------------------------------------------
    |
    | Full CRUD operations for legal pages (Terms, Privacy Policy, etc.)
    | with additional routes for status toggling.
    |
    */
    Route::resource('legal-pages', AdminLegalPageController::class);
    Route::patch('/legal-pages/{legalPage}/toggle-status', [AdminLegalPageController::class, 'toggleStatus'])
        ->name('legal-pages.toggle-status');
    
    // Blog Management
    Route::resource('blogs', \App\Http\Controllers\Admin\BlogController::class);
    Route::post('blogs/{blog}/toggle-featured', [\App\Http\Controllers\Admin\BlogController::class, 'toggleFeatured'])->name('blogs.toggle-featured');
    Route::post('blogs/{blog}/toggle-published', [\App\Http\Controllers\Admin\BlogController::class, 'togglePublished'])->name('blogs.toggle-published');

    // Community Management
    Route::resource('community', AdminCommunityController::class);
    Route::post('community/{community}/toggle-featured', [AdminCommunityController::class, 'toggleFeatured'])->name('community.toggle-featured');
    Route::post('community/{community}/approve', [AdminCommunityController::class, 'approve'])->name('community.approve');
    Route::post('community/{community}/reject', [AdminCommunityController::class, 'reject'])->name('community.reject');
    Route::post('community/bulk-action', [AdminCommunityController::class, 'bulkAction'])->name('community.bulk-action');

    // Community Pages Management
    Route::resource('community-pages', AdminCommunityPageController::class);
    Route::post('community-pages/{communityPage}/toggle-status', [AdminCommunityPageController::class, 'toggleStatus'])->name('community-pages.toggle-status');

    // Style Guides Management
    Route::resource('style-guides', \App\Http\Controllers\Admin\StyleGuideController::class);
    Route::post('style-guides/{styleGuide}/toggle-active', [\App\Http\Controllers\Admin\StyleGuideController::class, 'toggleActive'])->name('style-guides.toggle-active');
    Route::post('style-guides/{styleGuide}/duplicate', [\App\Http\Controllers\Admin\StyleGuideController::class, 'duplicate'])->name('style-guides.duplicate');

    // Trends Management
    Route::prefix('trends')->name('trends.')->group(function () {
        Route::get('/', [\App\Http\Controllers\Admin\TrendsController::class, 'index'])->name('index');
        
        // Weekly Trends
        Route::get('/weekly', [\App\Http\Controllers\Admin\TrendsController::class, 'weekly'])->name('weekly');
        Route::get('/weekly/create', [\App\Http\Controllers\Admin\TrendsController::class, 'createWeekly'])->name('create-weekly');
        Route::post('/weekly', [\App\Http\Controllers\Admin\TrendsController::class, 'storeWeekly'])->name('store-weekly');
        Route::get('/weekly/{id}/edit', [\App\Http\Controllers\Admin\TrendsController::class, 'editWeekly'])->name('edit-weekly');
        Route::put('/weekly/{id}', [\App\Http\Controllers\Admin\TrendsController::class, 'updateWeekly'])->name('update-weekly');
        Route::delete('/weekly/{id}', [\App\Http\Controllers\Admin\TrendsController::class, 'destroyWeekly'])->name('destroy-weekly');
        
        // Seasonal Forecasts
        Route::get('/seasonal', [\App\Http\Controllers\Admin\TrendsController::class, 'seasonal'])->name('seasonal');
        Route::get('/seasonal/create', [\App\Http\Controllers\Admin\TrendsController::class, 'createSeasonal'])->name('create-seasonal');
        Route::post('/seasonal', [\App\Http\Controllers\Admin\TrendsController::class, 'storeSeasonal'])->name('store-seasonal');
        Route::get('/seasonal/{slug}/edit', [\App\Http\Controllers\Admin\TrendsController::class, 'editSeasonal'])->name('edit-seasonal');
        Route::put('/seasonal/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'updateSeasonal'])->name('update-seasonal');
        Route::delete('/seasonal/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'destroySeasonal'])->name('destroy-seasonal');
        
        // Fashion News
        Route::get('/news', [\App\Http\Controllers\Admin\TrendsController::class, 'news'])->name('news');
        Route::get('/news/create', [\App\Http\Controllers\Admin\TrendsController::class, 'createNews'])->name('create-news');
        Route::post('/news', [\App\Http\Controllers\Admin\TrendsController::class, 'storeNews'])->name('store-news');
        Route::get('/news/{slug}/edit', [\App\Http\Controllers\Admin\TrendsController::class, 'editNews'])->name('edit-news');
        Route::put('/news/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'updateNews'])->name('update-news');
        Route::delete('/news/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'destroyNews'])->name('destroy-news');
        
        // Style Predictions
        Route::get('/predictions', [\App\Http\Controllers\Admin\TrendsController::class, 'predictions'])->name('predictions');
        Route::get('/predictions/create', [\App\Http\Controllers\Admin\TrendsController::class, 'createPredictions'])->name('create-predictions');
        Route::post('/predictions', [\App\Http\Controllers\Admin\TrendsController::class, 'storePredictions'])->name('store-predictions');
        Route::get('/predictions/{slug}/edit', [\App\Http\Controllers\Admin\TrendsController::class, 'editPredictions'])->name('edit-predictions');
        Route::put('/predictions/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'updatePredictions'])->name('update-predictions');
        Route::delete('/predictions/{slug}', [\App\Http\Controllers\Admin\TrendsController::class, 'destroyPredictions'])->name('destroy-predictions');
    });

});

// Authentication Routes
require __DIR__.'/auth.php';

// Merge guest cart when user logs in
Route::middleware(['auth'])->group(function () {
    Route::post('/merge-cart', [CartController::class, 'mergeGuestCart'])->name('cart.merge');
});

// Public Order Tracking
Route::get('/track-order', [TrackingController::class, 'index'])->name('order.track.index');
Route::post('/track-order', [TrackingController::class, 'track'])->name('order.track');

// Legal Pages (Dynamic from Database with backward compatibility)
Route::get('/terms-of-service', [LegalPageController::class, 'show'])->defaults('slug', 'terms-of-service')->name('terms');
Route::get('/privacy-policy', [LegalPageController::class, 'show'])->defaults('slug', 'privacy-policy')->name('privacy');
Route::get('/shipping-policy', [LegalPageController::class, 'show'])->defaults('slug', 'shipping-policy')->name('shipping');
Route::get('/refund-policy', [LegalPageController::class, 'show'])->defaults('slug', 'refund-policy')->name('refund');

// Image serving route for shared/limited hosting where storage:link may not work
Route::get('/images/storage/{path}', function ($path) {
    $path = str_replace('..', '', $path); // Security: prevent directory traversal

    if (!Storage::disk('public')->exists($path)) {
        abort(404);
    }

    $file = Storage::disk('public')->get($path);
    $type = Storage::disk('public')->mimeType($path);

    return response($file, 200)->header('Content-Type', $type);
})->where('path', '.*')->name('image.show');

// Comprehensive SEO Sitemap System
Route::get('/sitemap.xml', [SitemapController::class, 'index'])->name('sitemap.index');
Route::get('/sitemap-static.xml', [SitemapController::class, 'static'])->name('sitemap.static');
Route::get('/sitemap-products.xml', [SitemapController::class, 'products'])->name('sitemap.products');
Route::get('/sitemap-categories.xml', [SitemapController::class, 'categories'])->name('sitemap.categories');
Route::get('/sitemap-blogs.xml', [SitemapController::class, 'blogs'])->name('sitemap.blogs');

// Redirect old routes to new shopping routes (for SEO)
// Provide a *named* compatibility route so any code using route('products') will not break
Route::get('/products', function() {
    // This is a redirect to the canonical shop listing (301 SEO-friendly)
    return redirect()->route('shop.index', [], 301);
})->name('products');

// Legacy single product URL redirect (keeps old links working)
Route::get('/product/{slug}', function($slug) {
    return redirect()->route('shop.product', ['slug' => $slug], 301);
});