<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;
use App\Models\CommunityPost;
use App\Models\Product;

class CommunityTest extends TestCase
{
    use RefreshDatabase;

    /** @test */
    public function community_index_page_loads_successfully()
    {
        $response = $this->get(route('community.index'));
        
        $response->assertStatus(200);
        $response->assertViewIs('community.index');
        $response->assertSee('COMMUNITY');
    }

    /** @test */
    public function community_gallery_page_loads_successfully()
    {
        $response = $this->get(route('community.gallery'));
        
        $response->assertStatus(200);
        $response->assertViewIs('community.gallery');
        $response->assertSee('GALLERY');
    }

    /** @test */
    public function community_spotlight_page_loads_successfully()
    {
        $response = $this->get(route('community.spotlight'));
        
        $response->assertStatus(200);
        $response->assertViewIs('community.spotlight');
        $response->assertSee('SPOTLIGHT');
    }

    /** @test */
    public function community_submit_page_loads_successfully()
    {
        $response = $this->get(route('community.submit'));
        
        $response->assertStatus(200);
        $response->assertViewIs('community.submit');
        $response->assertSee('SHARE YOUR STYLE');
    }

    /** @test */
    public function user_can_submit_style_post()
    {
        Storage::fake('public');

        $file = UploadedFile::fake()->image('test-style.jpg');

        $response = $this->post(route('community.submit.store'), [
            'photo' => $file,
            'name' => 'Test User',
            'location' => 'Melbourne, VIC',
            'caption' => 'This is a test style post for the community gallery. I love wearing my Edition By Punjab pieces!',
            'products' => 'Test Hoodie, Test T-Shirt',
            'instagram' => 'testuser',
            'consent_gallery' => true,
            'consent_social' => false,
            'consent_marketing' => true
        ]);

        $response->assertStatus(302);
        $response->assertRedirect(route('community.index'));
        $response->assertSessionHas('success');

        $this->assertDatabaseHas('community_posts', [
            'name' => 'Test User',
            'location' => 'Melbourne, VIC',
            'products_featured' => 'Test Hoodie, Test T-Shirt',
            'instagram_handle' => 'testuser',
            'status' => 'pending'
        ]);

        // Check if file was stored
        Storage::disk('public')->assertExists('community/' . $file->hashName());
    }

    /** @test */
    public function community_post_validation_works()
    {
        $response = $this->post(route('community.submit.store'), [
            // Missing required fields
            'consent_gallery' => true
        ]);

        $response->assertStatus(302);
        $response->assertSessionHasErrors(['photo', 'name', 'location', 'caption']);
    }

    /** @test */
    public function community_post_caption_minimum_length_validation()
    {
        Storage::fake('public');
        
        $file = UploadedFile::fake()->image('test-style.jpg');

        $response = $this->post(route('community.submit.store'), [
            'photo' => $file,
            'name' => 'Test User',
            'location' => 'Melbourne, VIC',
            'caption' => 'Too short', // Less than 50 characters
            'consent_gallery' => true
        ]);

        $response->assertStatus(302);
        $response->assertSessionHasErrors(['caption']);
    }

    /** @test */
    public function approved_posts_show_on_community_index()
    {
        // Create approved post
        $approvedPost = CommunityPost::create([
            'name' => 'Approved User',
            'location' => 'Sydney, NSW',
            'caption' => 'This is an approved community post that should appear on the community index page.',
            'image_path' => 'community/approved-post.jpg',
            'consent_gallery' => true,
            'status' => 'approved'
        ]);

        // Create pending post (should not appear)
        $pendingPost = CommunityPost::create([
            'name' => 'Pending User',
            'location' => 'Brisbane, QLD',
            'caption' => 'This is a pending community post that should not appear on the community index page.',
            'image_path' => 'community/pending-post.jpg',
            'consent_gallery' => true,
            'status' => 'pending'
        ]);

        $response = $this->get(route('community.index'));
        
        $response->assertStatus(200);
        $response->assertSee('Approved User');
        $response->assertSee('Sydney, NSW');
        $response->assertDontSee('Pending User');
    }

    /** @test */
    public function community_post_model_scopes_work()
    {
        // Create posts with different statuses
        CommunityPost::create([
            'name' => 'Approved User',
            'location' => 'Melbourne, VIC',
            'caption' => 'Approved post content goes here and needs to be at least fifty characters long.',
            'image_path' => 'community/approved.jpg',
            'consent_gallery' => true,
            'status' => 'approved'
        ]);

        CommunityPost::create([
            'name' => 'Pending User',
            'location' => 'Sydney, NSW',
            'caption' => 'Pending post content goes here and needs to be at least fifty characters long.',
            'image_path' => 'community/pending.jpg',
            'consent_gallery' => true,
            'status' => 'pending'
        ]);

        CommunityPost::create([
            'name' => 'Featured User',
            'location' => 'Brisbane, QLD',
            'caption' => 'Featured post content goes here and needs to be at least fifty characters long.',
            'image_path' => 'community/featured.jpg',
            'consent_gallery' => true,
            'status' => 'approved',
            'featured' => true
        ]);

        $this->assertEquals(2, CommunityPost::approved()->count());
        $this->assertEquals(1, CommunityPost::pending()->count());
        $this->assertEquals(1, CommunityPost::featured()->count());
    }

    /** @test */
    public function community_post_instagram_url_accessor_works()
    {
        $post = CommunityPost::create([
            'name' => 'Test User',
            'location' => 'Perth, WA',
            'caption' => 'Test post content that needs to be at least fifty characters long for validation.',
            'image_path' => 'community/test.jpg',
            'instagram_handle' => 'testuser',
            'consent_gallery' => true,
            'status' => 'approved'
        ]);

        $this->assertEquals('https://instagram.com/testuser', $post->instagram_url);
    }
}