<?php

// validate.php
// Save this file in your project root and run: php validate.php
// This will check if all necessary files are in place

echo "=====================================\n";
echo "Laravel Export Portfolio Validator\n";
echo "=====================================\n\n";

$errors = [];
$warnings = [];
$success = [];

// Check PHP version
if (version_compare(PHP_VERSION, '8.1.0', '>=')) {
    $success[] = "✓ PHP version " . PHP_VERSION . " is compatible";
} else {
    $errors[] = "✗ PHP version " . PHP_VERSION . " is too old. Need PHP 8.1+";
}

// Check if composer.json exists
if (file_exists('composer.json')) {
    $success[] = "✓ composer.json exists";
} else {
    $errors[] = "✗ composer.json not found";
}

// Check if .env exists
if (file_exists('.env')) {
    $success[] = "✓ .env file exists";
    
    // Check database configuration
    $env = file_get_contents('.env');
    if (strpos($env, 'DB_DATABASE=export_portfolio') !== false) {
        $success[] = "✓ Database configured correctly";
    } else {
        $warnings[] = "⚠ Database name should be 'export_portfolio'";
    }
} else {
    $errors[] = "✗ .env file not found";
}

// Check Models
$models = [
    'app/Models/Category.php',
    'app/Models/Product.php',
    'app/Models/Inquiry.php',
    'app/Models/CompanyInfo.php',
    'app/Models/Testimonial.php'
];

echo "Checking Models...\n";
foreach ($models as $model) {
    if (file_exists($model)) {
        $success[] = "✓ $model exists";
    } else {
        $errors[] = "✗ $model is missing";
    }
}

// Check Controllers
$controllers = [
    'app/Http/Controllers/HomeController.php',
    'app/Http/Controllers/Admin/DashboardController.php',
    'app/Http/Controllers/Admin/ProductController.php',
    'app/Http/Controllers/Admin/CategoryController.php',
    'app/Http/Controllers/Admin/InquiryController.php',
    'app/Http/Controllers/Admin/TestimonialController.php',
    'app/Http/Controllers/Admin/CompanyInfoController.php'
];

echo "\nChecking Controllers...\n";
foreach ($controllers as $controller) {
    if (file_exists($controller)) {
        $success[] = "✓ $controller exists";
    } else {
        $errors[] = "✗ $controller is missing";
    }
}

// Check Views
$views = [
    'resources/views/layouts/app.blade.php',
    'resources/views/home.blade.php',
    'resources/views/about.blade.php',
    'resources/views/products.blade.php',
    'resources/views/product-detail.blade.php',
    'resources/views/contact.blade.php',
    'resources/views/admin/layouts/admin.blade.php',
    'resources/views/admin/dashboard.blade.php'
];

echo "\nChecking Views...\n";
foreach ($views as $view) {
    if (file_exists($view)) {
        $success[] = "✓ $view exists";
    } else {
        $errors[] = "✗ $view is missing";
    }
}

// Check Routes
if (file_exists('routes/web.php')) {
    $success[] = "✓ routes/web.php exists";
    
    // Check if routes file contains our custom routes
    $routes = file_get_contents('routes/web.php');
    if (strpos($routes, 'HomeController') !== false) {
        $success[] = "✓ Routes configured correctly";
    } else {
        $warnings[] = "⚠ Routes file may not be updated";
    }
} else {
    $errors[] = "✗ routes/web.php not found";
}

// Check Config
if (file_exists('config/seotools.php')) {
    $success[] = "✓ config/seotools.php exists";
} else {
    $warnings[] = "⚠ config/seotools.php is missing (optional)";
}

// Check Migrations
$migrations = glob('database/migrations/*_create_*.php');
if (count($migrations) >= 5) {
    $success[] = "✓ Migration files found";
} else {
    $warnings[] = "⚠ Some migration files may be missing";
}

// Check Seeder
if (file_exists('database/seeders/DatabaseSeeder.php')) {
    $success[] = "✓ DatabaseSeeder.php exists";
} else {
    $errors[] = "✗ DatabaseSeeder.php not found";
}

// Check vendor folder
if (is_dir('vendor')) {
    $success[] = "✓ Vendor folder exists";
} else {
    $errors[] = "✗ Vendor folder not found - run: composer install";
}

// Check node_modules
if (is_dir('node_modules')) {
    $success[] = "✓ Node modules installed";
} else {
    $warnings[] = "⚠ Node modules not installed - run: npm install";
}

// Check storage link
if (is_link('public/storage')) {
    $success[] = "✓ Storage link exists";
} else {
    $warnings[] = "⚠ Storage link not created - run: php artisan storage:link";
}

// Print Results
echo "\n=====================================\n";
echo "VALIDATION RESULTS\n";
echo "=====================================\n\n";

if (count($success) > 0) {
    echo "SUCCESS (" . count($success) . " items):\n";
    foreach ($success as $item) {
        echo $item . "\n";
    }
}

if (count($warnings) > 0) {
    echo "\nWARNINGS (" . count($warnings) . " items):\n";
    foreach ($warnings as $item) {
        echo $item . "\n";
    }
}

if (count($errors) > 0) {
    echo "\nERRORS (" . count($errors) . " items):\n";
    foreach ($errors as $item) {
        echo $item . "\n";
    }
    
    echo "\n=====================================\n";
    echo "❌ VALIDATION FAILED\n";
    echo "Please fix the errors above before proceeding.\n";
    echo "=====================================\n";
} else {
    echo "\n=====================================\n";
    echo "✅ VALIDATION PASSED\n";
    echo "Your installation appears to be correct!\n";
    echo "\nNext steps:\n";
    echo "1. Run: php artisan migrate:fresh --seed\n";
    echo "2. Run: php artisan serve\n";
    echo "3. Visit: http://localhost:8000\n";
    echo "=====================================\n";
}

echo "\n";