<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class CustomerController extends Controller
{


    public function load_customers(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

     
        $query = DB::table('customers')
            ->leftJoin('users as salesman', 'customers.saleman_id', '=', 'salesman.id')
            ->leftJoin('users as creator', 'customers.created_by', '=', 'creator.id')
            ->leftJoin('customer_ledgers', 'customers.id', '=', 'customer_ledgers.cus_id')
            ->select(
                'customers.id',
                'customers.name',
                'customers.company_name',
                'customers.saleman_id',
                'customers.city_id',
                'customers.area_id',
                'customers.admin_id',
                'customers.phone',
                DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),
                'customers.cnic',
                'customers.address',
                'customers.created_by',
                'customers.updated_by',
                'customers.cus_image',
                'customers.created_at',
                'customers.updated_at',
                'salesman.name as saleman_name',
                'creator.name as created_by_name',
                // FIXED: Exclude Sale Return totals from balance calculation and subtract return_amount
                DB::raw('
                    COALESCE(SUM(CASE WHEN customer_ledgers.adjustment != "Sale Return" THEN customer_ledgers.total ELSE 0 END), 0)
                    - (COALESCE(SUM(customer_ledgers.credit), 0) + COALESCE(SUM(customer_ledgers.bank), 0))
                    - COALESCE(SUM(customer_ledgers.return_amount), 0)
                    as ledger_balance
                ')
        )
        ->groupBy('customers.id',
            'customers.name',
            'customers.company_name',
            'customers.saleman_id',
            'customers.city_id',
            'customers.area_id',
            'customers.admin_id',
            'customers.phone',
            'customers.shop_name',
            'customers.cnic',
            'customers.address',
            'customers.created_by',
            'customers.updated_by',
            'customers.cus_image',
            'customers.created_at',
            'customers.updated_at',
            'salesman.name',
            'creator.name'
        );  

    


        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('customers.saleman_id', Auth::id());
        }


        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('created_at', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) {
            $query->where('customers.name', 'like', '%' . $search . '%');
        }



        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $users = $query->get();

   

        $all_data = [];
        foreach ($users as $user) {

           

            $btn = '<td class="text-right">';

            // View button
            $btn .= '<a href="' . route("customer_ledger", $user->id) . '" 
            class="btn btn-success btn-icon" data-toggle="tooltip" title="View" style="padding: 6px 10px;"><i class="fa-duotone fa-memo" style="margin-right: 3px"></i></a>';

            if (Auth::user()->can('update_customer')) {
                $btn .= '<button id="edit_btn" data-eid="'.$user->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_customer')) {
                $btn .= '<button id="delete_btn" data-did="'.$user->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            
            if($user->cus_image != ""){

                $image = '<img   height="50px" src="uploads/cus_image/'.$user->cus_image.'">';

            }else{

                $image = '<img  " height="50px" src="uploads/no_image.png">';

            }

            $all_data[] = [
                'id'                        => $user->id,
                'shop_name'                 => $user->shop_name,
                'name'                      => $user->name,
                'saleman_id'                => $user->saleman_name,
                'created_by_name'           => $user->created_by_name,
                'phone'                     => $user->phone,
                'ledger_balance'            => $user->ledger_balance,
                'cus_image'                 => $image,
                'btn'                       => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
        ];

        return response()->json($data);


    }

    public function insert_customer(Request $request)
    {
        try {
            $user_cnic = Customer::where('name', $request->input('cus_name'))->exists();

            if($user_cnic){

                return response()->json([
                    'status'=> "cnic_Exist",
                ]);

            }else{

                if ($request->hasFile('cus_image')) {
                    $image1 = $request->file('cus_image');
                    $cus_image = hexdec(uniqid()).'.'.$image1->getClientOriginalExtension();
                    $image1->move(public_path('/uploads/cus_image/'), $cus_image);
                } else {
                    $cus_image = null;
                }

                $data = new Customer();
                $data->name             = $request->input('cus_name');
                $data->phone            = $request->input('phone');
                $data->shop_name        = $request->input('shop_name');
                $data->cnic             = $request->input('cnic');
                $data->address          = $request->input('address');
                $data->cus_image        = $cus_image;
                $data->saleman_id       = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
                $data->city_id         = $request->input('city_id');
                $data->area_id         = $request->input('area_id');

                $data->created_by       = Auth::id();

                $data->save();

                // Handle Old Balance (Opening Balance)
                $old_balance = $request->input('old_balance', 0);
                if ($old_balance > 0) {
                    try {
                        // Create a customer ledger entry for the old balance
                        $ledger = new CustomerLedger();
                        $ledger->cus_id           = $data->id;
                        $ledger->invoice_date     = Carbon::now()->format('Y-m-d');
                        $ledger->detail           = 'Opening Balance';
                        $ledger->transaction_type = 'debit'; // Use 'debit' which is valid
                        $ledger->adjustment       = 'Opening'; // Set adjustment so it's included in balance calculation
                        $ledger->total            = $old_balance;
                        $ledger->debit            = $old_balance; // Set debit instead of 0
                        $ledger->credit           = 0;
                        $ledger->bank             = 0;
                        $ledger->save();
                    } catch (\Exception $e) {
                        Log::error('Error creating customer ledger for opening balance: ' . $e->getMessage());
                        Log::error('Stack trace: ' . $e->getTraceAsString());
                        // Continue anyway - customer is created
                    }
                }

                return response()->json([
                    'status'=>200
                ]);

            }
        } catch (\Exception $e) {
            Log::error('Error in insert_customer: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'Error creating customer: ' . $e->getMessage()
            ], 500);
        }
    }

    public function edit_customer($id)
    {

        $data = DB::table('customers')
            ->leftJoin('cities', 'customers.city_id', '=', 'cities.id')
            ->leftJoin('areas', 'customers.area_id', '=', 'areas.id')
            ->select(
                'customers.*',
                'cities.city_name',
                'areas.area_name'
            )
            ->where('customers.id', $id)
            ->first();

        return response()->json([
            'status'=>200,
            'data'=>$data,
        ]);
    }

    public function update_customer(Request $request)
    {
        $data = Customer::find($request->input('cus_id'));

        if(!$data){
            return response()->json([
                'status'=> 404,
            ]);
        }

        if ($request->hasFile('u_cus_image')) {
            $image1 = $request->file('u_cus_image');
            $cus_image = hexdec(uniqid()).'.'.$image1->getClientOriginalExtension();
            $image1->move(public_path('/uploads/cus_image/'), $cus_image);
        } else {
            $cus_image = $data->cus_image;
        }

        $data->name             = $request->input('u_cus_name');
        $data->phone            = $request->input('u_phone');
        $data->shop_name        = $request->input('u_shop_name');
        $data->cnic             = $request->input('u_cnic');
        $data->address          = $request->input('u_address');
        $data->cus_image        = $cus_image;
        $data->city_id         = $request->input('u_city_id');
        $data->area_id         = $request->input('u_area_id');
        $data->saleman_id       = $request->input('u_salesman_id') ? $request->input('u_salesman_id') : $data->saleman_id;
        $data->updated_by       = Auth::id();

        $data->save();

        return response()->json([
            'status'=>200
        ]);
    }

    public function delete_customer($id)
    {
        $data = Customer::find($id);
        $data->delete();

        return response()->json([
            'status'=>200
        ]);
    }

    public function get_customers()
    {
        // Check if user has Admin or Super Admin role
        if (Auth::user()->hasRole(['Admin', 'Super Admin', 'Store man'])) {
            // Admin/Super Admin can see all customers
            $customer = Customer::select('id', 'name', 'company_name', 'shop_name')->get();
        } else {
            // Other users get only their own created customers
            $customer = Customer::where('saleman_id', Auth::id())->select('id', 'name', 'company_name', 'shop_name')->get();
        }

        return response()->json($customer);
    }

    public function getCustomersBySalesman($salesman_id)
    {
        $customers = Customer::where('saleman_id', $salesman_id)->get(['id', 'name', 'company_name', 'shop_name']);
        return response()->json($customers);
    }

    public function customer()
    {
        try {
            $customer_id = null;
            
            // If user has Customer role, find their customer record and show ledger
            if (Auth::user()->hasRole('Customer')) {
                $user = Auth::user();
                $customer = null;
                
                // First try to find by created_by (most reliable match)
                $customer = Customer::where('created_by', $user->id)->first();
                
                // If not found, try to extract username from email and match
                if (!$customer && $user->email) {
                    $emailUsername = strtolower(explode('@', $user->email)[0]);
                    $customer = Customer::whereRaw('LOWER(TRIM(name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                    
                    if (!$customer) {
                        $customers = Customer::all();
                        foreach ($customers as $cust) {
                            $custNameClean = strtolower(preg_replace('/^[mM][-_\s]?/', '', $cust->name ?? ''));
                            $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                            $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                            if ($custNameClean === $emailUsernameClean || 
                                strpos($custNameClean, $emailUsernameClean) !== false ||
                                strpos($emailUsernameClean, $custNameClean) !== false) {
                                $customer = $cust;
                                break;
                            }
                        }
                    }
                }
                
                // If not found by email, try to find by matching name (case-insensitive)
                if (!$customer && $user->name) {
                    $customer = Customer::whereRaw('LOWER(name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
                    
                    if (!$customer) {
                        $customer = Customer::whereRaw('LOWER(name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
                    }
                }
                
                // If still not found, try exact match ignoring case
                if (!$customer && $user->name) {
                    $customer = Customer::whereRaw('LOWER(TRIM(name)) = ?', [strtolower(trim($user->name))])->first();
                }
                
                // If still not found, try partial match (remove spaces, prefixes and compare)
                if (!$customer && $user->name) {
                    $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                    $customers = Customer::all();
                    foreach ($customers as $cust) {
                        $custNameClean = preg_replace('/^[mM][-_\s]?/', '', strtolower($cust->name ?? ''));
                        $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                        
                        // Exact match
                        if ($userNameClean === $custNameClean) {
                            $customer = $cust;
                            break;
                        }
                        // Check if one contains the other
                        if (strpos($custNameClean, $userNameClean) !== false || 
                            strpos($userNameClean, $custNameClean) !== false) {
                            $customer = $cust;
                            break;
                        }
                        // Similar match (using similar_text for fuzzy matching)
                        $similarity = 0;
                        similar_text($userNameClean, $custNameClean, $similarity);
                        if ($similarity > 80) {
                            $customer = $cust;
                            break;
                        }
                    }
                }
                
                if ($customer) {
                    $customer_id = $customer->id;
                    return redirect()->route('customer_ledger', $customer_id);
                } else {
                    Log::warning('Customer record not found for user: ' . $user->email . ' (ID: ' . $user->id . ')');
                    return redirect()->route('dashboard')->with('error', 'Customer record not found. Please contact administrator.');
                }
            }
            
            // For non-Customer roles: show customer list
            return view('admin.customers');
        } catch (\Exception $e) {
            Log::error('Error in customer() method: ' . $e->getMessage());
            return redirect()->route('dashboard')->with('error', 'An error occurred. Please try again.');
        }
    }

    public function show_customer_ledger($id)
    {
        // If user has Customer role, validate they can only access their own ledger
        if (Auth::user()->hasRole('Customer')) {
            $user = Auth::user();
            $customer = null;
            
            $customer = Customer::where('created_by', $user->id)->first();
            
            if (!$customer && $user->email) {
                $emailUsername = strtolower(explode('@', $user->email)[0]);
                $customer = Customer::whereRaw('LOWER(TRIM(name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                
                if (!$customer) {
                    $customers = Customer::all();
                    foreach ($customers as $cust) {
                        $custNameClean = strtolower(preg_replace('/^[mM][-_\s]?/', '', $cust->name ?? ''));
                        $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                        $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                        if ($custNameClean === $emailUsernameClean || 
                            strpos($custNameClean, $emailUsernameClean) !== false ||
                            strpos($emailUsernameClean, $custNameClean) !== false) {
                            $customer = $cust;
                            break;
                        }
                    }
                }
            }
            
            if (!$customer && $user->name) {
                $customer = Customer::whereRaw('LOWER(name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
            }
            
            if (!$customer && $user->name) {
                $customer = Customer::whereRaw('LOWER(TRIM(name)) = ?', [strtolower(trim($user->name))])->first();
            }
            
            if (!$customer && $user->name) {
                $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                $customers = Customer::all();
                foreach ($customers as $cust) {
                    $custNameClean = preg_replace('/^[mM][-_\s]?/', '', strtolower($cust->name ?? ''));
                    $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                    if ($userNameClean === $custNameClean ||
                        strpos($custNameClean, $userNameClean) !== false ||
                        strpos($userNameClean, $custNameClean) !== false) {
                        $customer = $cust;
                        break;
                    }
                    $similarity = 0;
                    similar_text($userNameClean, $custNameClean, $similarity);
                    if ($similarity > 80) {
                        $customer = $cust;
                        break;
                    }
                }
            }
            
            if ($customer && $customer->id != $id) {
                return redirect()->route('customer_ledger', $customer->id);
            } elseif (!$customer) {
                return redirect()->route('dashboard')->with('error', 'Customer record not found. Please contact administrator.');
            }
        }
        
        // For other roles or valid access, show the customer ledger page
        return view('admin.customer_ledger');
    }

    // ledger area 
    public function load_customer_ledger(Request $request)
    {
        try {
            $cus_id = $request->input('cus_id');
            
            // Validate cus_id
            if (empty($cus_id)) {
                return response()->json([
                    'draw' => intval($request->input('draw', 0)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Customer ID is required'
                ], 400);
            }

        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");
    
            // CRITICAL: If user has Customer role, FORCE them to only see their own data
            if (Auth::user()->hasRole('Customer')) {
                $user = Auth::user();
                $customer = null;
                
                $customer = Customer::where('created_by', $user->id)->first();
                
                if (!$customer && $user->email) {
                    $emailUsername = strtolower(explode('@', $user->email)[0]);
                    $customer = Customer::whereRaw('LOWER(TRIM(name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                    
                    if (!$customer) {
                        $customers = Customer::all();
                        foreach ($customers as $cust) {
                            $custNameClean = strtolower(preg_replace('/^[mM][-_\s]?/', '', $cust->name ?? ''));
                            $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                            $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                            if ($custNameClean === $emailUsernameClean || 
                                strpos($custNameClean, $emailUsernameClean) !== false ||
                                strpos($emailUsernameClean, $custNameClean) !== false) {
                                $customer = $cust;
                                break;
                            }
                        }
                    }
                }
                
                if (!$customer && $user->name) {
                    $customer = Customer::whereRaw('LOWER(name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
                }
                
                if (!$customer && $user->name) {
                    $customer = Customer::whereRaw('LOWER(TRIM(name)) = ?', [strtolower(trim($user->name))])->first();
                }
                
                if (!$customer && $user->name) {
                    $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                    $customers = Customer::all();
                    foreach ($customers as $cust) {
                        $custNameClean = preg_replace('/^[mM][-_\s]?/', '', strtolower($cust->name ?? ''));
                        $custNameClean = preg_replace('/\s+/', '', $custNameClean);
                        
                        if ($userNameClean === $custNameClean ||
                            strpos($custNameClean, $userNameClean) !== false ||
                            strpos($userNameClean, $custNameClean) !== false) {
                            $customer = $cust;
                            break;
                        }
                        $similarity = 0;
                        similar_text($userNameClean, $custNameClean, $similarity);
                        if ($similarity > 80) {
                            $customer = $cust;
                            break;
                        }
                    }
                }
                
                if ($customer) {
                    $cus_id = $customer->id;
                    Log::info('Customer role user ' . $user->email . ' accessing their own ledger (Customer ID: ' . $cus_id . ')');
                } else {
                    Log::warning('Customer role user ' . $user->email . ' - customer record not found');
                    return response()->json([
                        'draw' => intval($request->input('draw', 0)),
                        'recordsTotal' => 0,
                        'recordsFiltered' => 0,
                        'data' => [],
                        'error' => 'Access denied. Customer record not found.'
                    ], 403);
                }
            }
    
            // Build the query with potential filters
        $query = DB::table('customer_ledgers')
            ->where('customer_ledgers.cus_id', '=', $cus_id)
        ->leftJoin('banks', 'banks.id', '=', 'customer_ledgers.bank_id')
        ->leftJoin('sales', 'sales.id', '=', 'customer_ledgers.sale_id')
        ->leftJoin('sale_returns', 'sale_returns.id', '=', 'customer_ledgers.sale_return_id')
        ->select(
            'customer_ledgers.*',
            'banks.bank_name',
            'sales.id as sale_id',
            'sale_returns.id as sale_return_id',
            'sale_returns.status as return_status'
        )
        ->whereNull('customer_ledgers.deleted_at')
        ->where(function($q) {
            $q->whereNull('customer_ledgers.sale_return_id')
              ->orWhere(function($q2) {
                  $q2->whereNotNull('customer_ledgers.sale_return_id')
                     ->where('sale_returns.status', '=', 'approved');
              });
        });

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('customer_ledgers.invoice_date', [$start_date, $end_date]);
        }

        \Log::info('Customer Ledger Query:', [
            'customer_id' => $cus_id,
            'query' => $query->toSql(),
            'bindings' => $query->getBindings()
        ]);
    
        if ($search = $request->input('search')) {
            $query->where('detail', 'like', '%' . $search . '%');
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('transaction_type', '=', $type);
        }
    
        $totalsQuery = clone $query;
        // FIXED: Exclude Sale Return totals from total_amount calculation
        $totals = $totalsQuery->select([
            DB::raw('SUM(CASE WHEN adjustment != "Sale Return" THEN total ELSE 0 END) as total_amount'),
            DB::raw('SUM(credit) as total_credit'),
            DB::raw('SUM(bank) as total_bank'),
            DB::raw('SUM(COALESCE(return_amount, 0)) as total_return'),
            DB::raw('SUM(debit) as total_debit')
        ])->first();
    
        // Calculate total before pagination
        $total_count = $query->count();
    
        // Fetch the data
        $supplier_ledger = $query->get();

        \Log::info('Customer Ledger Records Fetched:', [
            'total_records' => $supplier_ledger->count(),
            'sale_returns' => $supplier_ledger->where('sale_return_id', '!=', null)->count(),
            'records' => $supplier_ledger->map(function($item) {
                return [
                    'id' => $item->id,
                    'sale_return_id' => $item->sale_return_id,
                    'return_status' => $item->return_status,
                    'transaction_type' => $item->transaction_type,
                    'adjustment' => $item->adjustment
                ];
            })
        ]);

        $all_data = [];
        $currentBalance = 0;

        foreach ($supplier_ledger as $data) {

            // CRITICAL FIX: Correct balance calculation
            // Balance = What customer OWES (Total increases debt, Payments/Returns reduce debt)
            if ($data->detail == "Opening Balance") {
                // Opening Balance: Customer owes this amount from the start
                $currentBalance += $data->debit;
            } else if ($data->adjustment == "Sale") {
                // Sale: Customer owes more (total minus what they paid)
                $currentBalance += $data->total - $data->credit - $data->bank;
            } else if ($data->transaction_type == "Sale_return" || $data->adjustment == "Sale Return") {
                // CRITICAL FIX: Sale Return reduces customer debt by the return_amount
                // Use return_amount (product value) NOT debit (cash refund)
                // Debit only represents cash refund, return_amount represents debt cancellation
                $returnAmount = $data->return_amount ?? $data->total ?? 0;
                // Only reduce balance by the amount they actually owed (cannot go negative from returns)
                $reduction = min($returnAmount, max(0, $currentBalance));
                $currentBalance -= $reduction;
            } else if ($data->transaction_type == "Payment Receive") {
                // Payment: Customer owes less (payment reduces debt)
                $currentBalance -= ($data->credit + $data->bank);
            } else if ($data->debit > 0) {
                // Any other debit reduces balance
                $currentBalance -= $data->debit;
            } else if ($data->credit > 0 || $data->bank > 0) {
                // Any other credit/bank payment reduces balance
                $currentBalance -= ($data->credit + $data->bank);
            }

    
            // Default transaction type (to prevent undefined errors)
            $transaction_type = '<div class="label label-table label-default" style="width:100%; max-width:130px;">Unknown</div>';
    
            if ($data->transaction_type == "Opening Account") {
                $transaction_type = '<div class="label label-table label-purple" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } else if ($data->transaction_type == "Sale") {
                $transaction_type = '<div class="label label-table label-warning" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } else if ($data->transaction_type == "Sale_return") {
                $transaction_type = '<div class="label label-table label-danger" style="width:100%; max-width:130px;">Sale Return</div>';
            } else if ($data->transaction_type == "Payment Receive") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            }else if ($data->transaction_type == "Payment Return") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } 
            else if ($data->transaction_type == "Cash") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            }
            else if ($data->transaction_type == "credit") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            }
              else if ($data->transaction_type == "debit") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            }
            else if ($data->transaction_type == "Bank") {
                $transaction_type = '';
            }
    
            $btn = "";

            // Check if this is a Sale Return transaction - show view sale return button
            if ($data->transaction_type == "Sale_return" || ($data->sale_return_id != null && $data->sale_return_id > 0)) {
                $btn = '<td class="text-right">
                            <button id="view_sale_return_btn" data-return_id="' . $data->sale_return_id . '" class="btn btn-danger btn-icon" data-toggle="tooltip" title="View Sale Return" style="margin-right: 3px;"><i class="fa-duotone fa-arrows-to-eye"></i></button>
                        </td>';
            }
            // Always show invoice view button if sale_id exists
            else if ($data->sale_id != null || $data->sale_id > 0) {
                $btn = '<td class="text-right">
                            <button id="view_sale_btn" data-sale_id="' . $data->sale_id . '" class="btn btn-info btn-icon" data-toggle="tooltip" title="View Invoice" style="margin-right: 3px;"><i class="fa-duotone fa-arrows-to-eye"></i></button>
                        </td>';
            }
            // For other transactions, show empty action (no edit/delete buttons)
            else {
                $btn = '<td class="text-right"></td>';
            }
    
            // Display Sale ID or Return ID based on transaction type
            $displayId = '---';
            if ($data->transaction_type == "Sale_return" && !empty($data->sale_return_id)) {
                $displayId = 'SR-' . $data->sale_return_id;
            } else if (!empty($data->sale_id)) {
                $displayId = 'S-' . $data->sale_id;
            }

            $all_data[] = [
                'id'                        => $data->id,
                'invoice_date'              => Carbon::parse($data->invoice_date)->format('d-m-Y'),
                'transaction_details'       => $data->detail,
                'transaction_type'          => $transaction_type .''. $data->bank_name,
                'sale_id'                   => $displayId,
                'sale_return_id'            => !empty($data->sale_return_id) ? $data->sale_return_id : null,
                'return_status'             => !empty($data->return_status) ? $data->return_status : null,
                'adjustment'                => $data->adjustment,
                'total'                     => number_format($data->total, 0),
                'credit'                    => number_format($data->credit, 0),
                'debit'                     => number_format($data->debit, 0),
                'bank'                      => number_format($data->bank, 0),
                'balance'                   => number_format($currentBalance, 0),
                'btn'                       => $btn
            ];
        }
    
        $cus_name = Customer::find($cus_id);
        
        if (!$cus_name) {
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Customer not found.'
            ], 404);
        }
    
        // FIXED: Correct total balance calculation
        // Balance = Sale Totals - Credits - Bank Payments - Return Amounts
        $totalAmount = $totals->total_amount ?? 0;
        $totalCredit = $totals->total_credit ?? 0;
        $totalBank = $totals->total_bank ?? 0;
        $totalReturn = $totals->total_return ?? 0;
        $calculatedBalance = $totalAmount - $totalCredit - $totalBank - $totalReturn;
        // Ensure balance doesn't go negative from returns
        $calculatedBalance = max(0, $calculatedBalance);

        $total_results = array(
            "total_amount"     => ($totalAmount > 0) ? number_format($totalAmount) : 0,
            "total_debit"     => ($totals->total_debit > 0) ? number_format($totals->total_debit) : 0,
            "total_credit"    => ($totalCredit + $totalBank > 0) ? number_format($totalCredit + $totalBank) : 0,
            "total_balance"   => number_format($calculatedBalance),
            
            "customer_name"  => $cus_name->name . " Ledger (ID: " . $cus_name->id . ")",
            "shop_name"  => $cus_name->shop_name,
        );
    
        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
            "totals"          => $total_results,
        ];
    
        return response()->json($data);
        } catch (\Exception $e) {
            Log::error('Error in load_customer_ledger: ' . $e->getMessage());
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'An error occurred while loading customer ledger.'
            ], 500);
        }
    }
    



    public function insert_customer_payment(Request $request)
    {
        
        // Set credit and debit amounts based on transaction type
        $credit = $request->input('transaction_type') == "credit" ? $request->input('amount') : 0;
        $debit = $request->input('transaction_type') == "debit" ? $request->input('amount') : 0;
        $bank = $request->input('transaction_type') == "bank" ? $request->input('amount') : 0;

        if ($request->transaction_type === "credit") {
            // hand_cash column
            DB::table('hand_cashes')->increment('cash_amount', $request->amount);

        } elseif ($request->transaction_type === "bank") {

            DB::table('banks')->where('id', $request->bank_id)
            ->increment('bank_balance', $request->amount);
        }

    
        $customer_ledger = new CustomerLedger();
        $customer_ledger->cus_id = $request->input('cus_pay_id');
        $customer_ledger->invoice_date = Carbon::parse($request->input('transaction_date'))->format('Y-m-d');
        $customer_ledger->detail = $request->input('transaction_details') ?: "Payment Received";
        $customer_ledger->transaction_type = $request->input('transaction_type');
        $customer_ledger->bank_id = $request->input('bank_id');

        $customer_ledger->credit = $credit;
        $customer_ledger->debit = $debit;
        $customer_ledger->bank = $bank;
        $customer_ledger->total   = $credit + $bank;
        $customer_ledger->created_by = Auth::id();
        $customer_ledger->updated_by = Auth::id();
        $customer_ledger->save();
    
        return response()->json([
            'status' => 200
        ]);
    }


    public function edit_customer_payment($id)
    {
        // model relation ship
        $Data = CustomerLedger::with('bank')->find($id);

        if ($Data && $Data->transaction_date) {
            $Data->transaction_date = Carbon::parse($Data->transaction_date)->format('d-m-Y');
        }
        if ($Data) {
            $Data->ledger_bank_amount = $Data->bank;
        }


        return response()->json([
            'status' => 200,
            'data'   => $Data,
        ]);
    }


    public function update_customer_payment(Request $request)
    {
        $customer_ledger = CustomerLedger::find($request->input('pay_id'));

        if (!$customer_ledger) {
            return response()->json(['status' => 404, 'message' => 'Payment record not found']);
        }

        // Step 1: Reverse old payment effect
        if ($customer_ledger->transaction_type === "credit") {
            DB::table('hand_cashes')->decrement('cash_amount', $customer_ledger->credit);

        } elseif ($customer_ledger->transaction_type === "Bank") {
            DB::table('banks')->where('id', $customer_ledger->bank_id)
                ->decrement('bank_balance', $customer_ledger->bank);
        }

        // Step 2: Prepare new values
        $credit = 0;
        $debit  = 0;
        $bank   = 0;

        if ($request->transaction_type === "credit") {
            $credit = $request->amount;

        } elseif ($request->transaction_type === "debit") {
            $debit = $request->amount;

        } elseif ($request->transaction_type === "bank") {
            $bank = $request->amount;
        }

        // Step 3: Update ledger
        $customer_ledger->invoice_date     = Carbon::parse($request->transaction_date)->format('Y-m-d');
        $customer_ledger->detail           = $request->transaction_details ?: "Payment";
        $customer_ledger->transaction_type = $request->transaction_type;
        $customer_ledger->credit           = $credit;
        $customer_ledger->debit            = $debit;
        $customer_ledger->bank             = $bank;
        $customer_ledger->bank_id          = $request->bank_id;
        $customer_ledger->updated_by       = Auth::id();
        $customer_ledger->total            = $credit + $bank;
        $customer_ledger->save();

        // Step 4: Apply new effect
        if ($request->transaction_type === "credit") {
            DB::table('hand_cashes')->increment('cash_amount', $request->amount);

        } elseif ($request->transaction_type === "bank") {
            DB::table('banks')->where('id', $request->bank_id)
                ->increment('bank_balance', $request->amount);
        }

        return response()->json(['status' => 200, 'message' => 'Customer payment updated successfully']);
    }



    public function delete_customer_payment($delete_id)
    {
        $customer_ledger = CustomerLedger::find($delete_id);

        if (!$customer_ledger) {
            return response()->json([
                'status' => 404,
                'message' => 'Payment record not found'
            ]);
        }

        // Step 1: Reverse old payment effect
        if ($customer_ledger->transaction_type === "credit") {
            DB::table('hand_cashes')->decrement('cash_amount', $customer_ledger->credit);

        } elseif ($customer_ledger->transaction_type === "Bank") {
            DB::table('banks')->where('id', $customer_ledger->bank_id)
                ->decrement('bank_balance', $customer_ledger->bank);
        }

        // Step 2: Delete record
        $customer_ledger->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Customer payment deleted successfully'
        ]);
    }




}