<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\Bom;
use App\Models\CompanyInvestment;
use App\Models\Customer;
use App\Models\Expense;
use App\Models\HandCash;
use App\Models\Invoice;
use App\Models\Order;
use App\Models\OrderBooker;
use App\Models\OrderReturn;
use App\Models\Product;
use App\Models\Production;
use App\Models\Purchase;
use App\Models\Quotation;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\SaleReturn;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function dashboard(Request $request)
    {
        $user = Auth::user();

        
        // customers 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $customer = Customer::count();
        } else {
            // Normal user → only his own sales
            $customer = Customer::where('saleman_id', $user->id)->count();
        }


        // ========================================
        // CASH/BANK CALCULATIONS FROM APPROVED TRANSACTIONS
        // ========================================
        
        // Get base hand cash (initial amount)
        $base_hand_cash = HandCash::sum('cash_amount');
        
        // Calculate cash from approved salesman ledgers (Send to Cashier approved)
        $approved_salesman_cash = DB::table('saleman_ledgers')
            ->where('status', 'approved')
            ->where('detail', 'Send to cashier')
            ->whereNull('deleted_at')
            ->where(function($q) {
                $q->where('transaction_type', 'Cash')
                  ->orWhere('transaction_type', 'credit');
            })
            ->sum('saleman_cash');
        
        // Calculate expenses from approved salesman ledgers
        $approved_salesman_expenses = DB::table('saleman_ledgers')
            ->where('status', 'approved')
            ->where('detail', 'Send to cashier')
            ->whereNull('deleted_at')
            ->where('transaction_type', 'expense')
            ->sum('saleman_cash');
        
        // Calculate cash from approved order booker ledgers
        $approved_order_booker_cash = DB::table('order_booker_ledgers')
            ->where('status', 'approved')
            ->whereNull('deleted_at')
            ->sum('paid_amount');
        
        // Calculate expenses from approved order booker ledgers
        $approved_order_booker_expenses = DB::table('order_booker_ledgers')
            ->where('status', 'approved')
            ->whereNull('deleted_at')
            ->sum('expense_amount');
        
        // Total hand cash = base + approved cash from salesman + approved cash from order booker
        $hand_cash = $base_hand_cash;
        
        // Get all banks with their calculated balances
        $banks = Bank::all()->map(function($bank) {
            // Get base balance
            $base_balance = $bank->bank_balance;
            
            // Get approved bank amounts from salesman ledgers for this bank
            $approved_salesman_bank = DB::table('saleman_ledgers')
                ->where('status', 'approved')
                ->where('detail', 'Send to cashier')
                ->whereNull('deleted_at')
                ->where('transaction_type', 'Bank')
                ->where('bank_id', $bank->id)
                ->sum('saleman_cash');
            
            // Get approved bank amounts from order booker ledgers for this bank
            $approved_order_booker_bank = DB::table('order_booker_ledgers')
                ->where('status', 'approved')
                ->whereNull('deleted_at')
                ->where('bank_id', $bank->id)
                ->sum('paid_amount');
            
            // Update calculated balance (base already includes all transactions)
            $bank->calculated_balance = $base_balance;
            
            return $bank;
        });

        // Total approved expenses (for expense section)
        $total_approved_expenses = $approved_salesman_expenses + $approved_order_booker_expenses;


        // sales 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $totalSales = Sale::sum('grand_total');
            $sales_count = Sale::count();
            $sales_paid = Sale::sum('paid');
        } else {
            // Normal user → only his own sales
            $totalSales = Sale::where('saleman_id', $user->id)->sum('grand_total');
            $sales_count = Sale::where('saleman_id', $user->id)->count();
            $sales_paid = Sale::where('saleman_id', $user->id)->sum('paid');
        }
        $sales_receivable = max(0, $totalSales - $sales_paid);

        // sales return 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $sale_return = SaleReturn::sum('grand_total');
            $sale_return_count = SaleReturn::count();
            $sale_return_pending = SaleReturn::where('status', 'pending')->count();
            $sale_return_approved = SaleReturn::where('status', 'approved')->count();
        } else {
            // Normal user → only his own sales
            $sale_return = SaleReturn::where('saleman_id', $user->id)->sum('grand_total');
            $sale_return_count = SaleReturn::where('saleman_id', $user->id)->count();
            $sale_return_pending = SaleReturn::where('saleman_id', $user->id)->where('status', 'pending')->count();
            $sale_return_approved = SaleReturn::where('saleman_id', $user->id)->where('status', 'approved')->count();
        }

        // expense 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all expenses (total amount) + approved expenses from salesman/order booker
            $expense = Expense::sum('total_amt') + $total_approved_expenses;
        } else {
            // Normal user → only his own expenses (total amount)
            $expense = Expense::where('created_by', $user->id)->sum('total_amt');
        }

        // purchase 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all purchases
            $totalPurchases = Purchase::sum('grand_total');
            $purchases_count = Purchase::count();
            $purchase_paid_total = Purchase::sum('paid_amount');
        } else {
            // Normal user → only his own purchases
            $totalPurchases = Purchase::where('created_by', $user->id)->sum('grand_total');
            $purchases_count = Purchase::where('created_by', $user->id)->count();
            $purchase_paid_total = Purchase::where('created_by', $user->id)->sum('paid_amount');
        }
        $purchase_payable = max(0, $totalPurchases - $purchase_paid_total);

        // profit 
        $profit = SaleDetail::selectRaw('SUM((sale_price - purchase_price) * pro_qty) as total_profit')
        ->value('total_profit')
        - Expense::sum('total_amt');

        // customer ka udhar 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all 
            $customer_payable = Sale::sum('grand_total') - Sale::sum('paid');
        } else {
            // Normal user → only his own 
            $customer_payable = Sale::where('saleman_id', $user->id)->sum('grand_total') - Sale::where('saleman_id', $user->id)->sum('paid');
        }

        // saleman orders count 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $order = Order::count();
            $orders_pending = Order::where('status', 'pending')->count();
            $orders_approved = Order::where('status', 'approved')->count();
        } else {
            // Normal user → only his own sales
            $order = Order::where('salesman_id', $user->id)->count();
            $orders_pending = Order::where('salesman_id', $user->id)->where('status', 'pending')->count();
            $orders_approved = Order::where('salesman_id', $user->id)->where('status', 'approved')->count();
        }

        // saleman order return  count 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $order_return = OrderReturn::count();
            $order_return_pending = OrderReturn::where('status', 'pending')->count();
            $order_return_approved = OrderReturn::where('status', 'approved')->count();
        } else {
            // Normal user → only his own sales
            $order_return = OrderReturn::where('salesman_id', $user->id)->count();
            $order_return_pending = OrderReturn::where('salesman_id', $user->id)->where('status', 'pending')->count();
            $order_return_approved = OrderReturn::where('salesman_id', $user->id)->where('status', 'approved')->count();
        }

        // customer invoice count 
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            // Super admin/admin → all sales
            $invoice = Invoice::count();
        } else {
            // Normal user → only his own sales
            $invoice = Invoice::where('saleman_id', $user->id)->count();
        }

        // production count
        $production_total = Production::count();
        $production_pending = Production::where('status', 'pending')->count();
        $production_completed = Production::where('status', 'completed')->count();
        $production = Production::where('status', 'production')->count();
        $production_issue = Production::where('status', 'issue')->count();

        // top sale product
        $topProducts = DB::table('sale_details')
        ->join('products', 'sale_details.pro_id', '=', 'products.id')
        ->select(
            'products.pro_name',
            DB::raw('SUM(sale_details.pro_qty) as total_sold')
        )
        ->groupBy('products.pro_name')
        ->orderByDesc('total_sold')
        ->take(10) // top 10
        ->get();

        // top saleman 
        $topSalesman = DB::table('sales')
        ->join('users', 'sales.saleman_id', '=', 'users.id')
        ->select('users.id', 'users.name', DB::raw('SUM(sales.grand_total) as total_sales'))
        ->groupBy('users.id', 'users.name')
        ->orderByDesc('total_sales')
        ->first();

        // topCustomer
        $topCustomer = DB::table('sales')
        ->join('customers', 'sales.cus_id', '=', 'customers.id')
        ->select('customers.id', 'customers.name', DB::raw('SUM(sales.grand_total) as total_spent'))
        ->groupBy('customers.id', 'customers.name')
        ->orderByDesc('total_spent')
        ->first();

        // Investments (company + investor) and withdrawals
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            $company_investment = CompanyInvestment::sum('invested_amount');
            $investor_investment = DB::table('investments')->sum('invested_amount');
            $company_withdraw = DB::table('investment_withdraws')->where('type','Company')->sum('amount');
            $investor_withdraw = DB::table('investment_withdraws')->where('type','Investor')->sum('amount');
        } else {
            $company_investment = CompanyInvestment::where('created_by', $user->id)->sum('invested_amount');
            $investor_investment = DB::table('investments')->where('created_by', $user->id)->sum('invested_amount');
            $company_withdraw = DB::table('investment_withdraws')->where('created_by',$user->id)->where('type','Company')->sum('amount');
            $investor_withdraw = DB::table('investment_withdraws')->where('created_by',$user->id)->where('type','Investor')->sum('amount');
        }
        $total_investment = $company_investment + $investor_investment;
        $total_withdraw = $company_withdraw + $investor_withdraw;
        $investment_balance = $total_investment - $total_withdraw;

        // quotations
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            $quotations = Quotation::where('status', 'approved')->count();
            $quotations_total = Quotation::count();
            $quotations_pending = Quotation::where('status', 'pending')->count();
            // Quotations converted to sale (match by date, customer, salesman, and total)
            $quotations_convert_sale = DB::table('quotations')
                ->join('sales', function($join) {
                    $join->on('quotations.invoice_date', '=', 'sales.invoice_date')
                         ->on('quotations.cus_id', '=', 'sales.cus_id')
                         ->on('quotations.saleman_id', '=', 'sales.saleman_id')
                         ->on('quotations.grand_total', '=', 'sales.grand_total');
                })
                ->where('quotations.status', 'approved')
                ->count(DB::raw('DISTINCT quotations.id'));
        } elseif ($user->hasRole('Order Booker')) {
            // For Order Booker, show quotations created by them
            $quotations = Quotation::where('status', 'approved')->where('created_by', $user->id)->count();
            $quotations_total = Quotation::where('created_by', $user->id)->count();
            $quotations_pending = Quotation::where('status', 'pending')->where('created_by', $user->id)->count();
            // Quotations converted to sale
            $quotations_convert_sale = DB::table('quotations')
                ->join('sales', function($join) use ($user) {
                    $join->on('quotations.invoice_date', '=', 'sales.invoice_date')
                         ->on('quotations.cus_id', '=', 'sales.cus_id')
                         ->on('quotations.saleman_id', '=', 'sales.saleman_id')
                         ->on('quotations.grand_total', '=', 'sales.grand_total')
                         ->where('quotations.created_by', $user->id);
                })
                ->where('quotations.status', 'approved')
                ->count(DB::raw('DISTINCT quotations.id'));
        } else {
            $quotations = Quotation::where('status', 'approved')->where('saleman_id', $user->id)->count();
            $quotations_total = Quotation::where('saleman_id', $user->id)->count();
            $quotations_pending = Quotation::where('status', 'pending')->where('saleman_id', $user->id)->count();
            // Quotations converted to sale
            $quotations_convert_sale = DB::table('quotations')
                ->join('sales', function($join) use ($user) {
                    $join->on('quotations.invoice_date', '=', 'sales.invoice_date')
                         ->on('quotations.cus_id', '=', 'sales.cus_id')
                         ->on('quotations.saleman_id', '=', 'sales.saleman_id')
                         ->on('quotations.grand_total', '=', 'sales.grand_total')
                         ->where('quotations.saleman_id', $user->id);
                })
                ->where('quotations.status', 'approved')
                ->count(DB::raw('DISTINCT quotations.id'));
        }

        // Order Booker widgets
        if ($user->hasRole(['Super Admin', 'Admin'])) {
            $order_booker_total = DB::table('order_bookers')->count();
            $order_booker_pending = DB::table('order_bookers')->where('status','Pending')->count();
            $order_booker_approved = DB::table('order_bookers')->where('status','Complete')->count();
            // Advance amounts for Order Booker (Admin sees all)
            $order_booker_advance_pending = DB::table('order_bookers')
                ->where('status', 'Pending')
                ->sum('paid') ?? 0;
            $order_booker_advance_received = DB::table('order_bookers')
                ->where('status', 'Complete')
                ->sum('paid') ?? 0;
            $order_booker_advance_approved = DB::table('saleman_ledgers')
                ->join('order_bookers', 'saleman_ledgers.order_booker_id', '=', 'order_bookers.id')
                ->where('order_bookers.status', 'Complete')
                ->where('saleman_ledgers.adjustment', 'Sale')
                ->sum('saleman_ledgers.credit') ?? 0;
            $order_booker_advance_balance = $order_booker_advance_pending + $order_booker_advance_received - $order_booker_advance_approved;
        } elseif ($user->hasRole('Order Booker')) {
            // For Order Booker, use created_by to get their own orders
            $order_booker_total = DB::table('order_bookers')->where('created_by', $user->id)->count();
            $order_booker_pending = DB::table('order_bookers')
                ->where('created_by', $user->id)
                ->where('status','Pending')->count();
            $order_booker_approved = DB::table('order_bookers')
                ->where('created_by', $user->id)
                ->where('status','Complete')->count();
            // Advance amounts for Order Booker
            $order_booker_advance_pending = DB::table('order_bookers')
                ->where('created_by', $user->id)
                ->where('status', 'Pending')
                ->sum('paid') ?? 0;
            $order_booker_advance_received = DB::table('order_bookers')
                ->where('created_by', $user->id)
                ->where('status', 'Complete')
                ->sum('paid') ?? 0;
            // Advance approved = sum of credit from saleman_ledgers where order_booker_id matches and status is approved
            $order_booker_advance_approved = DB::table('saleman_ledgers')
                ->join('order_bookers', 'saleman_ledgers.order_booker_id', '=', 'order_bookers.id')
                ->where('order_bookers.created_by', $user->id)
                ->where('order_bookers.status', 'Complete')
                ->where('saleman_ledgers.adjustment', 'Sale')
                ->sum('saleman_ledgers.credit') ?? 0;
            $order_booker_advance_balance = $order_booker_advance_pending + $order_booker_advance_received - $order_booker_advance_approved;
        } else {
            // For Sales Man, use saleman_id
            $order_booker_total = DB::table('order_bookers')->where('saleman_id',$user->id)->count();
            $order_booker_pending = DB::table('order_bookers')->where('saleman_id',$user->id)->where('status','Pending')->count();
            $order_booker_approved = DB::table('order_bookers')->where('saleman_id',$user->id)->where('status','Complete')->count();
            $order_booker_advance_pending = 0;
            $order_booker_advance_received = 0;
            $order_booker_advance_approved = 0;
            $order_booker_advance_balance = 0;
        }

        // Recent Order Booker orders for Sales Man (for dashboard widget)
        $recentOrderBookerOrders = null;
        $order_booker_in_progress = 0;
        if ($user->hasRole('Sales Man')) {
            $order_booker_in_progress = OrderBooker::where('saleman_id', $user->id)
                ->where('status', 'In Progress')
                ->count();

            $recentOrderBookerOrders = OrderBooker::where('saleman_id', $user->id)
                ->with('customer')
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();
        }

        // products
        $all_products = Product::count();

        // supplier
        $all_suppliers = Supplier::count();

        // BOM count
        $total_bom = Bom::count();


 



        $low_stock_products = DB::table('products as p')
            ->select(
                'p.id',
                'p.pro_name',
                DB::raw('(
                    IFNULL((SELECT SUM(in_qty.stock_in_qty) 
                        FROM stocks in_qty 
                        WHERE in_qty.pro_id = p.id), 0) 
                    - 
                    IFNULL((SELECT SUM(out_qty.stock_out_qty) 
                        FROM stocks out_qty 
                        WHERE out_qty.pro_id = p.id), 0)
                ) as total_qty')
            )
            ->having('total_qty', '<', 15) // threshold set kar sakte ho
            ->take(10)
        ->get();




        return view('dashboard', compact(
            'user', 'customer', 'hand_cash', 'banks',
            'totalSales', 'sales_count', 'sales_paid', 'sales_receivable',
            'sale_return', 'sale_return_count', 'sale_return_pending', 'sale_return_approved',
            'expense',
            'totalPurchases', 'purchases_count', 'purchase_paid_total', 'purchase_payable',
            'profit', 'customer_payable',
            'order', 'orders_pending', 'orders_approved', 'order_return', 'order_return_pending', 'invoice', 'order_return_approved',
            'production_total', 'production_pending', 'production_completed', 'production', 'production_issue',
            'topProducts', 'topSalesman', 'topCustomer',
            'company_investment', 'investor_investment', 'company_withdraw', 'investor_withdraw', 'total_investment', 'total_withdraw', 'investment_balance',
            'quotations', 'quotations_total', 'quotations_pending', 'quotations_convert_sale',
            'all_products', 'all_suppliers', 'low_stock_products', 'total_bom',
            'order_booker_total', 'order_booker_pending', 'order_booker_approved', 'order_booker_in_progress',
            'order_booker_advance_pending', 'order_booker_advance_received', 'order_booker_advance_approved', 'order_booker_advance_balance',
            'recentOrderBookerOrders', 'total_approved_expenses'
        ));
    }


}