<?php

namespace App\Http\Controllers;

use App\Models\Expense;
use App\Models\ExpenseDetail;
use App\Models\OrderBookerLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ExpenseController extends Controller
{
    public function load_expenses(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        $query = DB::table('expenses')
        ->leftJoin('users', 'expenses.created_by', '=', 'users.id') // join with users
        ->select([
            'expenses.*', 
            'users.name as user_name'
        ])
        ->where('expenses.is_deleted', '!=', 1);


        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('expenses.created_by', Auth::id());
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('exp_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('exp_detail', 'like', '%' . $search . '%');
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('is_active', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $purchase = $query->get();

        $all_data = [];
        foreach ($purchase as $data) {

            // Fetch the data from the database
            $details = DB::table('expense_details')
            ->join('expense_categories', 'expense_details.exp_cat_id', '=', 'expense_categories.id')
            ->select('*')
            ->where('expense_details.exp_id', '=', $data->id)
            ->get(); // Get the results as a collection

            $exp_detail = "";

            // Iterate over the results and build the string
            foreach ($details as $detail) {
                $exp_detail .= $detail->exp_detail . " = " . floatval($detail->total) . '<span style="color: red;"> |</span> ';
            }


            $btn = '<td class="text-right">';

            if (Auth::user()->can('view_expense')) {
                $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-success btn-icon btn-sm" data-toggle="tooltip" title="View" ><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }

            if (Auth::user()->can('update_expense')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon btn-sm" data-toggle="tooltip" title="Edit" style="margin-left: 2px;"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }

            if (Auth::user()->can('delete_expense')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon btn-sm" data-toggle="tooltip" title="Delete" style="margin-left: 2px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }

            $btn .= '</td>';


            $all_data[] = [
                'id'            => $data->id,
                'exp_date'      => Carbon::parse($data->exp_date)->format('d-m-Y'),
                'user_name'     => $data->user_name,

                'exp_detail'    => $exp_detail.$data->exp_detail,
                'total'         => number_format($data->total_amt, 0),
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_expense(Request $request)
    {
        try {
            DB::beginTransaction();

            $expense = new Expense();
            $expense->exp_date         = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
            $expense->total_amt        = $request->input('total_amt');
            $expense->exp_detail       = $request->input('other_detail');
            $expense->created_by       = Auth::id();
            $expense->updated_by       = Auth::id();

            $expense->save();

            // Get the last inserted Book Test ID
            $exp_id = $expense->id;

            // Extract data from the request
            $cat_ids    = $request->input('cat_id');
            $sub_cat_id    = $request->input('sub_cat_id');

            $exp_detail = $request->input('exp_detail');
            $exp_total  = $request->input('total');

            // Loop through the data and save each entry
            foreach ($cat_ids as $index => $cat_id) {
                $expense_detil                    = new ExpenseDetail();
                $expense_detil->exp_id            = $exp_id;
                $expense_detil->exp_cat_id        = $cat_id;
                $expense_detil->sub_cat_id        = $sub_cat_id[$index];

                $expense_detil->exp_date          = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
                $expense_detil->exp_detail        = $exp_detail[$index];
                $expense_detil->total             = $exp_total[$index];
                $expense_detil->save();
            }

            // NEW: If user is an Order Booker, create an entry in OrderBookerLedger
            if (Auth::user()->hasRole('Order Booker')) {
                $orderBookerLedger = new OrderBookerLedger();
                $orderBookerLedger->created_by_user = Auth::id();
                $orderBookerLedger->invoice_date = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
                $orderBookerLedger->detail = 'Expense: ' . ($request->input('other_detail') ?: 'General Expense') . ' (#EXP-' . $exp_id . ')';
                $orderBookerLedger->transaction_type = 'expense';
                $orderBookerLedger->order_amount = 0;
                $orderBookerLedger->paid_amount = 0;
                $orderBookerLedger->expense_amount = $request->input('total_amt');
                $orderBookerLedger->status = null; // Not yet submitted to cashier
                $orderBookerLedger->save();

                \Log::info('Order Booker Expense Added', [
                    'user_id' => Auth::id(),
                    'expense_id' => $exp_id,
                    'ledger_id' => $orderBookerLedger->id,
                    'amount' => $request->input('total_amt')
                ]);
            }

            DB::commit();

            return response()->json([
                'status'=>200
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Insert Expense Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'Error saving expense: ' . $e->getMessage()
            ], 500);
        }
    }


    public function edit_expense($id)
    {
        $expense = DB::table('expenses')
        ->select(
            '*'
        )
        ->where('id', $id)
        ->first();

        $expense_details = DB::table('expense_details')
        ->select('*')->where('exp_id', '=', $id)
        ->get();

        $exp_detail = [];

        foreach($expense_details AS $expense_detail){

            $cats = DB::table('expense_categories')->select('*')
            ->where('id', '=', $expense_detail->exp_cat_id)->first();

            $sub_cats = DB::table('expense_sub_categories')->select('*')
            ->where('id', '=', $expense_detail->sub_cat_id)->first();
            
            $exp_detail[] = [
                'cat_id'        => $cats->id,
                'exp_cat_name'  => $cats->exp_cat_name,
                'sub_cat_id'        => $sub_cats->id,   // 👈 ye add karo
                'exp_sub_cat_name'  => $sub_cats->exp_sub_cat_name,
                'exp_detail'    => $expense_detail->exp_detail,
                'total'         => $expense_detail->total,
            ];
        }

        return response()->json([
            'status'=>200,
            'expense'           => $expense,
            'exp_detail'        => $exp_detail,
        ]);
    }

    public function update_expense(Request $request)
    {
        try {
            DB::beginTransaction();

            $expense = Expense::find($request->input('exp_id'));

            if (!$expense) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Expense not found'
                ], 404);
            }

            $expense->exp_date         = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
            $expense->total_amt        = $request->input('total_amt');
            $expense->exp_detail       = $request->input('other_detail');
            $expense->updated_by       = Auth::id();

            $expense->save();

            $delete = ExpenseDetail::where('exp_id', $request->input('exp_id'))->delete();

            // Get the last inserted Book Test ID
            $exp_id = $request->input('exp_id');

            // Extract data from the request
            $cat_ids    = $request->input('cat_id');
            $sub_cat_id    = $request->input('sub_cat_id');
            $exp_detail = $request->input('exp_detail');
            $exp_total  = $request->input('total');

            // Loop through the data and save each entry
            foreach ($cat_ids as $index => $cat_id) {
                $expense_detil                    = new ExpenseDetail();
                $expense_detil->exp_id            = $exp_id;
                $expense_detil->exp_cat_id        = $cat_id;
                $expense_detil->sub_cat_id        = $sub_cat_id[$index];
                $expense_detil->exp_date          = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
                $expense_detil->exp_detail        = $exp_detail[$index];
                $expense_detil->total             = $exp_total[$index];
                $expense_detil->save();
            }

            // NEW: If user is an Order Booker, update or create OrderBookerLedger entry
            if (Auth::user()->hasRole('Order Booker')) {
                // Find existing ledger entry for this expense (match by detail containing expense ID)
                $ledgerEntry = OrderBookerLedger::where('created_by_user', Auth::id())
                    ->where('transaction_type', 'expense')
                    ->where('detail', 'like', '%#EXP-' . $exp_id . '%')
                    ->whereNull('status') // Only update if not yet submitted to cashier
                    ->first();

                if ($ledgerEntry) {
                    // Update existing entry
                    $ledgerEntry->invoice_date = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
                    $ledgerEntry->detail = 'Expense: ' . ($request->input('other_detail') ?: 'General Expense') . ' (#EXP-' . $exp_id . ')';
                    $ledgerEntry->expense_amount = $request->input('total_amt');
                    $ledgerEntry->save();

                    \Log::info('Order Booker Expense Updated in Ledger', [
                        'user_id' => Auth::id(),
                        'expense_id' => $exp_id,
                        'ledger_id' => $ledgerEntry->id,
                        'amount' => $request->input('total_amt')
                    ]);
                } else {
                    // Create new entry if not found (shouldn't happen, but safe)
                    $orderBookerLedger = new OrderBookerLedger();
                    $orderBookerLedger->created_by_user = Auth::id();
                    $orderBookerLedger->invoice_date = Carbon::parse($request->input('exp_date'))->format('Y-m-d');
                    $orderBookerLedger->detail = 'Expense: ' . ($request->input('other_detail') ?: 'General Expense') . ' (#EXP-' . $exp_id . ')';
                    $orderBookerLedger->transaction_type = 'expense';
                    $orderBookerLedger->order_amount = 0;
                    $orderBookerLedger->paid_amount = 0;
                    $orderBookerLedger->expense_amount = $request->input('total_amt');
                    $orderBookerLedger->status = null;
                    $orderBookerLedger->save();

                    \Log::info('Order Booker Expense Created in Ledger (on update)', [
                        'user_id' => Auth::id(),
                        'expense_id' => $exp_id,
                        'ledger_id' => $orderBookerLedger->id,
                        'amount' => $request->input('total_amt')
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'status'=>200
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Update Expense Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'Error updating expense: ' . $e->getMessage()
            ], 500);
        }
    }


    public function view_expense($id)
    {
        $expense = DB::table('expenses')
        ->select(
            '*'
        )
        ->where('id', $id)
        ->first();

        $expense_details = DB::table('expense_details')
        ->select('*')->where('exp_id', '=', $id)
        ->get();

        $exp_detail = [];

        foreach($expense_details AS $expense_detail){

            $cats = DB::table('expense_categories')->select('*')
            ->where('id', '=', $expense_detail->exp_cat_id)->first();
            
            $exp_detail[] = [
                'cat_id'        => $cats->id,
                'exp_cat_name'  => $cats->exp_cat_name,
                'exp_detail'    => $expense_detail->exp_detail,
                'total'         => $expense_detail->total,
            ];
        }

        return response()->json([
            'status'=>200,
            'expense'           => $expense,
            'exp_detail'        => $exp_detail,
        ]);
    }


    public function delete_expense($id)
    {
        try {
            DB::beginTransaction();

            // Expense find karo
            $expense = Expense::find($id);

            if (!$expense) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Expense not found.'
                ]);
            }

            // Check if user is authorized to delete this expense
            if (!Auth::user()->hasRole(['Admin', 'Super Admin']) && $expense->created_by != Auth::id()) {
                return response()->json([
                    'status' => 403,
                    'message' => 'Unauthorized to delete this expense.'
                ], 403);
            }

            // Delete related expense_details
            ExpenseDetail::where('exp_id', $id)->delete();

            // NEW: If user is an Order Booker, delete related OrderBookerLedger entry
            if (Auth::user()->hasRole('Order Booker')) {
                $ledgerEntry = OrderBookerLedger::where('created_by_user', Auth::id())
                    ->where('transaction_type', 'expense')
                    ->where('detail', 'like', '%#EXP-' . $id . '%')
                    ->whereNull('status') // Only delete if not yet submitted to cashier
                    ->first();

                if ($ledgerEntry) {
                    $ledgerEntry->delete();

                    \Log::info('Order Booker Expense Deleted from Ledger', [
                        'user_id' => Auth::id(),
                        'expense_id' => $id,
                        'ledger_id' => $ledgerEntry->id
                    ]);
                }
            }

            // Delete the expense
            $expense->delete();

            DB::commit();

            return response()->json([
                'status' => 200,
                'message' => 'Expense and related details deleted successfully.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Delete Expense Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'Error deleting expense: ' . $e->getMessage()
            ], 500);
        }
    }

    public function get_total_expense()
    {
        // Role-based filter
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $total_expense = Expense::where('is_deleted', '!=', 1)->sum('total_amt');
        } else {
            $total_expense = Expense::where('created_by', Auth::id())
                ->where('is_deleted', '!=', 1)
                ->sum('total_amt');
        }

        return response()->json([
            'total_expense' => number_format($total_expense, 0)
        ]);
    }


}
