<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\Investment;
use App\Models\Invoice;
use App\Models\InvoiceDetail;
use App\Models\PurchaseDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class InvoiceController extends Controller
{


    public function load_invoices(Request $request)
    {
        try {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build query for regular invoices
        $invoicesQuery = DB::table('invoices')
            ->join('customers', 'invoices.cus_id', '=', 'customers.id')
            ->join('users', 'invoices.saleman_id', '=', 'users.id')
            ->select([
                'invoices.id as id',
                'invoices.invoice_date',
                'customers.name as cus_name',
                'users.name as saleman_name',
                'invoices.subtotal',
                'invoices.dis_percent',
                'invoices.dis_amount',
                'invoices.grand_total',
                'invoices.paid',
                'invoices.due',
                DB::raw("'Invoice' as source_type"),
                'invoices.created_by',
                'invoices.saleman_id'
            ]);

        // Build query for completed Order Booker records
        $orderBookersQuery = DB::table('order_bookers')
            ->join('customers', 'order_bookers.cus_id', '=', 'customers.id')
            ->join('users', 'order_bookers.saleman_id', '=', 'users.id')
            ->where('order_bookers.status', '=', 'Complete')
            ->select([
                'order_bookers.id as id',
                'order_bookers.invoice_date',
                'customers.name as cus_name',
                'users.name as saleman_name',
                'order_bookers.subtotal',
                'order_bookers.dis_percent',
                'order_bookers.dis_amount',
                'order_bookers.grand_total',
                'order_bookers.paid',
                'order_bookers.due',
                DB::raw("'Order Booker' as source_type"),
                'order_bookers.created_by',
                'order_bookers.saleman_id'
            ]);

        // Debug: Check how many completed Order Booker records exist
        $completedOrderBookersCount = DB::table('order_bookers')
            ->where('status', 'Complete')
            ->count();

        \Log::info('Completed Order Bookers Check', [
            'total_completed_orders' => $completedOrderBookersCount,
            'user_id' => Auth::id(),
            'user_roles' => Auth::user()->getRoleNames()
        ]);

        // Combine both queries using UNION
        $query = $invoicesQuery->unionAll($orderBookersQuery);

        // Wrap the union query in a subquery for filtering
        $query = DB::table(DB::raw("({$query->toSql()}) as combined_data"))
            ->mergeBindings($query);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            if (Auth::user()->hasRole('Sales Man')) {
                // Salesmen see invoices/orders assigned to them via saleman_id
                $query->where('combined_data.saleman_id', Auth::id());

                // Debug logging
                \Log::info('Invoice Query - Sales Man Filter Applied', [
                    'user_id' => Auth::id(),
                    'user_roles' => Auth::user()->getRoleNames(),
                    'saleman_id_filter' => Auth::id()
                ]);
            } else {
                // Other users see their own records (created by them)
                $query->where('combined_data.created_by', Auth::id());
            }
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('combined_data.invoice_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) {
            $query->where('combined_data.cus_name', 'like', '%' . $search . '%');
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        if (!empty($order_by) && in_array($order_by, ['id', 'invoice_date', 'cus_name', 'saleman_name', 'subtotal', 'grand_total', 'paid', 'due'])) {
            $query->orderBy('combined_data.' . $order_by, $dir);
        } else {
            $query->orderBy('combined_data.invoice_date', 'desc');
        }

        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $sale = $query->get();

        // Debug logging
        \Log::info('Invoice Query Results', [
            'total_count' => $total_count,
            'returned_rows' => $sale->count(),
            'user_id' => Auth::id(),
            'user_roles' => Auth::user()->getRoleNames(),
            'raw_sql' => $query->toSql()
        ]);

        $all_data = [];
        foreach ($sale as $data) {

            // Add prefix based on source type
            $idDisplay = $data->source_type === 'Order Booker'
                ? 'OB-' . $data->id
                : $data->id;

            $btn = '<td class="text-right">';

            // Only show View button for Order Booker records, full CRUD for Invoice records
            if ($data->source_type === 'Order Booker') {
                if (Auth::user()->can('view_order_booker')) {
                    $btn .= '<button id="view_ob_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
                }
                $btn .= '<span class="label label-success" style="margin-left: 5px;">From Order Booker</span>';
            } else {
                if (Auth::user()->can('view_invoice')) {
                    $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
                }
                if (Auth::user()->can('update_invoice')) {
                    $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
                }
                if (Auth::user()->can('delete_invoice')) {
                    $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
                }
            }
            $btn .= '</td>';

            $all_data[] = [
                'id'            => $idDisplay,
                'invoice_date'     => Carbon::parse($data->invoice_date)->format('d-m-Y'),
                'cus_name'      => $data->cus_name,
                'saleman_name'  => $data->saleman_name,
                'subtotal'      => number_format($data->subtotal, 0),
                'dis_percent'   => number_format($data->dis_percent, 0),
                'dis_amount'    => number_format($data->dis_amount, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'paid'          => number_format($data->paid, 0),
                'due'           => number_format($data->due, 0),
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);

        } catch (\Exception $e) {
            \Log::error('Error in load_invoices: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'An error occurred while loading invoices. Please check the logs.'
            ], 500);
        }
    }


    public function insert_invoice(Request $request)
    {

        $sale = new Invoice();
        $sale->invoice_date             = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $sale->cus_id                   = $request->input('cus_id');
        $sale->bank_id                  = $request->input('bank_id');
        $sale->Payment_type             = $request->input('payment_type');
        $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $sale->detail                   = $request->input('detail');
        $sale->subtotal                 = $request->input('subtotal');
        $sale->dis_percent              = $request->input('total_dis_per');
        $sale->dis_amount               = $request->input('total_dis_amt');
        $sale->grand_total              = $request->input('grand_total');
        $sale->paid                     = $request->input('paid');
        $sale->due                      = $request->input('due');
        $sale->created_by               = Auth::id();
        $sale->save();


        // Get the last inserted Book Test ID
        $sale_id = $sale->id;
        // Extract data from the request
        $pro_ids    = $request->input('pro_id');
        $purchase_price   = $request->input('pro_cost');
        $pro_sku   = $request->input('pro_sku');
        $batch_no   = $request->input('batch_no');

        $pro_sale   = $request->input('pro_sale');
        $pro_qty    = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt    = $request->input('dis_amt');
        $total      = $request->input('total');

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $product_stock                  = new InvoiceDetail();
            $product_stock->invoice_id      = $sale_id;
            $product_stock->pro_id          = $pro_id;
            $product_stock->purchase_price  = $purchase_price[$index];
            $product_stock->pro_sku         = $pro_sku[$index];
            $product_stock->batch_no        = $batch_no[$index];

            $product_stock->sale_price      = $pro_sale[$index];
            $product_stock->dis_per         = $dis_percent[$index];
            $product_stock->dis_amount      = $dis_amt[$index];
            $product_stock->pro_qty         = $pro_qty[$index];
            $product_stock->total           = $total[$index];
            $product_stock->save();

        }



        return response()->json([
            'status'=>200,
        ]);

        
    }


    public function edit_invoice($id, Request $request)
    {
        $salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();

        // Retrieve the sale with customer details
        $sale = DB::table('invoices')
            ->leftJoin('customers', 'invoices.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'invoices.saleman_id', '=', 'users.id')
            ->leftJoin('banks', 'invoices.bank_id', '=', 'banks.id')
            ->select(
                'invoices.*',
                'customers.name as customer_name',
                'banks.bank_name',
                'users.name as salesman_name',
                'invoices.id as id'
            )
            ->where('invoices.id', $id)
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Invoice not found'
            ], 404);
        }

        // Retrieve invoice details for the specific invoice
        $invoice_details = DB::table('invoice_details')
            ->join('products', 'invoice_details.pro_id', '=', 'products.id')
            ->select(
                'invoice_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('invoice_details.invoice_id', '=', $id)
            ->get();

        // Calculate stock for each product (filtered by selected salesman)
        foreach ($invoice_details as $detail) {
            $stock = DB::table('saleman_stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->where('salesman_id', '=', $salesman_id) // <-- Filter by selected salesman
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $invoice_details
        ]);
    }
    


    public function view_invoice($id)
    {
        $invoice = DB::table('invoices')
            ->leftJoin('customers', 'invoices.cus_id', '=', 'customers.id')
            ->select(
                '*', 'customers.name', // Select specific fields
                'invoices.id as id'
            )
            ->where('invoices.id', $id)
            ->first();

        if (!$invoice) {
            return response()->json([
                'status' => 404,
                'message' => 'Invoice not found'
            ], 404);
        }

        $all_customers = DB::table('customers')->select('id', 'name')->get();

        $invoice_details = DB::table('invoice_details')
        ->select('*')->where('invoice_id', '=', $id)
        ->get();

        $products = [];

        foreach($invoice_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();

            $products[] = [
                'no'        => $index+1,
                'pro_id'     => $pro->id,
                'pro_name'   => $pro->pro_name,
                'batch_no'   => $pros->batch_no,
                'sale_price' => $pros->sale_price,
                'pro_qty'    => $pros->pro_qty,
                'dis_per'=> $pros->dis_per,
                'dis_amount' => $pros->dis_amount,
                'total'=> $pros->total,
            ];
        }

        return response()->json([
            'status' => 200,
            'sale' => $invoice,
            'products' => $products,
            'all_customers' => $all_customers,  
        ]);
    }


    public function invoice_print()
    {
        // Fetch the last inserted sale
        $sale = DB::table('sales')
            ->leftJoin('customers', 'sales.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'sales.saleman_id', '=', 'users.id')
            ->select(
                'sales.*',
                'customers.name as customer_name',
                'users.name as salesman_name',
                'sales.id as sale_id'
            )
            ->orderBy('sales.id', 'desc') // Get the most recent sale
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found'
            ], 404);
        }

        // Fetch the details for the sale
        $sale_details = DB::table('sale_details')
            ->where('sale_id', $sale->id)
            ->get();

        $products = [];

        foreach ($sale_details as $index => $detail) {
            $product = DB::table('products')
                ->where('id', $detail->pro_id)
                ->first();

            if ($product) {
                $products[] = [
                    'no'        => $index + 1,
                    'pro_id'    => $product->id,
                    'pro_name'  => $product->pro_name,
                    'sale_price'=> $detail->sale_price,
                    'pro_qty'   => $detail->pro_qty,
                    'dis_per'   => $detail->dis_per,
                    'dis_amount'=> $detail->dis_amount,
                    'total'     => $detail->total,
                ];
            }
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $products,
        ]);
    }

    

    public function update_invoice(Request $request)
    {
        // Find the invoice record by ID
        $invoice = Invoice::find($request->input('invoice_id'));

        if (!$invoice) {
            return response()->json(['status' => 404, 'message' => 'Invoice not found.']);
        }

        // Update the main invoice record
        $invoice->invoice_date             = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $invoice->cus_id                   = $request->input('cus_id');
        $invoice->Payment_type             = $request->input('payment_type');
        $invoice->bank_id                  = $request->input('bank_id');
        $invoice->status                  = $request->input('status');

        $invoice->detail                    = $request->input('detail');
        $invoice->subtotal                  = $request->input('subtotal');
        $invoice->dis_percent               = $request->input('total_dis_per');
        $invoice->dis_amount                = $request->input('total_dis_amt');
        $invoice->grand_total               = $request->input('grand_total');
        $invoice->paid                      = $request->input('paid');
        $invoice->due                       = $request->input('due');
        $invoice->saleman_id                = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $invoice->updated_by = Auth::id();
        $invoice->save();

        

        // Delete old invoice details for this invoice
        InvoiceDetail::where('invoice_id', $invoice->id)->delete();
        SalemanStock::where('invoice_id', $invoice->id)->delete();
        CustomerLedger::where('invoice_id', $invoice->id)->delete();
        SalemanLedger::where('invoice_id', $invoice->id)->delete();
        DB::table('investment_profits')->where('invoice_id', $invoice->id)->delete();
        Transaction::where('source_type', 'Invoice')
        ->where('source_id', $invoice->id)
        ->delete();

        if ($request->payment_type === "Cash") {
            // hand_cash column
            DB::table('hand_cashes')->increment('cash_amount', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Cash',
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Invoice',
                'source_id'         => $invoice->id,
                'description'       => 'Invoice Payment Received',
                'created_at'        => now(),
            ]);

        } elseif ($request->payment_type === "Bank") {
            
            DB::table('banks')->where('id', $request->bank_id)
            ->increment('bank_balance', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Bank',
                'bank_id'           => $request->input('bank_id'),
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Invoice',
                'source_id'         => $invoice->id,
                'description'       => 'Invoice Payment Received',
                'created_at'        => now(),
            ]);
        }



        // Reinsert the sale details
        $pro_ids = $request->input('pro_id');
        $purchase_price = $request->input('pro_cost');
        $pro_sku = $request->input('pro_sku');
        $batch_no = $request->input('batch_no');

        $pro_sale = $request->input('pro_sale');
        $pro_qty = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt = $request->input('dis_amt');
        $total = $request->input('total');

        foreach ($pro_ids as $index => $pro_id) {
            $product_stock = new InvoiceDetail();
            $product_stock->invoice_id = $invoice->id;
            $product_stock->pro_id = $pro_id; 
            $product_stock->purchase_price = $purchase_price[$index];
            $product_stock->pro_sku = $pro_sku[$index];
            $product_stock->batch_no = $batch_no[$index];

            $product_stock->sale_price = $pro_sale[$index];
            $product_stock->dis_per = $dis_percent[$index];
            $product_stock->dis_amount = $dis_amt[$index];
            $product_stock->pro_qty = $pro_qty[$index];
            $product_stock->total = $total[$index];
            $product_stock->save();

            // 🟢 2. PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
            ->where('batch_no', $batch_no[$index])
            ->first()?->increment('sale_qty', $pro_qty[$index]);

            // Update stock
            $salemanstock                  = new SalemanStock();
            $salemanstock->pro_id          = $pro_id;
            $salemanstock->invoice_id         = $invoice->id;
            $salemanstock->sale_price      = $pro_sale[$index];
            $salemanstock->purchase_price  = $purchase_price[$index];
            $salemanstock->stock_out_qty   = $pro_qty[$index];
            $salemanstock->invoice_date    = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $salemanstock->salesman_id       = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $salemanstock->adjustment       = 'Sale';
            $salemanstock->save();

            // Sale & Purchase
            $saleAmount = $total[$index]; // qty * sale price
            $purchaseAmount = $purchase_price[$index] * $pro_qty[$index]; // qty * purchase price

            // Actual profit
            $actualProfit = $saleAmount - $purchaseAmount;

            // Investments fetch karo
            $investments = Investment::where('pro_id', $pro_id)
                ->where('remaining_amount', '>', 0)
                ->get();

            foreach ($investments as $investment) {

                // Investor ka share jitne profit se adjust ho sakta hai
                $adjustableProfit = min($actualProfit, $investment->remaining_amount);

                if ($adjustableProfit > 0) {
                    // Investor ka profit = adjustableProfit ka % according to investor's share
                    $profit = ($adjustableProfit * $investment->profit_percentage) / 100;

                    DB::table('investment_profits')->insert([
                        'investment_id' => $investment->id,
                        'investor_id'   => $investment->investor_id,
                        'invoice_id'       => $invoice->id,
                        'pro_id'        => $pro_id,
                        'profit_amount' => $profit,
                        'created_at'    => now(),
                    ]);

                    // ✅ Investment table update karo (remaining_amount minus karo by adjustableProfit)
                    $investment->remaining_amount -= $adjustableProfit;
                    if ($investment->remaining_amount < 0) {
                        $investment->remaining_amount = 0; // negative na ho
                    }
                    $investment->save();

                    // Actual profit bhi kam karna zaroori hai
                    $actualProfit -= $adjustableProfit;
                }
            }



        }

        // Extract customer  ledger data
        $products    = implode(",", $request->input('pro_id'));
        $pro_qty1     = implode(",", $request->input('pro_qty'));
        
        // Insert into SupplierLedger table
        $supplier_ledger = new CustomerLedger();
        $supplier_ledger->cus_id                    = $request->input('cus_id');
        $supplier_ledger->invoice_id                   = $invoice->id;
        $supplier_ledger->invoice_date              = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $supplier_ledger->detail                    = $request->input('detail');
        $supplier_ledger->pro_id                    = $products;
        $supplier_ledger->pro_qty                   = $pro_qty1;
        $supplier_ledger->transaction_type          = $request->input('payment_type');
        $supplier_ledger->total                     = $request->input('grand_total');
        $supplier_ledger->adjustment                 = 'Sale';
        $supplier_ledger->credit                    = $request->input('paid');
        $supplier_ledger->debit                     = $request->input('due');
        $supplier_ledger->created_by                = Auth::id();
        $supplier_ledger->updated_by                = Auth::id();
        $supplier_ledger->save();

        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id                    = $request->input('salesman_id');
        $saleman_ledger->cus_id                   = $request->input('cus_id');
        $saleman_ledger->invoice_id                   = $invoice->id;
        $saleman_ledger->invoice_date              = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $saleman_ledger->detail                    = $request->input('detail');
        $saleman_ledger->pro_id                    = $products;
        $saleman_ledger->pro_qty                   = $pro_qty1;
        $saleman_ledger->transaction_type          = $request->input('payment_type');
        $saleman_ledger->total                     = $request->input('grand_total');
        $saleman_ledger->credit                    = $request->input('paid');
        $saleman_ledger->debit                     = $request->input('due');
        $saleman_ledger->adjustment                     = 'Sale';

        $saleman_ledger->save();

        return response()->json(['status' => 200, 'message' => 'Sale updated successfully.']);
    }



    public function delete_invoice($id)
    {
        $data = Invoice::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Invoice not found.'
            ]);
        }

        // Reverse entry using saved data
        if ($data->Payment_type === "Cash") {
            DB::table('hand_cashes')->decrement('cash_amount', $data->paid);

        } elseif ($data->Payment_type === "Bank" && $data->bank_id) {
            DB::table('banks')->where('id', $data->bank_id)
                ->decrement('bank_balance', $data->paid);
        }

        // Delete invoice details first
        $data->invoiceDetails()->delete();
        $data->salemanStock()->delete();
        $data->customerLedger()->delete();
        $data->salemanLedger()->delete();
        $data->investorLedger()->delete();
        // 3️⃣ Related Transactions delete
        Transaction::where('source_type', 'Invoice')
            ->where('source_id', $data->id)
            ->delete();
        // Delete the invoice
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Invoice and its details deleted successfully.'
        ]);
    }


}
