<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\Investment;
use App\Models\InvestmentProfit;
use App\Models\OrderBooker;
use App\Models\OrderBookerDetail;
use App\Models\OrderBookerLedger;
use App\Models\PurchaseDetail;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\SalesMan;
use App\Models\Transaction;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderBookerController extends Controller
{
    public function load_order_booker(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        $query = DB::table('order_bookers')
        ->join('customers', 'order_bookers.cus_id', '=', 'customers.id')
        ->join('users', 'order_bookers.saleman_id', '=', 'users.id')
        ->select([
            'order_bookers.*',
            'customers.name as cus_name',
            'order_bookers.id as id',
            'users.name as saleman_name',
        ]);

        // Exclude completed orders (they appear in Sales section)
        $query->where('order_bookers.status', '!=', 'Complete');

        // Role-based filter - Show orders created by the Order Booker OR assigned to the Salesman
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            // For Order Bookers: show orders they created
            // For Salesmen: show orders assigned to them
            if (Auth::user()->hasRole('Order Booker')) {
                $query->where('order_bookers.created_by', Auth::id());
            } else {
                $query->where('order_bookers.saleman_id', Auth::id());
            }
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('customers.name', 'like', '%' . $search . '%');
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('order_bookers.is_active', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $sale = $query->get();

        $all_data = [];
        foreach ($sale as $data) {

            // Status badge with color coding
            $statusBadge = '';
            if ($data->status === 'Pending') {
                $statusBadge = '<span class="label label-warning">Pending</span>';
            } elseif ($data->status === 'In Progress') {
                $statusBadge = '<span class="label label-info">In Progress</span>';
            } elseif ($data->status === 'Complete') {
                $statusBadge = '<span class="label label-success">Complete</span>';
            }

            // Status dropdown for Salesman role only
            $statusDropdown = '';
            if (Auth::user()->hasRole('Sales Man') && $data->saleman_id == Auth::id()) {
                $statusDropdown = '<select class="form-control status-dropdown" data-order-id="'.$data->id.'" style="min-width: 120px;">';
                $statusDropdown .= '<option value="Pending" '.($data->status === 'Pending' ? 'selected' : '').'>Pending</option>';
                $statusDropdown .= '<option value="In Progress" '.($data->status === 'In Progress' ? 'selected' : '').'>In Progress</option>';
                $statusDropdown .= '<option value="Complete" '.($data->status === 'Complete' ? 'selected' : '').'>Complete</option>';
                $statusDropdown .= '</select>';
            } else {
                $statusDropdown = $statusBadge;
            }

            $btn = '<td class="text-right">';
            // View button - available to all
            if (Auth::user()->can('view_order_booker')) {
                $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }
            // Edit and Delete buttons - only for Admin/Super Admin, NOT for Order Bookers
            if (Auth::user()->can('update_order_booker') && !Auth::user()->hasRole('Order Booker')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_order_booker') && !Auth::user()->hasRole('Order Booker')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            $all_data[] = [
                'id'            => $data->id,
                'invoice_date'     => Carbon::parse($data->invoice_date)->format('d-m-Y'),
                'cus_name'      => $data->cus_name,
                'saleman_name'  => $data->saleman_name,
                'subtotal'      => number_format($data->subtotal, 0),
                'dis_percent'   => number_format($data->dis_percent, 0),
                'dis_amount'    => number_format($data->dis_amount, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'paid'          => number_format($data->paid, 0),
                'due'           => number_format($data->due, 0),
                'status'        => $statusDropdown,
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_order_booker(Request $request)
    {
        try {
            $data = new OrderBooker();
            $data->invoice_date             = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $data->cus_id                   = $request->input('cus_id');
            $data->bank_id                  = $request->input('bank_id');
            $data->city_id                  = $request->input('city_id');
            $data->area_id                  = $request->input('area_id');
            $data->Payment_type             = $request->input('payment_type');
            $data->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $data->detail                   = $request->input('detail');
            $data->subtotal                 = $request->input('subtotal');
            $data->dis_percent              = $request->input('total_dis_per');
            $data->dis_amount               = $request->input('total_dis_amt');
            $data->grand_total              = $request->input('grand_total');
            $data->paid                     = $request->input('paid');
            $data->due                      = $request->input('due');
            $data->status                   = 'Pending'; // Default status
            $data->created_by               = Auth::id();
            $data->save();


            // Get the last inserted Book Test ID
            $order_booker_id = $data->id;
            // Extract data from the request
            $pro_ids    = $request->input('pro_id');
            $pro_sku    = $request->input('pro_sku');
            $batch_no   = $request->input('batch_no');

            $purchase_price   = $request->input('pro_cost');
            $pro_sale   = $request->input('pro_sale');
            $pro_qty    = $request->input('pro_qty');
            $dis_percent = $request->input('dis_percent');
            $dis_amt    = $request->input('dis_amt');
            $total      = $request->input('total');

            // Loop through the data and save each entry
            foreach ($pro_ids as $index => $pro_id) {
                $detail                     = new OrderBookerDetail();
                $detail->order_booker_id    = $order_booker_id;
                $detail->pro_id             = $pro_id;
                $detail->pro_sku            = $pro_sku[$index];
                $detail->batch_no           = $batch_no[$index];
                $detail->purchase_price     = $purchase_price[$index];
                $detail->sale_price         = $pro_sale[$index];
                $detail->dis_per            = $dis_percent[$index];
                $detail->dis_amount         = $dis_amt[$index];
                $detail->pro_qty            = $pro_qty[$index];
                $detail->total              = $total[$index];
                $detail->save();

                // Update purchase detail sale quantity
                PurchaseDetail::where('pro_id', $pro_id)
                    ->where('batch_no', $batch_no[$index])
                    ->first()?->increment('sale_qty', $pro_qty[$index]);

                // Update salesman stock
                $salemanstock                  = new SalemanStock();
                $salemanstock->pro_id          = $pro_id;
                $salemanstock->order_booker_id = $order_booker_id;
                $salemanstock->sale_price      = $pro_sale[$index];
                $salemanstock->purchase_price  = $purchase_price[$index];
                $salemanstock->stock_out_qty   = $pro_qty[$index];
                $salemanstock->invoice_date    = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
                $salemanstock->salesman_id     = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
                $salemanstock->adjustment      = 'Sale';
                $salemanstock->save();
            }

            // Create order booker ledger entry - Track Credit and Balance
            $order_booker_ledger = new OrderBookerLedger();
            $order_booker_ledger->order_booker_id = $order_booker_id;
            $order_booker_ledger->created_by_user = Auth::id();
            $order_booker_ledger->invoice_date = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $order_booker_ledger->detail = 'Order #OB-' . $order_booker_id;
            $order_booker_ledger->transaction_type = $request->input('payment_type');
            $order_booker_ledger->bank_id = $request->input('bank_id');
            $order_booker_ledger->order_amount = $request->input('grand_total');
            // FIXED: If paid amount > 0, it's credit. Otherwise, the full amount is balance (due)
            $order_booker_ledger->paid_amount = $request->input('paid') > 0 ? $request->input('paid') : 0;
            $order_booker_ledger->expense_amount = 0;
            $order_booker_ledger->save();

            // Note: Customer Ledger and Salesman Ledger will be created when order is marked Complete
            // This prevents showing pending orders in the ledger

            return response()->json([
                'status'=>200,
            ]);
        } catch (\Exception $e) {
            \Log::error('Order Booker Insert Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            return response()->json([
                'status' => 500,
                'message' => 'Error saving order: ' . $e->getMessage()
            ], 500);
        }
    }


    public function edit_order_booker($id, Request $request)
    {
        $salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();

        // Retrieve the sale with customer details
        $order_bookers = DB::table('order_bookers')
            ->leftJoin('customers', 'order_bookers.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'order_bookers.saleman_id', '=', 'users.id')
            ->leftJoin('banks', 'order_bookers.bank_id', '=', 'banks.id')
            ->leftJoin('cities', 'order_bookers.city_id', '=', 'cities.id')
            ->leftJoin('areas', 'order_bookers.area_id', '=', 'areas.id')

            ->select(
                'order_bookers.*',
                'customers.name as customer_name',
                'banks.bank_name',
                'users.name as salesman_name',
                'order_bookers.id as id'
            )
            ->where('order_bookers.id', $id)
            ->first();

        if (!$order_bookers) {
            return response()->json([
                'status' => 404,
                'message' => 'Order not found'
            ], 404);
        }

        // Retrieve order details for the specific order
        $order_details = DB::table('order_booker_details')
            ->join('products', 'order_booker_details.pro_id', '=', 'products.id')
            ->select(
                'order_booker_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('order_booker_details.order_booker_id', '=', $id)
            ->get();

        // Calculate stock for each product (filtered by selected salesman)
        foreach ($order_details as $detail) {
            $stock = DB::table('saleman_stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->where('salesman_id', '=', $salesman_id) // <-- Filter by selected salesman
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'sale' => $order_bookers,
            'products' => $order_details
        ]);
    }
    


    public function view_order_booker($id) 
    {
        $order_booker = DB::table('order_bookers')
            ->leftJoin('customers', 'order_bookers.cus_id', '=', 'customers.id')
            ->select(
                '*', 'customers.name', // Select specific fields
                'order_bookers.id as id'
            )
            ->where('order_bookers.id', $id)
            ->first();

        if (!$order_booker) {
            return response()->json([
                'status' => 404,
                'message' => 'Order not found'
            ], 404);
        }

        $all_customers = DB::table('customers')->select('id', 'name')->get();

        $order_booker_details = DB::table('order_booker_details')
        ->select('*')->where('order_booker_id', '=', $id)
        ->get();

        $products = [];

        foreach($order_booker_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();
            
            $products[] = [
                'no'        => $index+1,
                'pro_id'     => $pro->id,
                'pro_name'   => $pro->pro_name,
                'sale_price' => $pros->sale_price,
                'pro_qty'    => $pros->pro_qty,
                'dis_per'=> $pros->dis_per,
                'dis_amount' => $pros->dis_amount,
                'total'=> $pros->total,
            ];
        }

        return response()->json([
            'status' => 200,
            'sale' => $order_booker,
            'products' => $products,
            'all_customers' => $all_customers,  
        ]);
    }

    

    public function update_order_booker(Request $request)
    {
        // Authorization: Only Admin/Super Admin can update, NOT Order Bookers
        if (Auth::user()->hasRole('Order Booker')) {
            return response()->json([
                'status' => 403,
                'message' => 'Unauthorized. Order Bookers cannot edit orders.'
            ], 403);
        }

        // Find the order booker record by ID
        $order_booker = OrderBooker::find($request->input('order_booker_id'));

        if (!$order_booker) {
            return response()->json(['status' => 404, 'message' => 'Order Booker not found.']);
        }

        // Update the main order booker record
        $order_booker->invoice_date             = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $order_booker->cus_id                   = $request->input('cus_id');
        $order_booker->Payment_type             = $request->input('payment_type');
        $order_booker->bank_id                  = $request->input('bank_id');
        $order_booker->city_id                  = $request->input('city_id');
        $order_booker->area_id                  = $request->input('area_id');

        $order_booker->status                  = $request->input('status');

        $order_booker->detail                    = $request->input('detail');
        $order_booker->subtotal                  = $request->input('subtotal');
        $order_booker->dis_percent               = $request->input('total_dis_per');
        $order_booker->dis_amount                = $request->input('total_dis_amt');
        $order_booker->grand_total               = $request->input('grand_total');
        $order_booker->paid                      = $request->input('paid');
        $order_booker->due                       = $request->input('due');
        $order_booker->saleman_id                = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $order_booker->updated_by = Auth::id();
        $order_booker->save();

       

        // Delete old order booker details for this order booker
        OrderBookerDetail::where('order_booker_id', $order_booker->id)->delete();
        SalemanStock::where('order_booker_id', $order_booker->id)->delete();
        OrderBookerLedger::where('order_booker_id', $order_booker->id)->whereNull('status')->delete();
        DB::table('investment_profits')->where('order_booker_id', $order_booker->id)->delete();

        // Reinsert the sale details
        $pro_ids = $request->input('pro_id');
        $pro_sku = $request->input('pro_sku');
        $batch_no = $request->input('batch_no');

        $purchase_price = $request->input('pro_cost');
        $pro_sale = $request->input('pro_sale');
        $pro_qty = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt = $request->input('dis_amt');
        $total = $request->input('total');

        foreach ($pro_ids as $index => $pro_id) {
            $product_stock = new OrderBookerDetail();
            $product_stock->order_booker_id = $order_booker->id;
            $product_stock->pro_id = $pro_id; 
            $product_stock->pro_sku = $pro_sku[$index];
            $product_stock->batch_no = $batch_no[$index];

            $product_stock->purchase_price = $purchase_price[$index];

            $product_stock->sale_price = $pro_sale[$index];
            $product_stock->dis_per = $dis_percent[$index];
            $product_stock->dis_amount = $dis_amt[$index];
            $product_stock->pro_qty = $pro_qty[$index];
            $product_stock->total = $total[$index];
            $product_stock->save();

            // PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
            ->where('batch_no', $batch_no[$index])
            ->first()?->increment('sale_qty', $pro_qty[$index]);

            // Update stock
            $salemanstock                  = new SalemanStock();
            $salemanstock->pro_id          = $pro_id;
            $salemanstock->order_booker_id         = $order_booker->id;
            $salemanstock->sale_price      = $pro_sale[$index];
            $salemanstock->purchase_price  = $purchase_price[$index];
            $salemanstock->stock_out_qty   = $pro_qty[$index];
            $salemanstock->invoice_date    = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $salemanstock->salesman_id       = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $salemanstock->adjustment       = 'Sale';
            $salemanstock->save();
        }

        // Create order booker ledger entry
        $order_booker_ledger = new OrderBookerLedger();
        $order_booker_ledger->order_booker_id = $order_booker->id;
        $order_booker_ledger->created_by_user = $order_booker->created_by;
        $order_booker_ledger->invoice_date = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $order_booker_ledger->detail = 'Order #OB-' . $order_booker->id;
        $order_booker_ledger->transaction_type = $request->input('payment_type');
        $order_booker_ledger->bank_id = $request->input('bank_id');
        $order_booker_ledger->order_amount = $request->input('grand_total');
        $order_booker_ledger->paid_amount = $request->input('paid') > 0 ? $request->input('paid') : 0;
        $order_booker_ledger->expense_amount = 0;
        $order_booker_ledger->save();

        return response()->json(['status' => 200, 'message' => 'Order Booker updated successfully.']);
    }



    public function delete_order_booker($id)
    {
        // Authorization: Only Admin/Super Admin can delete, NOT Order Bookers
        if (Auth::user()->hasRole('Order Booker')) {
            return response()->json([
                'status' => 403,
                'message' => 'Unauthorized. Order Bookers cannot delete orders.'
            ], 403);
        }

        $data = OrderBooker::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Order Booker not found.'
            ]);
        }

        // Delete order booker details first
        $data->orderBookerDetails()->delete();
        $data->salemanStock()->delete();
        $data->customerLedger()->delete();
        $data->salemanLedger()->delete();
        $data->investorLedger()->delete();
        OrderBookerLedger::where('order_booker_id', $data->id)->whereNull('status')->delete();
        Transaction::where('source_type', 'Order Booker')
            ->where('source_id', $data->id)
            ->delete();
        // Delete the order booker
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Order Booker and its details deleted successfully.'
        ]);
    }


    public function getSalesmenByArea(Request $request)
    {
        $area_id = $request->input('area_id');

        $salesmen = User::role('Sales Man') // spatie ka scope
            ->where('area_id', $area_id)
            ->get(['id', 'name']);

        return response()->json($salesmen);
    }

    public function getCustomersByArea(Request $request)
    {
        $area_id = $request->input('area_id');

        $customers = DB::table('customers')
            ->where('area_id', $area_id)
            ->where('customers.id', '!=', 0)
            ->select('id', 'name', 'company_name', 'shop_name')
            ->get();

        return response()->json($customers);
    }

    public function update_status(Request $request)
    {
        try {
            $order_id = $request->input('order_id');
            $new_status = $request->input('status');

            // Validate inputs
            if (!$order_id || !$new_status) {
                return response()->json([
                    'status' => 400,
                    'message' => 'Order ID and status are required.'
                ], 400);
            }

            // Validate status values
            $allowed_statuses = ['Pending', 'In Progress', 'Complete'];
            if (!in_array($new_status, $allowed_statuses)) {
                return response()->json([
                    'status' => 400,
                    'message' => 'Invalid status value.'
                ], 400);
            }

            // Find the order
            $order = OrderBooker::find($order_id);

            if (!$order) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Order not found.'
                ], 404);
            }

            // Authorization: Only the assigned Salesman can change status
            if (!Auth::user()->hasRole('Sales Man') || $order->saleman_id != Auth::id()) {
                return response()->json([
                    'status' => 403,
                    'message' => 'Unauthorized. Only the assigned salesman can change order status.'
                ], 403);
            }

            // Update status
            $old_status = $order->status;
            $order->status = $new_status;
            $order->updated_by = Auth::id();
            $order->save();

            // Verify the save by refreshing from database
            $order->refresh();

            // Convert to Sale if status is Complete
            $sale_id = null;
            if ($new_status === 'Complete') {
                try {
                    $sale_id = $this->convertOrderBookerToSale($order_id);
                } catch (\Exception $e) {
                    \Log::error('Failed to convert Order Booker to Sale during status update', [
                        'order_id' => $order_id,
                        'error' => $e->getMessage()
                    ]);
                    // Continue without throwing error - status was saved successfully
                }
            }

            // Log the status change for debugging
            \Log::info('Order Booker Status Updated', [
                'order_id' => $order_id,
                'old_status' => $old_status,
                'new_status' => $new_status,
                'status_in_db_after_save' => $order->status,
                'updated_by' => Auth::id(),
                'salesman_id' => $order->saleman_id,
                'customer_id' => $order->cus_id,
                'save_successful' => ($order->status === $new_status),
                'converted_to_sale_id' => $sale_id
            ]);

            return response()->json([
                'status' => 200,
                'message' => 'Status updated successfully.',
                'new_status' => $new_status,
                'order_id' => $order_id,
                'sale_id' => $sale_id
            ]);

        } catch (\Exception $e) {
            \Log::error('Status Update Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error updating status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Convert completed Order Booker to Sale
     * Called automatically when status changes to 'Complete'
     * FIXED: Now properly creates all ledger entries and transactions
     */
    private function convertOrderBookerToSale($order_booker_id)
    {
        return DB::transaction(function () use ($order_booker_id) {
            try {
                // Get Order Booker with details
                $orderBooker = OrderBooker::with('orderBookerDetails')->find($order_booker_id);

                if (!$orderBooker) {
                    \Log::error('Order Booker not found for conversion', ['id' => $order_booker_id]);
                    throw new \Exception("Order Booker #{$order_booker_id} not found");
                }

                // 1. Create Sale record
                $sale = new Sale();
                $sale->invoice_date = $orderBooker->invoice_date;
                $sale->cus_id = $orderBooker->cus_id;
                $sale->saleman_id = $orderBooker->saleman_id;
                $sale->bank_id = $orderBooker->bank_id;
                $sale->Payment_type = $orderBooker->Payment_type;
                $sale->detail = $orderBooker->detail ?? "Converted from Order Booker #OB-{$order_booker_id}";
                $sale->subtotal = $orderBooker->subtotal;
                $sale->dis_percent = $orderBooker->dis_percent;
                $sale->dis_amount = $orderBooker->dis_amount;
                $sale->grand_total = $orderBooker->grand_total;
                $sale->paid = $orderBooker->paid;
                $sale->due = $orderBooker->due;
                $sale->is_active = 1;
                $sale->is_deleted = 0;
                $sale->created_by = $orderBooker->created_by;
                $sale->save();

                \Log::info('Sale record created from Order Booker', [
                    'sale_id' => $sale->id,
                    'order_booker_id' => $order_booker_id,
                    'saleman_id' => $sale->saleman_id,
                    'cus_id' => $sale->cus_id,
                    'grand_total' => $sale->grand_total,
                    'paid' => $sale->paid,
                    'due' => $sale->due
                ]);

                $sale_id = $sale->id;

                // 2. Create Sale Details from Order Booker Details
                foreach ($orderBooker->orderBookerDetails as $detail) {
                    $saleDetail = new SaleDetail();
                    $saleDetail->sale_id = $sale_id;
                    $saleDetail->pro_id = $detail->pro_id;
                    $saleDetail->date = $orderBooker->invoice_date;
                    $saleDetail->sale_price = $detail->sale_price;
                    $saleDetail->pro_qty = $detail->pro_qty;
                    $saleDetail->dis_per = $detail->dis_per ?? 0;
                    $saleDetail->dis_amount = $detail->dis_amount ?? 0;
                    $saleDetail->total = $detail->total;
                    $saleDetail->is_active = 1;
                    $saleDetail->is_deleted = 0;
                    $saleDetail->created_by = $orderBooker->created_by;
                    $saleDetail->updated_by = $orderBooker->created_by;
                    $saleDetail->save();
                }

                // 3. Update existing SalemanStock records to link to Sale
                SalemanStock::where('order_booker_id', $order_booker_id)
                    ->update(['sale_id' => $sale_id]);

                // 4. FIXED: Create Customer Ledger entry for this sale
                // This makes the completed order appear in Customer Ledger
                $products = $orderBooker->orderBookerDetails->pluck('pro_id')->implode(',');
                $pro_qty = $orderBooker->orderBookerDetails->pluck('pro_qty')->implode(',');

                $customer_ledger = new CustomerLedger();
                $customer_ledger->cus_id = $orderBooker->cus_id;
                $customer_ledger->sale_id = $sale_id;
                $customer_ledger->order_booker_id = $order_booker_id;
                $customer_ledger->invoice_date = $orderBooker->invoice_date;
                $customer_ledger->detail = $orderBooker->detail ?? "Sale from Order #OB-{$order_booker_id}";
                $customer_ledger->pro_id = $products;
                $customer_ledger->pro_qty = $pro_qty;
                $customer_ledger->transaction_type = $orderBooker->Payment_type ?? 'Cash';
                $customer_ledger->bank_id = $orderBooker->bank_id;
                $customer_ledger->total = $orderBooker->grand_total;
                $customer_ledger->adjustment = 'Sale';
                // FIXED: Credit is the paid amount, balance (due) goes to debit
                if ($orderBooker->Payment_type === 'Cash') {
                    $customer_ledger->credit = $orderBooker->paid;
                    $customer_ledger->bank = 0;
                } else {
                    $customer_ledger->credit = 0;
                    $customer_ledger->bank = $orderBooker->paid;
                }
                $customer_ledger->debit = 0; // No debit for sales
                $customer_ledger->created_by = $orderBooker->created_by;
                $customer_ledger->save();

                \Log::info('Customer Ledger created for completed Order Booker', [
                    'customer_ledger_id' => $customer_ledger->id,
                    'cus_id' => $customer_ledger->cus_id,
                    'sale_id' => $sale_id,
                    'total' => $customer_ledger->total,
                    'credit' => $customer_ledger->credit,
                    'bank' => $customer_ledger->bank
                ]);

                // 5. Create Salesman Ledger entry
                $saleman_ledger = new SalemanLedger();
                $saleman_ledger->salesman_id = $orderBooker->saleman_id;
                $saleman_ledger->cus_id = $orderBooker->cus_id;
                $saleman_ledger->sale_id = $sale_id;
                $saleman_ledger->order_booker_id = $order_booker_id;
                $saleman_ledger->invoice_date = $orderBooker->invoice_date;
                $saleman_ledger->detail = $orderBooker->detail ?? "Sale from Order #OB-{$order_booker_id}";
                $saleman_ledger->pro_id = $products;
                $saleman_ledger->pro_qty = $pro_qty;
                $saleman_ledger->transaction_type = $orderBooker->Payment_type ?? 'Cash';
                $saleman_ledger->bank_id = $orderBooker->bank_id;
                $saleman_ledger->total = $orderBooker->grand_total;
                $saleman_ledger->adjustment = 'Sale';
                if ($orderBooker->Payment_type === 'Cash') {
                    $saleman_ledger->credit = $orderBooker->paid;
                    $saleman_ledger->bank = 0;
                } else {
                    $saleman_ledger->credit = 0;
                    $saleman_ledger->bank = $orderBooker->paid;
                }
                $saleman_ledger->debit = $orderBooker->due; // Due amount goes to debit
                $saleman_ledger->save();

                // 6. Handle payment transactions - only if payment was collected
                if ($orderBooker->paid > 0) {
                    if ($orderBooker->Payment_type === "Cash") {
                        DB::table('hand_cashes')->increment('cash_amount', $orderBooker->paid);
                        DB::table('transactions')->insert([
                            'payment_type' => 'Cash',
                            'amount' => $orderBooker->paid,
                            'transaction_type' => 'deposit',
                            'source_type' => 'Sale',
                            'source_id' => $sale_id,
                            'description' => 'Payment from Order Booker #OB-' . $order_booker_id,
                            'created_at' => now(),
                        ]);
                    } elseif ($orderBooker->Payment_type === "Bank" && $orderBooker->bank_id) {
                        DB::table('banks')->where('id', $orderBooker->bank_id)
                            ->increment('bank_balance', $orderBooker->paid);
                        DB::table('transactions')->insert([
                            'payment_type' => 'Bank',
                            'bank_id' => $orderBooker->bank_id,
                            'amount' => $orderBooker->paid,
                            'transaction_type' => 'deposit',
                            'source_type' => 'Sale',
                            'source_id' => $sale_id,
                            'description' => 'Payment from Order Booker #OB-' . $order_booker_id,
                            'created_at' => now(),
                        ]);
                    }
                }

                // 7. Update existing InvestmentProfit records to link to Sale (if any)
                InvestmentProfit::where('order_booker_id', $order_booker_id)
                    ->update(['sale_id' => $sale_id]);

                // 8. Calculate and store investment profits
                foreach ($orderBooker->orderBookerDetails as $detail) {
                    $saleAmount = $detail->total;
                    $purchaseAmount = $detail->purchase_price * $detail->pro_qty;
                    $actualProfit = $saleAmount - $purchaseAmount;

                    $investments = Investment::where('pro_id', $detail->pro_id)
                        ->where('remaining_amount', '>', 0)
                        ->get();

                    foreach ($investments as $investment) {
                        $adjustableProfit = min($actualProfit, $investment->remaining_amount);

                        if ($adjustableProfit > 0) {
                            $profit = ($adjustableProfit * $investment->profit_percentage) / 100;

                            DB::table('investment_profits')->insert([
                                'investment_id' => $investment->id,
                                'investor_id' => $investment->investor_id,
                                'sale_id' => $sale_id,
                                'order_booker_id' => $order_booker_id,
                                'pro_id' => $detail->pro_id,
                                'profit_amount' => $profit,
                                'created_at' => now(),
                            ]);

                            $investment->remaining_amount -= $adjustableProfit;
                            if ($investment->remaining_amount < 0) {
                                $investment->remaining_amount = 0;
                            }
                            $investment->save();

                            $actualProfit -= $adjustableProfit;
                        }
                    }
                }

                \Log::info('Order Booker successfully converted to Sale', [
                    'order_booker_id' => $order_booker_id,
                    'sale_id' => $sale_id,
                    'grand_total' => $sale->grand_total,
                    'paid' => $sale->paid,
                    'due' => $sale->due
                ]);

                return $sale_id;

            } catch (\Exception $e) {
                \Log::error('Failed to convert Order Booker to Sale', [
                    'order_booker_id' => $order_booker_id,
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                throw $e; // Re-throw to rollback transaction
            }
        });
    }

    // Order Booker Cashier Methods
    public function order_booker_cashier()
    {
        return view('admin.order_booker_cashier');
    }

    public function load_order_booker_cashier(Request $request)
    {
        try {
            // Parse dates - handle both formats (DD-MM-YYYY from frontend and Y-m-d for DB)
            $start_date = !empty($request->input('startdate'))
                ? Carbon::createFromFormat('d-m-Y', $request->input('startdate'))->format('Y-m-d')
                : Carbon::today()->format('Y-m-d');
            $end_date = !empty($request->input('enddate'))
                ? Carbon::createFromFormat('d-m-Y', $request->input('enddate'))->format('Y-m-d')
                : Carbon::today()->format('Y-m-d');

            $limit = $request->input('length');
            $offset = $request->input('start');

            $user_id = Auth::id();

            // FIXED: Role-based filtering
            // Order Booker: See only their own records
            // Cashier: See all pending/received submissions (for receiving)
            // Admin: See all records

            if (Auth::user()->hasRole('Order Booker')) {
                // Order Booker sees only their own records
                $ordersQuery = DB::table('order_bookers')
                    ->where('created_by', $user_id)
                    ->whereBetween('invoice_date', [$start_date, $end_date]);

                $ledgerQuery = OrderBookerLedger::where('created_by_user', $user_id)
                    ->whereNull('deleted_at')
                    ->whereNull('order_booker_id')
                    ->whereBetween('invoice_date', [$start_date, $end_date]);

            } elseif (Auth::user()->hasRole('Cashier')) {
                // Cashier sees only submission summaries (not individual expenses)
                $ordersQuery = DB::table('order_bookers')->whereRaw('1=0'); // No orders for cashier

                $ledgerQuery = OrderBookerLedger::whereNull('deleted_at')
                    ->whereNull('order_booker_id')
                    ->where('transaction_type', 'submission') // Only show submission summaries
                    ->whereIn('status', ['pending', 'received'])
                    ->whereBetween('invoice_date', [$start_date, $end_date]);

            } else {
                // Admin sees ALL records
                $ordersQuery = DB::table('order_bookers')
                    ->whereBetween('invoice_date', [$start_date, $end_date]);

                $ledgerQuery = OrderBookerLedger::whereNull('deleted_at')
                    ->whereNull('order_booker_id')
                    ->whereBetween('invoice_date', [$start_date, $end_date]);
            }

            $orders = $ordersQuery->get();
            $ledgers = $ledgerQuery->get();

            // Combine both datasets
            $currentCredit = 0;
            $currentBalance = 0;
            $totalOrderAmount = 0;
            $counter = 1;

            $combinedRecords = [];

            // Add orders - FIXED: Show credit for paid orders, balance for unpaid
            foreach ($orders as $order) {
                $combinedRecords[] = [
                    'date' => $order->invoice_date,
                    'detail' => 'Order #OB-' . $order->id . ' (' . $order->status . ')',
                    'order_amount' => $order->grand_total,
                    'paid_amount' => $order->paid, // Credit if paid
                    'balance_amount' => $order->due, // Balance if unpaid
                    'expense' => 0,
                    'status' => $order->status,
                    'type' => 'order'
                ];
            }

            // Add ledger entries (expenses and submissions)
            foreach ($ledgers as $ledger) {
                $combinedRecords[] = [
                    'date' => $ledger->invoice_date,
                    'detail' => $ledger->detail,
                    'order_amount' => $ledger->order_amount ?? 0,
                    'paid_amount' => $ledger->paid_amount ?? 0,
                    'balance_amount' => 0,
                    'expense' => $ledger->expense_amount ?? 0,
                    'status' => $ledger->status,
                    'type' => 'ledger',
                    'ledger_id' => $ledger->id
                ];
            }

            // Sort by date
            usort($combinedRecords, function($a, $b) {
                return strtotime($a['date']) - strtotime($b['date']);
            });

            // Calculate running totals - FIXED: Proper credit and balance calculation
            $processedData = [];
            foreach ($combinedRecords as $record) {
                // Calculate totals
                $totalOrderAmount += $record['order_amount'];
                
                // FIXED: Credit = paid amounts, Balance = unpaid amounts
                if ($record['paid_amount'] > 0) {
                    $currentCredit += $record['paid_amount'];
                }

                // Calculate balance (unpaid amounts)
                if ($record['balance_amount'] > 0) {
                    $currentBalance += $record['balance_amount'];
                }

                // Calculate expenses (deducted from credit)
                if ($record['expense'] > 0) {
                    $currentCredit -= $record['expense'];
                }

                // Status label
                $statusLabel = '';
                $actionButtons = '';

                if ($record['type'] === 'order') {
                    if ($record['status'] === 'Pending') {
                        $statusLabel = '<span class="label label-warning">Pending</span>';
                    } elseif ($record['status'] === 'In Progress') {
                        $statusLabel = '<span class="label label-info">In Progress</span>';
                    } elseif ($record['status'] === 'Complete') {
                        $statusLabel = '<span class="label label-success">Completed</span>';
                    }
                } else {
                    // For ledger entries (submissions)
                    $ledgerId = $record['ledger_id'] ?? null;

                    if ($record['status'] === 'pending') {
                        $statusLabel = '<span class="label label-warning">Sent to Cashier</span>';
                        // Cashier can receive pending submissions
                        if (Auth::user()->hasRole('Cashier') && $ledgerId) {
                            $actionButtons = '<button class="btn btn-sm btn-success receive-btn" data-id="' . $ledgerId . '"><i class="fa fa-check"></i> Receive</button>';
                        }
                    } elseif ($record['status'] === 'received') {
                        $statusLabel = '<span class="label label-info">Received by Cashier</span>';
                        // Admin can approve received submissions
                        if (Auth::user()->hasRole(['Admin', 'Super Admin']) && $ledgerId) {
                            $actionButtons = '<button class="btn btn-sm btn-primary approve-btn" data-id="' . $ledgerId . '"><i class="fa fa-thumbs-up"></i> Approve</button>';
                        }
                    } elseif ($record['status'] === 'approved') {
                        $statusLabel = '<span class="label label-success">Approved by Admin</span>';
                    } elseif ($record['expense'] > 0 && !$record['status']) {
                        $statusLabel = '<span class="label label-danger">Expense</span>';
                    }
                }

                $processedData[] = [
                    'id' => $counter++,
                    'date' => Carbon::parse($record['date'])->format('d-m-Y'),
                    'detail' => $record['detail'],
                    'order_amount' => number_format($record['order_amount'], 2),
                    'paid_amount' => number_format($record['paid_amount'], 2),
                    'expense' => number_format($record['expense'], 2),
                    'credit' => number_format($currentCredit, 2),
                    'balance' => number_format($currentBalance, 2),
                    'status' => $statusLabel,
                    'action' => $actionButtons
                ];
            }

            // Apply pagination
            $totalRecords = count($processedData);
            if ($limit != -1) {
                $processedData = array_slice($processedData, $offset, $limit);
            }

            return response()->json([
                'draw' => intval($request->input('draw')),
                'recordsTotal' => $totalRecords,
                'recordsFiltered' => $totalRecords,
                'data' => $processedData,
                'summary' => [
                    'total_order_amount' => number_format($totalOrderAmount, 2),
                    'total_credit' => number_format($currentCredit, 2),
                    'total_balance' => number_format($currentBalance, 2)
                ]
            ]);

        } catch (\Exception $e) {
            \Log::error('Load Order Booker Cashier Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Error loading data: ' . $e->getMessage()
            ], 500);
        }
    }

    public function getPendingOrderBookerBalances(Request $request)
    {
        try {
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $user_id = Auth::id();

            // Convert dates to proper format or use today's date
            if (!empty($start_date) && !empty($end_date)) {
                $start_date = Carbon::createFromFormat('d-m-Y', $start_date)->format('Y-m-d');
                $end_date = Carbon::createFromFormat('d-m-Y', $end_date)->format('Y-m-d');
            } else {
                $start_date = Carbon::today()->format('Y-m-d');
                $end_date = Carbon::today()->format('Y-m-d');
            }

            // FIXED: Get ALL orders for the date range (not just completed)
            $orders = DB::table('order_bookers')
                ->where('created_by', $user_id)
                ->whereBetween('invoice_date', [$start_date, $end_date])
                ->get();

            // Get expense entries for the date range
            $expenses = OrderBookerLedger::where('created_by_user', $user_id)
                ->whereNull('deleted_at')
                ->whereNull('status') // Only entries that haven't been sent to cashier
                ->where('expense_amount', '>', 0)
                ->whereBetween('invoice_date', [$start_date, $end_date])
                ->get();

            $totalCredit = 0;
            $totalBalance = 0;
            $totalExpense = 0;
            $totalOrderAmount = 0;

            // FIXED: Calculate credit from paid orders, balance from unpaid
            foreach ($orders as $order) {
                $totalOrderAmount += $order->grand_total;
                if ($order->paid > 0) {
                    $totalCredit += $order->paid; // Credit = paid amount
                }
                if ($order->due > 0) {
                    $totalBalance += $order->due; // Balance = unpaid amount
                }
            }

            // Subtract expenses from credit
            foreach ($expenses as $expense) {
                $totalExpense += $expense->expense_amount;
            }

            $netCredit = $totalCredit - $totalExpense;

            return response()->json([
                'status' => 200,
                'total_order_amount' => number_format($totalOrderAmount, 2),
                'total_credit' => number_format($totalCredit, 2),
                'total_expense' => number_format($totalExpense, 2),
                'net_amount' => number_format($netCredit, 2),
                'total_balance' => number_format($totalBalance, 2)
            ]);

        } catch (\Exception $e) {
            \Log::error('Get Pending Order Booker Balances Error: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());
            return response()->json([
                'status' => 500,
                'message' => 'Error calculating balances: ' . $e->getMessage()
            ], 500);
        }
    }

    public function check_can_send_to_cashier()
    {
        try {
            $user_id = Auth::id();
            $today = Carbon::today()->format('Y-m-d');

            // Check if user has already submitted today
            $todaySubmission = OrderBookerLedger::where('created_by_user', $user_id)
                ->where('transaction_type', 'submission')
                ->whereDate('created_at', $today)
                ->first();

            return response()->json([
                'status' => 200,
                'can_send' => $todaySubmission ? false : true,
                'message' => $todaySubmission ? 'You have already sent to cashier today' : 'You can send to cashier'
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Can Send to Cashier Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'can_send' => true,
                'message' => 'Error checking submission status'
            ], 500);
        }
    }

    public function send_order_booker_to_cashier(Request $request)
    {
        try {
            $start_date = $request->input('start_date');
            $end_date = $request->input('end_date');
            $user_id = Auth::id();

            // Check if user has already submitted today
            $today = Carbon::today()->format('Y-m-d');
            $todaySubmission = OrderBookerLedger::where('created_by_user', $user_id)
                ->where('transaction_type', 'submission')
                ->whereDate('created_at', $today)
                ->first();

            if ($todaySubmission) {
                return response()->json([
                    'status' => 400,
                    'message' => 'You have already sent to cashier today. Please try again tomorrow.'
                ], 400);
            }

            // Convert dates to proper format
            if (!empty($start_date) && !empty($end_date)) {
                $start_date_formatted = Carbon::createFromFormat('d-m-Y', $start_date)->format('Y-m-d');
                $end_date_formatted = Carbon::createFromFormat('d-m-Y', $end_date)->format('Y-m-d');
            } else {
                return response()->json([
                    'status' => 400,
                    'message' => 'Date range is required'
                ], 400);
            }

            // Get ALL orders for the date range
            $orders = DB::table('order_bookers')
                ->where('created_by', $user_id)
                ->whereBetween('invoice_date', [$start_date_formatted, $end_date_formatted])
                ->get();

            // Get expense entries
            $expenses = OrderBookerLedger::where('created_by_user', $user_id)
                ->whereNull('deleted_at')
                ->whereNull('status')
                ->where('expense_amount', '>', 0)
                ->whereBetween('invoice_date', [$start_date_formatted, $end_date_formatted])
                ->get();

            $totalCredit = 0;
            $totalBalance = 0;
            $totalExpense = 0;
            $totalOrderAmount = 0;

            foreach ($orders as $order) {
                $totalOrderAmount += $order->grand_total;
                $totalCredit += $order->paid;
                $totalBalance += $order->due;
            }

            foreach ($expenses as $expense) {
                $totalExpense += $expense->expense_amount;
            }

            $netAmount = $totalCredit - $totalExpense;

            // Mark existing ledger entries as part of this submission
            OrderBookerLedger::where('created_by_user', $user_id)
                ->whereNull('deleted_at')
                ->whereNull('status')
                ->whereBetween('invoice_date', [$start_date_formatted, $end_date_formatted])
                ->update(['status' => 'pending']);

            // Create "Send to cashier" submission entry
            $submission = new OrderBookerLedger();
            $submission->created_by_user = $user_id;
            $submission->invoice_date = $end_date_formatted; // Use end date of the range
            $submission->detail = 'Submission to Cashier (' . $start_date . ' to ' . $end_date . ')';
            $submission->transaction_type = 'submission';
            $submission->order_amount = $totalOrderAmount;
            $submission->paid_amount = $totalCredit; // Total credit collected
            $submission->expense_amount = $totalExpense; // Total expenses
            $submission->status = 'pending'; // Pending cashier approval
            $submission->save();

            \Log::info('Order Booker sent to cashier', [
                'user_id' => $user_id,
                'submission_id' => $submission->id,
                'total_credit' => $totalCredit,
                'total_expense' => $totalExpense,
                'net_amount' => $netAmount,
                'total_balance' => $totalBalance
            ]);

            return response()->json([
                'status' => 200,
                'message' => 'Successfully sent to cashier for approval.',
                'submission_id' => $submission->id,
                'total_credit' => number_format($totalCredit, 2),
                'total_expense' => number_format($totalExpense, 2),
                'net_amount' => number_format($netAmount, 2)
            ]);

        } catch (\Exception $e) {
            \Log::error('Send to Cashier Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error sending to cashier: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Add expense entry for Order Booker
     */
    public function add_order_booker_expense(Request $request)
    {
        try {
            $user_id = Auth::id();

            $expense = new OrderBookerLedger();
            $expense->created_by_user = $user_id;
            $expense->invoice_date = Carbon::parse($request->input('expense_date'))->format('Y-m-d');
            $expense->detail = $request->input('expense_detail');
            $expense->transaction_type = 'expense';
            $expense->order_amount = 0;
            $expense->paid_amount = 0;
            $expense->expense_amount = $request->input('expense_amount');
            $expense->status = null; // Not yet submitted
            $expense->save();

            return response()->json([
                'status' => 200,
                'message' => 'Expense added successfully.'
            ]);

        } catch (\Exception $e) {
            \Log::error('Add Order Booker Expense Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error adding expense: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Cashier receives submission from Order Booker
     */
    public function receive_order_booker_submission(Request $request)
    {
        try {
            $submission_id = $request->input('submission_id');
            
            $submission = OrderBookerLedger::find($submission_id);
            
            if (!$submission) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Submission not found'
                ], 404);
            }

            $submission->status = 'received';
            $submission->received_at = now();
            $submission->received_by = Auth::id();
            $submission->save();

            return response()->json([
                'status' => 200,
                'message' => 'Submission received successfully. Pending admin approval.'
            ]);

        } catch (\Exception $e) {
            \Log::error('Receive Order Booker Submission Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error receiving submission: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Admin approves submission from Cashier
     */
    public function approve_order_booker_submission(Request $request)
    {
        try {
            $submission_id = $request->input('submission_id');
            
            $submission = OrderBookerLedger::find($submission_id);
            
            if (!$submission) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Submission not found'
                ], 404);
            }

            if ($submission->status !== 'received') {
                return response()->json([
                    'status' => 400,
                    'message' => 'Submission must be received by cashier first'
                ], 400);
            }

            $submission->status = 'approved';
            $submission->approved_at = now();
            $submission->approved_by = Auth::id();
            $submission->save();

            // Create transaction record for approved amount
            $netAmount = $submission->paid_amount - $submission->expense_amount;
            if ($netAmount > 0) {
                DB::table('transactions')->insert([
                    'payment_type' => 'Cash',
                    'amount' => $netAmount,
                    'transaction_type' => 'deposit',
                    'source_type' => 'Order Booker Submission',
                    'source_id' => $submission->id,
                    'description' => 'Approved Order Booker submission - ' . $submission->detail,
                    'created_at' => now(),
                ]);

                // Add to hand cash
                DB::table('hand_cashes')->increment('cash_amount', $netAmount);
            }

            return response()->json([
                'status' => 200,
                'message' => 'Submission approved successfully.',
                'net_amount' => number_format($netAmount, 2)
            ]);

        } catch (\Exception $e) {
            \Log::error('Approve Order Booker Submission Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error approving submission: ' . $e->getMessage()
            ], 500);
        }
    }


}