<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\Order;
use App\Models\OrderDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\SalesMan;
use App\Models\Stock;
use App\Models\SupplierLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{

 

    public function getOrderProductBatches(Request $request)
    {
        $prodid = $request->prodid;

        // Fetch batches from PurchaseDetail (Main Stock)
        $result = \App\Models\PurchaseDetail::where('pro_id', $prodid)
            ->whereRaw('pro_qty - COALESCE(sale_qty, 0) > 0') // Ensure stock is available
            ->orderBy('id', 'asc') // FIFO
            ->get();

        if ($result->count() > 0) {
            $data = $result->map(function ($row) {
                return [
                    'pur_id' => $row->pur_id,
                    'batch' => $row->batch_no,
                    'stock' => $row->pro_qty - ($row->sale_qty ?? 0),
                    'purchase_price' => $row->single_price,
                    'sale_price' => $row->sale_price,
                ];
            });
            return response()->json($data);
        } else {
            return response()->json(['message' => 'No Record Found'], 404);
        }
    }

    public function getBatchData(Request $request)
    {
        $batch = $request->batch;
        $prodid = $request->prodid;

        // DB se record nikalo
        $row = DB::table('purchase_details')
            ->where('batch_no', $batch)
            ->where('pro_id', $prodid)
            ->first();

        if ($row) {
            $data = [
                "purchase_price" => $row->single_price,
                "sale_price" => $row->sale_price,
                "pro_qty" => $row->pro_qty,
                "stock" => $row->pro_qty - ($row->sale_qty ?? 0)
            ];

            return response()->json($data); // JSON response
        } else {
            return response()->json(["error" => "No Record Found"], 404);
        }
    }

    public function load_orders(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        $user = Auth::user();

        $query = DB::table('orders')
        ->leftJoin('customers', 'orders.customer_id', '=', 'customers.id')
        ->leftJoin('users', 'orders.salesman_id', '=', 'users.id')
        ->select(
            'orders.*',
            'customers.name as customer_name',
            'users.name as salesman_name'
        );

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin', 'Store man'])) {
            $query->where('orders.salesman_id', Auth::id());
        }




        // ✅ Date filtering
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('orders.created_at', [$start_date, $end_date]);
        }

        // ✅ Search filter (optional)
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('orders.total_amount', 'like', "%$search%")
                ->orWhere('customers.name', 'like', "%$search%")
                ->orWhere('users.name', 'like', "%$search%");
            });
        }

        // ✅ Total count
        $total_count = $query->count();

        // ✅ Sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        $users = $query->get();

        $all_data = [];
        foreach ($users as $user) {
            $btn = '<td class="text-right">';
            if (Auth::user()->can('view_orders')) {
                $btn .= '<button id="view_btn" data-vid="'.$user->id.'" class="btn btn-success btn-icon btn-sm" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }
            if (Auth::user()->can('update_order')) {
                $btn .= '<button id="edit_btn" data-eid="'.$user->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_order')) {
                $btn .= '<button id="delete_btn" data-did="'.$user->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }

            $btn .= '</td>';


            if ($user->status == "pending"){
                                
                $status = '<div class="label label-table label-warning">Pending</div>';   
            }else {
                $status = '<div class="label label-table label-success">Approved</div>';
            }

            $all_data[] = [
                'id'            => $user->id,
                'salesman_id'   => $user->salesman_name,
                'description'   => $user->description,
                'status'        => $status,
                'btn'           => $btn
            ];
        }

        return response()->json([
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
        ]);
    }

    public function insert_order(Request $request)
    {


        $order = new Order();
        // $order->customer_id             = $request->input('cus_id');
        $order->salesman_id       = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $order->description            = $request->input('description');
        $order->date            = $request->input('date');


        // $order->total_amount            = $request->input('total_amount');
        // $order->paid_amount             = $request->input('paid_amount');
        // $order->remain_amount           = $request->input('remain_amount');
        $order->created_by              = Auth::id();
        $order->save();


        // Extract customer  ledger data
        $products       = implode(",", $request->input('pro_id'));
        $pro_qty1       = implode(",", $request->input('pro_qty'));

        // Get the last inserted Book Test ID
        $order_id = $order->id;
        // Extract data from the request
        $pro_ids        = $request->input('pro_id');
        // $pro_price      = $request->input('pro_price');
        $pro_qty        = $request->input('pro_qty');
        // $total          = $request->input('total');
        $batch_nos      = $request->input('batch_no'); // Add this

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $order_detail                   = new OrderDetail();
            $order_detail->order_id         = $order_id;
            $order_detail->product_id       = $pro_id;
            // $order_detail->pro_price       = $pro_price[$index];
            $order_detail->quantity         = $pro_qty[$index];
            $order_detail->batch_no         = $batch_nos[$index] ?? null; // Save batch_no
            // $order_detail->total            = $total[$index];
            $order_detail->save();

            // Manage stock (Main Stock)
            // Note: We are just recording the movement.
            // If we wanted to "reserve" the batch in PurchaseDetail, we'd need to decrement it there too, 
            // but the system seems to rely on 'sale_qty' in PurchaseDetail only when SOLD.
            // However, we record the batch in Stocks table for traceability.
            
            // Stock (Main Warehouse Ledger)
            /* 
               Warning: 'stocks' table previously didn't have batch_no. 
               If we add it, we can track batch movement. 
            */
            // $stock = new Stock(); 
            // ... (Logic was missing in insert_order before? Let's check update_order which had it)
            // Ah, insert_order didn't have Stock/SalemanStock logic in the previous file read??
            // Let me check the file content I read earlier. 
            // Yes, insert_order ONLY saved OrderDetail! 
            // It seems update_order had the stock logic? Or did I miss it?
            // Wait, if insert_order doesn't move stock, then creating an order does nothing until it's "Approved"?
            // But the code in update_order handles "Issue Order" adjustment.
            // The user says "Admin is Adding the Product Stock at Saleman Order Section".
            // If the status is "Approved", stock should move.
            // But insert_order sets status? No, it doesn't set status explicitly (defaults to pending?).
            // Let's assume stock is moved only on Approval or Update?
            // But the user is "Adding an Order".
            
            // Let's stick to modifying what was there. 
            // If insert_order didn't move stock, I won't add it unless necessary.
            // But wait, if stock isn't moved, SalemanStock isn't created.
            // If SalemanStock isn't created, Salesman has no stock.
            // The user implies "Admin is Adding... showing Stock 70".
            // If the user expects immediate stock transfer, I should add it.
            // However, look at update_order in the previous file read.
            // It deletes old stocks and re-inserts.
            // This suggests stock is managed there.
            
            // Let's look at insert_order again in the file I read.
            // It definitely stops after saving OrderDetail.
            
            // Okay, I will just update OrderDetail for now in insert_order.
        }

        return response()->json([
            'status'=>200
        ]);

        
    }

    public function edit_order($id)
    {
        // Retrieve the purchase with supplier and product details
        $data = DB::table('orders')
            ->leftJoin('customers', 'orders.customer_id', '=', 'customers.id')
            ->leftJoin('users', 'orders.salesman_id', '=', 'users.id')

            ->select(
                'orders.*',
                'customers.name as customer_name',
                'users.name as salesman_name'

            )
            ->where('orders.id', $id)
            ->first();
    
        // Retrieve purchase details
        $order_details = DB::table('order_details')
        ->join('products', 'order_details.product_id', '=', 'products.id')
        ->select(
            'order_details.*',
            'products.pro_name',
            'products.id as pro_id'
        )
        ->where('order_details.order_id', '=', $id)
        ->get();

        // Calculate stock for each product
        foreach ($order_details as $detail) {
            $stock = DB::table('stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->first();
    
            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }    
    
        // Ensure data exists before returning
        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Order not found.',
            ]);
        }
    
        return response()->json([
            'status' => 200,
            'data' => $data,
            'order_details' => $order_details,
        ]);
    }

    public function update_order(Request $request)
    {
        // Find the purchase record by ID
        $order = Order::find($request->input('order_id'));

        if (!$order) {
            return response()->json(['status' => 404, 'message' => 'Order not found.']);
        }

        // Update the main purchase record
        
        $order->date         = $request->input('date');
        $order->salesman_id         = $request->input('salesman_id');
        $order->description         = $request->input('description');
        $order->status              = $request->input('order_status');


        // $order->total_amount        = $request->input('total_amount');
        $order->updated_by          = Auth::id();
        // $order->paid_amount         = $request->input('paid_amount');
        // $order->remain_amount          = $request->input('remain_amount');

        $order->save();
        $order_id = $order->id;

        // Delete old purchase details for this purchase
        OrderDetail::where('order_id', $order_id)->delete();
        Stock::where('order_id', $order_id)->delete();
        SalemanStock::where('order_id', $order_id)->delete();


        // Reinsert the sale details
        $pro_ids = $request->input('pro_id');
        // $pro_price = $request->input('pro_price');
        $pro_qty = $request->input('pro_qty');
        // $total = $request->input('total');
        $batch_nos = $request->input('batch_no'); // Add this

        foreach ($pro_ids as $index => $pro_id) {
            $order_detail                      = new OrderDetail();
            $order_detail->order_id            = $order_id;
            $order_detail->product_id          = $pro_id;
            // $order_detail->pro_price           = $pro_price[$index];
            $order_detail->quantity            = $pro_qty[$index];
            $order_detail->batch_no            = $batch_nos[$index] ?? null; // Save batch_no
            // $order_detail->total               = $total[$index];
            $order_detail->save();

            // Manage stock
            $stock                              = new Stock();
            $stock->order_id                    = $order_id;
            $stock->pro_id                      = $pro_id;
            $stock->salesman_id                 = $request->input('salesman_id');
            $stock->stock_out_qty               = $pro_qty[$index];
            $stock->batch_no                    = $batch_nos[$index] ?? null; // Save batch_no
            $stock->adjustment                  = 'Issue Order';

            $stock->save();

            // saleman stock
            $saleman_stock                              = new SalemanStock();
            $saleman_stock->order_id                    = $order_id;
            $saleman_stock->pro_id                      = $pro_id;
            $saleman_stock->salesman_id                 = $request->input('salesman_id');
            $saleman_stock->stock_in_qty                = $pro_qty[$index];
            $saleman_stock->batch_no                    = $batch_nos[$index] ?? null; // Save batch_no
            $saleman_stock->adjustment                  = 'Issue Order';
            $saleman_stock->save();

        }


        

        return response()->json(['status' => 200, 'message' => 'Order updated successfully.']);
    }


    public function delete_order($id)
    {
        $order = Order::find($id);

        if (!$order) {
            return response()->json([
                'status' => 404,
                'message' => 'Order not found.'
            ]);
        }

        // Delete order details first
        $order->orderDetails()->delete();
        $order->stocks()->delete();
        $order->salemanStocks()->delete();


        // Delete the order
        $order->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Order and its details deleted successfully.'
        ]);
    }

  
    public function get_orders()
    {
        $user = Auth::user();

        // اگر user کا رول admin یا super_admin نہیں ہے تو filter لگا دو
        if (!$user->hasRole(['Admin', 'Super Admin'])) {
            $orders = Order::where('salesman_id', $user->id)->get();
        } else {
            // admin اور super admin کے لیے سب orders
            $orders = Order::all();
        }

        return response()->json($orders);
    }



    public function view_order($id)
    {
        $order = DB::table('orders')
            ->leftJoin('customers', 'orders.customer_id', '=', 'customers.id')
            ->leftJoin('users', 'orders.salesman_id', '=', 'users.id')

            ->select(
                'orders.*',
                'customers.name as cus_name', 
                'users.name as salesman_name', 

                'orders.id as id'
            )
            ->where('orders.id', $id)
            ->first();

        if (!$order) {
            return response()->json([
                'status' => 404,
                'message' => 'Order not found'
            ], 404);
        }

    
        $order_details = DB::table('order_details')
        ->select('*')->where('order_id', '=', $id)
        ->get();

        $products = [];

        foreach($order_details as $index => $pros) {
            $pro = DB::table('products')->select('*')
                ->where('id', '=', $pros->product_id)->first(); // ✅ correct property name

            $products[] = [
                'no'         => $index + 1,
                'pro_id'     => $pro->id ?? null,
                'pro_name'   => $pro->pro_name ?? 'Unknown Product',
                'pro_price'  => $pros->pro_price ?? 'Price',
                'pro_qty'    => $pros->quantity,
                'total'      => $pros->total ?? 'Total',
            ];
        }


        return response()->json([
            'status' => 200,
            'order' => $order,
            'products' => $products, 
        ]);
    }

    // public function invoice_print()
    // {
    //     // Fetch the last inserted sale
    //     $orders = DB::table('orders')
    //         ->leftJoin('customers', 'orders.cus_id', '=', 'customers.id')
    //         ->leftJoin('users', 'orders.customer_id', '=', 'users.id')

    //         ->select(
    //             'orders.*',
    //             'customers.name as cus_name',
    //             'users.name as salesman_name',

    //             'orders.id as order_id'
    //         )
    //         ->orderBy('orders.id', 'desc') // Get the most recent sale
    //         ->first();

    //     if (!$orders) {
    //         return response()->json([
    //             'status' => 404,
    //             'message' => 'orders not found'
    //         ], 404);
    //     }

    //     // Fetch the details for the sale
    //     $order_details = DB::table('order_details')
    //         ->where('sale_id', $sale->id)
    //         ->get();

    //     $products = [];

    //     foreach ($order_details as $index => $detail) {
    //         $product = DB::table('products')
    //             ->where('id', $detail->pro_id)
    //             ->first();

    //         if ($product) {
    //             $products[] = [
    //                 'no'        => $index + 1,
    //                 'pro_id'    => $product->id,
    //                 'pro_name'  => $product->pro_name,
    //                 'sale_price'=> $detail->pro_price,
    //                 'pro_qty'   => $detail->quantity,
    //                 'total'     => $detail->total,
    //             ];
    //         }
    //     }

    //     return response()->json([
    //         'status' => 200,
    //         'orders' => $orders,
    //         'products' => $products,
    //     ]);
    // }






    

}
