<?php

namespace App\Http\Controllers;

use App\Models\Bom;
use App\Models\ContracterLedger;
use App\Models\Production;
use App\Models\PurchaseDetail;
use App\Models\Stock;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ProductionController extends Controller
{
    
    
    public function load_production(Request $request)
    {
        // $users = User::all();

        // return view('admin.users', compact('users'));

        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        $query = DB::table('productions')
        ->leftJoin('products', 'productions.pro_id', '=', 'products.id')
        ->leftJoin('boms', 'productions.bom_id', '=', 'boms.id')
        ->leftJoin('contractors', 'productions.contractor_id', '=', 'contractors.id')

        ->select(
            'productions.*',
            'products.pro_name','products.sku',
            'boms.bom_name',
            'contractors.name as contractor_name'
        );

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin', 'Store man', 'Production Manager'])) {
            $query->where('productions.created_by', Auth::id());
        }


        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('created_at', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('products.pro_name', 'like', '%' . $search . '%');
        }



        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $users = $query->get();

   

        $all_data = [];
        foreach ($users as $user) {

           

            $btn = '<td class="text-right">';

            // View button
            // $btn .= '<a href="' . route("customer_ledger", $user->id) . '" 
            // class="btn btn-success btn-icon" data-toggle="tooltip" title="View" style="padding: 6px 10px;"><i class="fa-duotone fa-memo" style="margin-right: 3px"></i></a>';

            // Hide Edit button only for Storeman when status is 'issue'
            if (Auth::user()->can('update_production') && !($user->status == 'issue' && Auth::user()->hasRole('Store man'))) {
                $btn .= '<button id="edit_btn" data-eid="'.$user->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_production')) {
                $btn .= '<button id="delete_btn" data-did="'.$user->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            if ($user->status == "pending") {
                $status = '<div class="label label-table label-purple" style="width:100%; max-width:130px;">Pending</div>';
            } else if ($user->status == "issue") {
                $status = '<div class="label label-table label-warning" style="width:100%; max-width:130px;">Issue</div>';
            } else if ($user->status == "completed") {
                $status = '<div class="label label-table label-success" style="width:100%; max-width:130px;">Completed</div>';
            } else if ($user->status == "cancelled") {
                $status = '<div class="label label-table label-danger" style="width:100%; max-width:130px;">Cancel</div>';
            } else if ($user->status == "production") {
                $status = '<div class="label label-table label-danger" style="width:100%; max-width:130px;">Production</div>';
            }

           

            $all_data[] = [
                'id'                        => $user->id,
                'pro_name'                  => $user->pro_name,
                'sku'                       => $user->sku,
                'pro_qty'                   => $user->pro_qty,
                'status'                    => $status,
                'start_date'                => $user->start_date,
                'btn'                       => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
        ];

        return response()->json($data);


    }

    public function insert_production(Request $request)
    {

        $data = new Production();
        $data->contractor_id            = $request->input('contractor_id');
        $data->pro_id                   = $request->input('pro_id');
        $data->bom_id                   = $request->input('bom_id');
        $data->pro_qty                  = $request->input('pro_qty');
        $data->batch_no                 = $request->input('batch_no');
        $data->bom_price                = $request->input('bom_price');
        $data->single_product_price     = $request->input('single_product_price');
        $data->total_amount             = $request->input('total_amount');

        // Save labour cost from total_cost field (form sends it as total_cost)
        $data->total_cost               = $request->input('total_cost', 0);
        $data->labour_cost              = $request->input('total_cost', 0);

        $data->start_date               = Carbon::parse($request->input('start_date'))->format('Y-m-d');
        $data->created_by               = Auth::id();

        $data->save();

        return response()->json([
            'status'=>200
        ]);
    }

    public function edit_production($id)
    {

        $data = DB::table('productions')
        ->leftJoin('products', 'productions.pro_id', '=', 'products.id')
        ->leftJoin('contractors', 'productions.contractor_id', '=', 'contractors.id')
        ->leftJoin('boms', 'productions.bom_id', '=', 'boms.id')
        ->leftJoin('banks', 'productions.bank_id', '=', 'banks.id')



        ->select(
            'productions.*',
            'products.pro_name',
            'contractors.name as contractor_name',
            'boms.bom_name',
            'banks.bank_name'
        )
        ->where('productions.id', $id)
        ->first();

         
        return response()->json([
            'status' => 200,
            'data' => $data, 
        ]);
    }

    public function update_production(Request $request)
    {

        $data = Production::findOrFail($request->input('production_id'));
        $data->contractor_id                    = $request->input('contractor_id');
        $data->bom_id                           = $request->input('bom_id');
        $data->pro_id                           = $request->input('pro_id');
        $data->pro_qty                          = $request->input('pro_qty');
        $data->payment_type                     = $request->input('payment_type');
        $data->bank_id                          = $request->input('bank_id');

        // Save labour cost - form sends it as total_cost
        $data->total_cost                       = $request->input('total_cost');
        $data->labour_cost                      = $request->input('total_cost');

        $data->single_product_price             = $request->input('single_product_price');
        $data->sale_price                       = $request->input('sale_price');

        $data->total_amount                     = $request->input('total_amount');
        $data->batch_no                         = $request->input('batch_no');

        $data->updated_by                       = Auth::id();

        // status sirf tab update kare jab request me ho
        if ($request->filled('status')) {
            $data->status = $request->status;
        }

        $data->save();

        // ===============================
        // Inventory (Main Store) Handling
        // ===============================

        // ---------------- STOCK ADJUSTMENTS ----------------
        if ($request->status === 'issue') {
            // BOM Products ko stock se less karna hai
            $bom = Bom::with('bomDetails')->find($data->bom_id); // items = BOM products

            if ($bom) {
                foreach ($bom->bomDetails as $item) {
                    Stock::create([
                        'pro_id'        => $item->pro_id,
                        'production_id' => $data->id,

                        'stock_in_qty'  => 0,
                        'stock_out_qty' => ($item->quantity * $data->pro_qty), // required * production qty
                        'adjustment'    => 'Production Issue',
                    ]);
                }
            }
        }

        if ($request->status === 'completed') {
            // Production complete â†' Final product stock me add
            Stock::create([
                'pro_id'        => $data->pro_id,
                'production_id' => $data->id,
                'stock_in_qty'  => $data->pro_qty,
                'stock_out_qty' => 0,
                'adjustment'    => 'Production Completed',
            ]);

            PurchaseDetail::create([
                'pro_id'            => $request->input('pro_id'),
                'production_id'     => $data->id,
                'pro_qty'           => $data->pro_qty,
                'single_price'      => $request->input('single_product_price'),
                'sale_price'        => $request->input('sale_price'),
                'total_amount'      => $request->input('total_amount'),
                'batch_no'          => $request->input('batch_no'),

            ]);

            // Create contractor ledger entry when production is completed
            if ($data->contractor_id) {
                // Check if contractor ledger entry already exists for this production
                $existingEntry = ContracterLedger::where('production_id', $data->id)->first();

                if (!$existingEntry) {
                    // Create contracter ledger entry for payment tracking
                    $contracter_ledger = new ContracterLedger();
                    $contracter_ledger->contractor_id       = $data->contractor_id;
                    $contracter_ledger->date                = $data->start_date ?? now()->format('Y-m-d');
                    $contracter_ledger->production_id       = $data->id;
                    $contracter_ledger->pro_id              = $data->pro_id;
                    $contracter_ledger->pro_qty             = $data->pro_qty;
                    $contracter_ledger->cost                = $data->total_cost ?? 0;
                    $contracter_ledger->status              = 'Completed';
                    $contracter_ledger->batch_number        = $data->batch_no;
                    $contracter_ledger->price               = $data->single_product_price ?? 0;
                    $contracter_ledger->labour_cost         = $data->labour_cost ?? 0;
                    $contracter_ledger->material_cost       = 0;
                    $contracter_ledger->total_price         = $data->total_amount ?? 0;
                    $contracter_ledger->payment_type        = 'Cash';
                    $contracter_ledger->payment_status      = 'Pending';
                    $contracter_ledger->created_by          = Auth::id();
                    $contracter_ledger->save();
                }
            }
        }

        return response()->json([
            'status'=>200
        ]);
    }

    public function delete_production($id)
    {
        $data = Production::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Production record not found.'
            ]);
        }

        // Delete related stocks first
        Stock::where('production_id', $data->id)->delete();
        PurchaseDetail::where('production_id', $data->id)->delete();
        ContracterLedger::where('production_id', $data->id)->delete();

        // Delete production
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Production and related stocks deleted successfully.'
        ]);
    }



    public function get_production()
    {
      
        $production = Production::all();

        return response()->json($production);
    }

    public function get_bom_products($bom_id)
    {
        $bom_products = DB::table('bom_details')
            ->join('products', 'bom_details.pro_id', '=', 'products.id')
            ->select(
                'bom_details.pro_id',
                'products.pro_name',
                'products.sku',
                'bom_details.quantity'
            )
            ->where('bom_details.bom_id', $bom_id)
            ->get();

        return response()->json($bom_products);
    }

    public function get_product_balance($product_id)
    {
        $balance = DB::table('stocks')
            ->select(
                DB::raw('COALESCE(SUM(stocks.stock_in_qty), 0) - COALESCE(SUM(stocks.stock_out_qty), 0) as balance')
            )
            ->where('stocks.pro_id', $product_id)
            ->groupBy('stocks.pro_id')
            ->first();

        return response()->json([
            'balance' => $balance ? $balance->balance : 0
        ]);
    }

    /**
     * Pay Labour Cost for Production
     * Deducts the amount from Cash or Bank based on payment type
     */
    public function pay_labour_cost(Request $request)
    {
        DB::beginTransaction();
        try {
            $production_id = $request->input('production_id');
            $payment_type = $request->input('payment_type');
            $bank_id = $request->input('bank_id');
            $labour_cost = $request->input('labour_cost');

            // Find the production
            $production = Production::findOrFail($production_id);

            // Update production with payment details
            $production->payment_type = $payment_type;
            $production->bank_id = $bank_id;
            $production->labour_cost = $labour_cost;
            $production->payment_status = 'paid';
            $production->updated_by = Auth::id();
            $production->save();

            // Deduct from Cash or Bank based on payment type
            if ($payment_type === 'Cash') {
                // Deduct from hand cash
                DB::table('hand_cashes')->decrement('cash_amount', $labour_cost);

                // Record transaction
                DB::table('transactions')->insert([
                    'payment_type'      => 'Cash',
                    'amount'            => $labour_cost,
                    'transaction_type'  => 'withdraw',
                    'source_type'       => 'Production Labour Cost',
                    'source_id'         => $production->id,
                    'description'       => 'Production Labour Cost Payment',
                    'created_at'        => now(),
                ]);

            } elseif ($payment_type === 'Bank' && $bank_id) {
                // Deduct from bank balance
                DB::table('banks')->where('id', $bank_id)
                    ->decrement('bank_balance', $labour_cost);

                // Record transaction
                DB::table('transactions')->insert([
                    'payment_type'      => 'Bank',
                    'bank_id'           => $bank_id,
                    'amount'            => $labour_cost,
                    'transaction_type'  => 'withdraw',
                    'source_type'       => 'Production Labour Cost',
                    'source_id'         => $production->id,
                    'description'       => 'Production Labour Cost Payment',
                    'created_at'        => now(),
                ]);
            }

            // Also update Contractor Ledger if contractor is assigned
            if ($production->contractor_id) {
                DB::table('contracter_ledgers')->insert([
                    'contractor_id'     => $production->contractor_id,
                    'production_id'     => $production->id,
                    'date'              => now()->format('Y-m-d'),
                    'detail'            => 'Labour Cost Payment',
                    'batch_number'      => $production->batch_no,
                    'batch_price'       => $production->bom_price ?? 0,
                    'labour_cost'       => $labour_cost,
                    'transaction_type'  => 'Labour Cost',
                    'payment_type'      => $payment_type,
                    'bank_id'           => $bank_id,
                    'pay'               => $payment_type === 'Cash' ? $labour_cost : 0,
                    'bank'              => $payment_type === 'Bank' ? $labour_cost : 0,
                    'created_at'        => now(),
                ]);
            }

            DB::commit();

            return response()->json([
                'status' => 200,
                'message' => 'Labour cost payment successful!'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error in pay_labour_cost: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error: ' . $e->getMessage()
            ], 500);
        }
    }


}