<?php

namespace App\Http\Controllers;

use App\Models\Purchase;
use App\Models\PurchaseDetail;
use App\Models\Stock;
use App\Models\SupplierLedger;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    

    public function load_purchase(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

    

        $query = DB::table('purchases')
        ->join('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
     
        ->select([
            'purchases.*', 
            'suppliers.sup_name', 
            'suppliers.company_name', 

            'purchases.id as id'
        ]);



        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('purchases.date', [$start_date, $end_date]);
        }

     
        if ($search = $request->input('search')) { 
            $query->where(function ($q) use ($search) {
                $q->where('suppliers.sup_name', 'like', '%' . $search . '%')
                  ->orWhere('suppliers.company_name', 'like', '%' . $search . '%')
                  ->orWhere('purchases.date', 'like', '%' . $search . '%');
            });
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('purchases.is_active', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $purchases = $query->get();

        $all_data = [];

        
        foreach ($purchases as $data) {

           

            $btn = '<td >';
            if (Auth::user()->can('view_purchases')) {
                $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }

            if (Auth::user()->can('update_purchase')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_purchase')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            $all_data[] = [
                'id'            => $data->id,
                'date'          => $data->date,
                'company_name'  => $data->company_name,
                'sup_name'      => $data->sup_name,
                
                'grand_total'   => $data->grand_total,
                'paid_amount'   => $data->paid_amount,
                'remain_amount' => $data->remain_amount,
                'btn'           => $btn
            ];
        };

      

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }



    public function insert_purchase(Request $request)
    {
        DB::beginTransaction();
        try {
            $purchase = new Purchase();
            $purchase->date                 = Carbon::parse($request->input('date'))->format('Y-m-d');
            $purchase->supplier_id          = $request->input('supplier_id');
            $purchase->payment_type         = $request->input('payment_type');
            $purchase->bank_id              = $request->input('bank_id');

            $purchase->pur_detail           = $request->input('pur_detail');
            $purchase->grand_total          = $request->input('grand_total');
            $purchase->paid_amount          = $request->input('paid_amount') ?? 0;
            $purchase->remain_amount        = $request->input('remain_amount') ?? 0;
            $purchase->created_by           = Auth::id();
            $purchase->save();

            $purchase_id = $purchase->id;
            $paid_amount = floatval($request->input('paid_amount')) ?? 0;

            // Handle payment - deduct from hand cash or bank when purchase is made with payment
            if ($paid_amount > 0) {
                if ($request->payment_type === "Cash") {
                    // Check if hand_cashes table has any rows
                    $handCashExists = DB::table('hand_cashes')->exists();

                    if ($handCashExists) {
                        // Decrement existing row(s)
                        DB::table('hand_cashes')->decrement('cash_amount', $paid_amount);
                    } else {
                        // Create first row with negative amount (purchase withdraws cash)
                        DB::table('hand_cashes')->insert([
                            'cash_amount' => -1 * $paid_amount,
                            'created_by' => Auth::id(),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }

                    // Record transaction
                    DB::table('transactions')->insert([
                        'payment_type'      => 'Cash',
                        'amount'            => $paid_amount,
                        'transaction_type'  => 'withdraw',
                        'source_type'       => 'Purchase',
                        'source_id'         => $purchase->id,
                        'description'       => 'Purchase Payment Made',
                        'created_at'        => now(),
                    ]);

                } elseif ($request->payment_type === "Bank" && $request->bank_id) {

                    DB::table('banks')->where('id', $request->bank_id)
                        ->decrement('bank_balance', $paid_amount);

                    DB::table('transactions')->insert([
                        'payment_type'      => 'Bank',
                        'bank_id'           => $request->input('bank_id'),
                        'amount'            => $paid_amount,
                        'transaction_type'  => 'withdraw',
                        'source_type'       => 'Purchase',
                        'source_id'         => $purchase->id,
                        'description'       => 'Purchase Payment Made',
                        'created_at'        => now(),
                    ]);
                }
            }

            // Extract supplier ledger data
            $products           = implode(",", $request->input('pro_id'));
            $pro_qty1           = implode(",", $request->input('pro_qty'));

            // Extract data from the request
            $pro_ids            = $request->input('pro_id');
            $pro_unit           = $request->input('pro_unit');
            $purchase_price     = $request->input('purchase_price');
            $sale_price         = $request->input('sale_price');
            $batch_no           = $request->input('batch_no');
            $purchase_fright    = $request->input('purchase_fright');
            $purchase_tax       = $request->input('purchase_tax');
            $pro_qty            = $request->input('pro_qty');
            $single_price       = $request->input('single_price');
            $total              = $request->input('total');


            // Loop through the data and save each entry
            foreach ($pro_ids as $index => $pro_id) {

                $product_stock                      = new PurchaseDetail();
                $product_stock->purchase_id         = $purchase_id;
                $product_stock->pro_id              = $pro_id;
                $product_stock->pro_unit            = $pro_unit[$index] ?? '';
                $product_stock->purchase_price      = $purchase_price[$index] ?? 0;
                $product_stock->sale_price          = $sale_price[$index] ?? 0;
                $product_stock->batch_no            = $batch_no[$index] ?? '';
                $product_stock->purchase_fright     = $purchase_fright[$index] ?? 0;
                $product_stock->purchase_tax        = $purchase_tax[$index] ?? 0;
                $product_stock->pro_qty             = $pro_qty[$index] ?? 0;
                $product_stock->single_price        = $single_price[$index] ?? 0;

                $product_stock->total_amount        = $total[$index] ?? 0;
                $product_stock->save();

                // Manage stock
                $stock                      = new Stock();
                $stock->purchase_id         = $purchase_id;
                $stock->pro_id              = $pro_id;
                $stock->invoice_date        = Carbon::parse($request->input('date'))->format('Y-m-d');
                $stock->purchase_price      = $single_price[$index] ?? 0;
                $stock->stock_in_qty        = $pro_qty[$index] ?? 0;
                $stock->adjustment          = 'Purchase';
                $stock->save();


            }

            // Insert into SupplierLedger table
            $supplier_ledger = new SupplierLedger();
            $supplier_ledger->supplier_id               = $request->input('supplier_id');
            $supplier_ledger->purchase_id               = $purchase_id;
            $supplier_ledger->date                      = Carbon::parse($request->input('date'))->format('Y-m-d');
            $supplier_ledger->detail                    = "Purchase";
            $supplier_ledger->pro_id                    = $products;
            $supplier_ledger->pro_qty                   = $pro_qty1;
            $supplier_ledger->transaction_type          = "Purchase";
            $supplier_ledger->bank_id                   = $request->input('bank_id');

            // Handle payment type
            if ($request->input('payment_type') === "Cash") {
                $supplier_ledger->pay   = $paid_amount;
                $supplier_ledger->bank  = 0;

            } elseif ($request->input('payment_type') === "Bank") {
                $supplier_ledger->pay   = 0;
                $supplier_ledger->bank  = $paid_amount;
            }
 
            $supplier_ledger->pending = floatval($request->input('grand_total')) - $paid_amount;
            $supplier_ledger->save();

            DB::commit();
            return response()->json(['status' => 200, 'message' => 'Purchase saved successfully.']);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error inserting purchase: ' . $e->getMessage());
            return response()->json(['status' => 500, 'message' => 'Error: ' . $e->getMessage()]);
        }

        
    }


    public function edit_purchase($id)
    {
        // Retrieve the purchase with supplier and product details
        $data = DB::table('purchases')
            ->leftJoin('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
            ->leftJoin('banks', 'purchases.bank_id', '=', 'banks.id')
            ->select(
                'purchases.*',
                'suppliers.sup_name',
                'banks.bank_name'
            )
            ->where('purchases.id', $id)
            ->first();
    
        // Retrieve purchase details
        $purchase_details = DB::table('purchase_details')
            ->join('products', 'purchase_details.pro_id', '=', 'products.id')
            ->select(
                'purchase_details.*',
                'products.pro_name'
            )
            ->where('purchase_details.purchase_id', '=', $id)
            ->get();
    
        // Ensure data exists before returning
        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Purchase not found.',
            ]);
        }
    
        return response()->json([
            'status' => 200,
            'data' => $data,
            'purchase_details' => $purchase_details,
        ]);
    }
    

    public function update_purchase(Request $request)
    {
        DB::beginTransaction();
        try {
            // Find the purchase record by ID
            $purchase = Purchase::find($request->input('pur_id'));

            if (!$purchase) {
                return response()->json(['status' => 404, 'message' => 'Purchase not found.']);
            }

            // Step 1: Reverse old payment
            $old_paid_amount = floatval($purchase->paid_amount) ?? 0;
            if ($old_paid_amount > 0) {
                if ($purchase->payment_type === "Cash") {
                    DB::table('hand_cashes')->increment('cash_amount', $old_paid_amount);
                } elseif ($purchase->payment_type === "Bank" && $purchase->bank_id) {
                    DB::table('banks')->where('id', $purchase->bank_id)
                        ->increment('bank_balance', $old_paid_amount);
                }
            }

            // Update the main purchase record
            $purchase->supplier_id      = $request->input('supplier_id');
            $purchase->date             = Carbon::parse($request->input('date'))->format('Y-m-d');
            $purchase->payment_type     = $request->input('payment_type');
            $purchase->bank_id          = $request->input('bank_id');
            $purchase->pur_detail       = $request->input('pur_detail');
            $purchase->grand_total      = $request->input('grand_total');
            $purchase->paid_amount      = $request->input('paid_amount') ?? 0;
            $purchase->remain_amount    = $request->input('remain_amount') ?? 0;
            $purchase->updated_by       = Auth::id();
            $purchase->save();

            
            
            // Delete old purchase details for this purchase
            PurchaseDetail::where('purchase_id', $purchase->id)->delete();
            Stock::where('purchase_id', $purchase->id)->delete();
            SupplierLedger::where('purchase_id', $purchase->id)->delete();
            Transaction::where('source_type', 'Purchase')->where('source_id', $purchase->id)->delete();


            // Apply new payment
            $new_paid_amount = floatval($request->input('paid_amount')) ?? 0;

            if ($new_paid_amount > 0) {
                if ($request->payment_type === "Cash") {
                    // Check if hand_cashes table has any rows
                    $handCashExists = DB::table('hand_cashes')->exists();

                    if ($handCashExists) {
                        // Decrement existing row(s)
                        DB::table('hand_cashes')->decrement('cash_amount', $new_paid_amount);
                    } else {
                        // Create first row with negative amount (purchase withdraws cash)
                        DB::table('hand_cashes')->insert([
                            'cash_amount' => -1 * $new_paid_amount,
                            'created_by' => Auth::id(),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }

                    DB::table('transactions')->insert([
                        'payment_type'      => 'Cash',
                        'amount'            => $new_paid_amount,
                        'transaction_type'  => 'withdraw',
                        'source_type'       => 'Purchase',
                        'source_id'         => $purchase->id,
                        'description'       => 'Purchase Payment Made',
                        'created_at'        => now(),

                    ]);


                } elseif ($request->payment_type === "Bank" && $request->bank_id) {

                    DB::table('banks')->where('id', $request->bank_id)
                        ->decrement('bank_balance', $new_paid_amount);

                    DB::table('transactions')->insert([
                        'payment_type'      => 'Bank',
                        'bank_id'           => $request->input('bank_id'),
                        'amount'            => $new_paid_amount,
                        'transaction_type'  => 'withdraw',
                        'source_type'       => 'Purchase',
                        'source_id'         => $purchase->id,
                        'description'       => 'Purchase Payment Made',
                        'created_at'        => now(),
                    ]);
                }
            }


            // Reinsert the purchase details
            $pro_ids = $request->input('pro_id');
            $pro_unit = $request->input('pro_unit');
            $purchase_price = $request->input('purchase_price');
            $sale_price = $request->input('sale_price');
            $batch_no = $request->input('batch_no');

            $purchase_fright = $request->input('purchase_fright');
            $purchase_tax = $request->input('purchase_tax');
            $pro_qty = $request->input('pro_qty');
            $single_price = $request->input('single_price');

            $total = $request->input('total');

            foreach ($pro_ids as $index => $pro_id) {
                $product_stock = new PurchaseDetail();
                $product_stock->purchase_id         = $purchase->id;
                $product_stock->pro_id              = $pro_id;
                $product_stock->pro_unit            = $pro_unit[$index] ?? '';
                $product_stock->purchase_price      = $purchase_price[$index] ?? 0;
                $product_stock->sale_price          = $sale_price[$index] ?? 0;
                $product_stock->batch_no            = $batch_no[$index] ?? '';
                $product_stock->purchase_fright     = $purchase_fright[$index] ?? 0;
                $product_stock->purchase_tax        = $purchase_tax[$index] ?? 0;
                $product_stock->pro_qty             = $pro_qty[$index] ?? 0;
                $product_stock->single_price        = $single_price[$index] ?? 0;

                $product_stock->total_amount        = $total[$index] ?? 0;
                $product_stock->save();


                $stock                  = new Stock();
                $stock->pro_id          = $pro_id;
                $stock->purchase_id     = $purchase->id;
                $stock->purchase_price  = $single_price[$index] ?? 0;
                $stock->stock_in_qty    = $pro_qty[$index] ?? 0;
                $stock->invoice_date    = Carbon::parse($request->input('date'))->format('Y-m-d');
                $stock->adjustment      = 'Purchase';
                $stock->save();
              
            }

            // Extract supplier ledger data
            $products    = implode(",", $request->input('pro_id'));
            $pro_qty1    = implode(",", $request->input('pro_qty'));
            
            // Insert into SupplierLedger table
            $supplier_ledger = new SupplierLedger();
            $supplier_ledger->supplier_id               = $request->input('supplier_id');
            $supplier_ledger->purchase_id               = $purchase->id;
            $supplier_ledger->date                      = Carbon::parse($request->input('date'))->format('Y-m-d');
            $supplier_ledger->detail                    = "Purchase";
            $supplier_ledger->pro_id                    = $products;
            $supplier_ledger->pro_qty                   = $pro_qty1;
            $supplier_ledger->transaction_type          = "Purchase";
            $supplier_ledger->bank_id                   = $request->input('bank_id');

            // Handle payment type
            if ($request->input('payment_type') === "Cash") {
                $supplier_ledger->pay   = $new_paid_amount;
                $supplier_ledger->bank  = 0;

            } elseif ($request->input('payment_type') === "Bank") {
                $supplier_ledger->pay   = 0;
                $supplier_ledger->bank  = $new_paid_amount;
            }
 
            $supplier_ledger->pending = floatval($request->input('grand_total')) - $new_paid_amount;
            $supplier_ledger->save();

            DB::commit();
            return response()->json(['status' => 200, 'message' => 'Purchase updated successfully.']);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error updating purchase: ' . $e->getMessage());
            return response()->json(['status' => 500, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    
    public function view_purchase($id)
    {
        $purchase = DB::table('purchases')
            ->leftJoin('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
            ->select(
                '*', 'suppliers.sup_name',
                'purchases.id as id'
            )
            ->where('purchases.id', $id)
            ->first();

        if (!$purchase) {
            return response()->json([
                'status' => 404,
                'message' => 'Purchase not found'
            ], 404);
        }

        $all_suppliers = DB::table('suppliers')->select('id', 'sup_name')->get();

        $purchase_details = DB::table('purchase_details')
        ->select('*')->where('purchase_id', '=', $id)
        ->get();

        $products = [];

        foreach($purchase_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();
            
            $products[] = [
                'no'        => $index+1,
                'pro_id'     => $pro->id ?? null,
                'pro_name'   => $pro->pro_name ?? 'Unknown',
                'purchase_price' => $pros->purchase_price,
                'pro_qty'    => $pros->pro_qty,
                'total_amount'=> $pros->total_amount,
            ];
        }

        return response()->json([
            'status' => 200,
            'purchase' => $purchase,
            'products' => $products,
            'all_suppliers' => $all_suppliers,
        ]);
    }

    public function delete_purchase($id)
    {
        DB::beginTransaction();
        try {
            $data = Purchase::find($id);

            if (!$data) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Purchase not found.'
                ]);
            }

            // Reverse entry using saved data
            $paid_amount = floatval($data->paid_amount) ?? 0;
            if ($paid_amount > 0) {
                if ($data->payment_type === "Cash") {
                    DB::table('hand_cashes')->increment('cash_amount', $paid_amount);

                } elseif ($data->payment_type === "Bank" && $data->bank_id) {
                    DB::table('banks')->where('id', $data->bank_id)
                        ->increment('bank_balance', $paid_amount);
                }
            }

            // Delete PurchaseDetails
            PurchaseDetail::where('purchase_id', $data->id)->delete();
            Stock::where('purchase_id', $data->id)->delete();
            SupplierLedger::where('purchase_id', $data->id)->delete();
            Transaction::where('source_type', 'Purchase')
                ->where('source_id', $data->id)
                ->delete();


            // Delete Purchase
            $data->delete();

            DB::commit();
            return response()->json([
                'status' => 200,
                'message' => 'Purchase and its details deleted successfully.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error deleting purchase: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }
  


    public function todaypurchase(Request $request)
    {
        $today = now()->toDateString();

        $today_purchase = DB::table('purchases')
            ->whereDate('date', $today)
            ->whereNotNull('grand_total')
            ->sum('grand_total');

        return response()->json(['today_purchase' => 'Rs. ' . number_format($today_purchase, 2)]);
    }


    public function totalpurchase()
    {
        $total_purchase = DB::table('purchases')
        ->whereNotNull('grand_total')
        ->sum('grand_total');

        return response()->json([
            'total_purchase' => 'Rs. ' . number_format($total_purchase, 2)
        ]);
    }


    public function weeklypurchase()
    {
        $startDate = Carbon::now()->subDays(7)->toDateString();
        $endDate = Carbon::now()->toDateString();

        $weeklypurchase = Purchase::whereBetween('date', [$startDate, $endDate])
                        ->sum('grand_total');

        return response()->json(['weeklypurchase' => 'Rs. ' . number_format($weeklypurchase, 2)]);
    }


    public function monthlypurchase()
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $monthlypurchase = DB::table('purchases')
            ->whereYear('date', $currentYear)
            ->whereMonth('date', $currentMonth)
            ->sum('grand_total');

        return response()->json([
            'monthlypurchase' =>  'Rs. ' . number_format($monthlypurchase, 2)
        ]);
    }




}