<?php

namespace App\Http\Controllers;

use App\Models\PurchaseDetail;
use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnDetail;
use App\Models\Stock;
use App\Models\SupplierLedger;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseReturnController extends Controller
{


    public function load_purchase_return(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        $query = DB::table('purchase_returns')
        ->join('suppliers', 'purchase_returns.supplier_id', '=', 'suppliers.id')
        ->leftJoin('users as updater', 'purchase_returns.updated_by', '=', 'updater.id')
        ->select([
            'purchase_returns.*',
            'suppliers.sup_name as supplier_name',
            DB::raw('COALESCE(suppliers.sup_name, "---") as shop_name'),
            'purchase_returns.id as id',
            'purchase_returns.created_at',
            'updater.name as updated_by_name',
            DB::raw('(SELECT SUM(CAST(pro_qty AS DECIMAL(10,2))) FROM purchase_return_details WHERE purchase_return_details.purchase_return_id = purchase_returns.id) as total_qty')
        ]);

        // Role-based filter - Only Admin and Super Admin can see all purchase returns
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('purchase_returns.created_by', Auth::id());
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('return_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) {
            $query->where('suppliers.sup_name', 'like', '%' . $search . '%');
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $purchase_returns = $query->get();

        $all_data = [];
        foreach ($purchase_returns as $data) {

            $btn = '<td class="text-right">';

                if (Auth::user()->can('update_purchase_return')) {
                    $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
                }
                if (Auth::user()->can('delete_purchase_return')) {
                    $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
                }
            $btn .= '</td>';

            // Format status for display
            $status_display = '----';
            if ($data->status == 0) {
                $status_display = 'Pending';
            } elseif ($data->status == 1) {
                $status_display = 'Approved';
            } elseif ($data->status == 2) {
                $status_display = 'Rejected';
            }

            $all_data[] = [
                'id'            => $data->id,
                'date'          => Carbon::parse($data->created_at)->format('d-m-Y'),
                'shop_name'     => $data->shop_name,
                'total_qty'     => number_format($data->total_qty ?? 0, 0),
                'updated_by'    => $data->updated_by_name ?? '----',
                'subtotal'      => number_format($data->subtotal, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'refund_amount' => number_format($data->refund_amount, 0),
                'adjustment_amount' => number_format($data->adjustment_amount, 0),
                'status'        => $status_display,
                'btn'           => $btn
            ];

        }

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_purchase_return(Request $request)
    {
        DB::beginTransaction();

        try {
            $purchaseReturn = new PurchaseReturn();
            $purchaseReturn->return_date            = Carbon::parse($request->input('return_date'))->format('Y-m-d');
            $purchaseReturn->supplier_id            = $request->input('supplier_id');
            $purchaseReturn->bank_id                = $request->input('bank_id');
            $purchaseReturn->payment_type           = $request->input('payment_type');
            $purchaseReturn->detail                 = $request->input('detail');
            $purchaseReturn->subtotal               = $request->input('subtotal');
            $purchaseReturn->grand_total            = $request->input('grand_total');
            $purchaseReturn->refund_amount          = $request->input('refund_amount');
            $purchaseReturn->adjustment_amount      = $request->input('adjustment_amount');
            $purchaseReturn->status                 = 0; // Set default status to pending
            $purchaseReturn->created_by             = Auth::id();
            $purchaseReturn->save();

            // Extract product data
            $products    = implode(",", $request->input('pro_id'));
            $pro_qty1    = implode(",", $request->input('pro_qty'));

            // Get the last inserted purchase return ID
            $purchase_return_id = $purchaseReturn->id;

            // Extract data from the request
            $pro_ids         = $request->input('pro_id');
            $batch_no        = $request->input('batch_no');
            $purchase_price  = $request->input('purchase_price');
            $pro_qty         = $request->input('pro_qty');
            $total           = $request->input('total');

            // Loop through the data and save each entry
            foreach ($pro_ids as $index => $pro_id) {
                $returnDetail                          = new PurchaseReturnDetail();
                $returnDetail->purchase_return_id      = $purchase_return_id;
                $returnDetail->pro_id                  = $pro_id;
                $returnDetail->batch_no                = $batch_no[$index];
                $returnDetail->purchase_price          = $purchase_price[$index];
                $returnDetail->pro_qty                 = $pro_qty[$index];
                $returnDetail->total                   = $total[$index];
                $returnDetail->save();
            }

            // Insert into Supplier Ledger table
            $ledger = new SupplierLedger();
            $ledger->supplier_id            = $request->input('supplier_id');
            $ledger->pur_return_id          = $purchase_return_id;
            $ledger->date                   = Carbon::parse($request->input('return_date'))->format('Y-m-d');
            $ledger->detail                 = $request->input('detail') ?: 'Purchase Return';
            $ledger->pro_id                 = $products;
            $ledger->pro_qty                = $pro_qty1;
            $ledger->transaction_type       = 'Pur_return';
            $ledger->bank_id                = $request->input('bank_id');
            // Purchase return: amount paid back to us (positive pay)
            $ledger->pay                    = $request->input('refund_amount') ?: 0;
            // Pending/adjustment amount
            $ledger->pending                = $request->input('adjustment_amount') ?: 0;
            $ledger->bank                   = ($request->input('payment_type') === 'Bank' ? $request->input('refund_amount') : 0);
            $ledger->save();

            DB::commit();

            return response()->json([
                'status' => 200
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error creating purchase return: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error creating purchase return: ' . $e->getMessage()
            ]);
        }
    }


    public function edit_purchase_return($id, Request $request)
    {
        // Retrieve the purchase return with supplier details
        $purchaseReturn = DB::table('purchase_returns')
            ->leftJoin('suppliers', 'purchase_returns.supplier_id', '=', 'suppliers.id')
            ->leftJoin('banks', 'purchase_returns.bank_id', '=', 'banks.id')
            ->select(
                'purchase_returns.*',
                'suppliers.sup_name as supplier_name',
                DB::raw('COALESCE(suppliers.sup_name, suppliers.sup_name) as supplier_business_name'),
                'banks.bank_name',
                'purchase_returns.id as id'
            )
            ->where('purchase_returns.id', $id)
            ->first();

        if (!$purchaseReturn) {
            return response()->json([
                'status' => 404,
                'message' => 'Purchase return not found'
            ], 404);
        }

        // Retrieve purchase return details
        $return_details = DB::table('purchase_return_details')
            ->join('products', 'purchase_return_details.pro_id', '=', 'products.id')
            ->select(
                'purchase_return_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('purchase_return_details.purchase_return_id', '=', $id)
            ->get();

        // Calculate stock for each product
        foreach ($return_details as $detail) {
            $stock = DB::table('stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'purchase_return' => $purchaseReturn,
            'products' => $return_details
        ]);
    }



    public function view_purchase_return($id)
    {
        $purchaseReturn = DB::table('purchase_returns')
            ->leftJoin('suppliers', 'purchase_returns.supplier_id', '=', 'suppliers.id')
            ->select(
                'purchase_returns.*',
                'suppliers.sup_name',
                'suppliers.phone',
                'suppliers.address',
                DB::raw('COALESCE(suppliers.sup_name, suppliers.sup_name) as supplier_business_name'),
                'purchase_returns.id as id'
            )
            ->where('purchase_returns.id', $id)
            ->first();

        if (!$purchaseReturn) {
            return response()->json([
                'status' => 404,
                'message' => 'Purchase Return not found'
            ], 404);
        }

        $all_suppliers = DB::table('suppliers')->select('id', 'sup_name')->get();

        $return_details = DB::table('purchase_return_details')
        ->select('*')->where('purchase_return_id', '=', $id)
        ->get();

        $products = [];

        foreach($return_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();

            if ($pro) {
                $products[] = [
                    'no'             => $index+1,
                    'pro_id'         => $pro->id,
                    'pro_name'       => $pro->pro_name,
                    'purchase_price' => $pros->purchase_price,
                    'pro_qty'        => $pros->pro_qty,
                    'total'          => $pros->total,
                ];
            }
        }

        return response()->json([
            'status' => 200,
            'purchase_return' => $purchaseReturn,
            'products' => $products,
            'all_suppliers' => $all_suppliers,
        ]);
    }


    public function update_purchase_return(Request $request)
    {
        // Wrap everything in a database transaction for speed and data integrity
        DB::beginTransaction();

        try {
            // Find the purchase return record by ID
            $purchaseReturn = PurchaseReturn::find($request->input('purchase_return_id'));

            if (!$purchaseReturn) {
                DB::rollBack();
                return response()->json(['status' => 404, 'message' => 'Purchase return not found.']);
            }

            // Check if status is being approved
            $oldStatus = $purchaseReturn->status;
            $newStatus = $request->input('order_status');
            $isBeingApproved = ($oldStatus != 1 && $newStatus == 1);

            // Update the main purchase return record
            $purchaseReturn->return_date            = Carbon::parse($request->input('return_date'))->format('Y-m-d');
            $purchaseReturn->supplier_id            = $request->input('supplier_id');
            $purchaseReturn->bank_id                = $request->input('bank_id');
            $purchaseReturn->payment_type           = $request->input('payment_type');
            $purchaseReturn->detail                 = $request->input('detail');
            $purchaseReturn->subtotal               = $request->input('subtotal');
            $purchaseReturn->grand_total            = $request->input('grand_total');
            $purchaseReturn->refund_amount          = $request->input('refund_amount');
            $purchaseReturn->adjustment_amount      = $request->input('adjustment_amount');
            $purchaseReturn->status                 = $newStatus;
            $purchaseReturn->updated_by             = Auth::id();
            $purchaseReturn->save();

            // Delete old purchase return details
            PurchaseReturnDetail::where('purchase_return_id', $purchaseReturn->id)->delete();
            SupplierLedger::where('pur_return_id', $purchaseReturn->id)->delete();
            Stock::where('pur_return_id', $purchaseReturn->id)->delete();
            Transaction::where('source_type', 'Purchase Return')
                ->where('source_id', $purchaseReturn->id)
                ->delete();

            // If being approved, process refund/adjustment
            if ($isBeingApproved) {
                // Process refund to cash/bank if applicable
                if ($request->refund_amount > 0) {
                    if ($request->payment_type === "Cash") {
                        DB::table('hand_cashes')->increment('cash_amount', $request->refund_amount);

                        DB::table('transactions')->insert([
                            'payment_type'      => 'Cash',
                            'amount'            => $request->refund_amount,
                            'transaction_type'  => 'deposit',
                            'source_type'       => 'Purchase Return',
                            'source_id'         => $purchaseReturn->id,
                            'description'       => 'Purchase Return Approved - Refund',
                            'created_at'        => now(),
                        ]);

                    } elseif ($request->payment_type === "Bank" && $request->bank_id) {
                        DB::table('banks')->where('id', $request->bank_id)
                            ->increment('bank_balance', $request->refund_amount);

                        DB::table('transactions')->insert([
                            'payment_type'      => 'Bank',
                            'bank_id'           => $request->bank_id,
                            'amount'            => $request->refund_amount,
                            'transaction_type'  => 'deposit',
                            'source_type'       => 'Purchase Return',
                            'source_id'         => $purchaseReturn->id,
                            'description'       => 'Purchase Return Approved - Refund',
                            'created_at'        => now(),
                        ]);
                    }
                }
            }

            // Extract product data
            $products    = implode(",", $request->input('pro_id'));
            $pro_qty1    = implode(",", $request->input('pro_qty'));

            // Get the purchase return ID
            $purchase_return_id = $purchaseReturn->id;

            // Extract data from the request
            $pro_ids         = $request->input('pro_id');
            $batch_no        = $request->input('batch_no');
            $purchase_price  = $request->input('purchase_price');
            $pro_qty         = $request->input('pro_qty');
            $total           = $request->input('total');

            // Loop through the data and save each entry
            foreach ($pro_ids as $index => $pro_id) {
                $returnDetail                          = new PurchaseReturnDetail();
                $returnDetail->purchase_return_id      = $purchase_return_id;
                $returnDetail->pro_id                  = $pro_id;
                $returnDetail->batch_no                = $batch_no[$index];
                $returnDetail->purchase_price          = $purchase_price[$index];
                $returnDetail->pro_qty                 = $pro_qty[$index];
                $returnDetail->total                   = $total[$index];
                $returnDetail->save();

                // Update stock - remove returned quantity from main stock
                if ($isBeingApproved) {
                    // Reduce stock from purchase details
                    $purchaseDetail = PurchaseDetail::where('pro_id', $pro_id)
                        ->where('batch_no', $batch_no[$index])
                        ->first();

                    if ($purchaseDetail) {
                        // Increment sale_qty to reduce available stock (stock = pro_qty - sale_qty)
                        $purchaseDetail->increment('sale_qty', $pro_qty[$index]);
                    }

                    // Add to stocks table with negative stock_in_qty (reduction)
                    $stock                      = new Stock();
                    $stock->pur_return_id       = $purchase_return_id;
                    $stock->pro_id              = $pro_id;
                    $stock->stock_out_qty       = $pro_qty[$index];
                    $stock->adjustment          = 'Purchase Return';
                    $stock->invoice_date        = Carbon::parse($request->input('return_date'))->format('Y-m-d');
                    $stock->save();
                }
            }

            // Insert into Supplier Ledger table
            $ledger = new SupplierLedger();
            $ledger->supplier_id            = $request->input('supplier_id');
            $ledger->pur_return_id          = $purchase_return_id;
            $ledger->date                   = Carbon::parse($request->input('return_date'))->format('Y-m-d');
            $ledger->detail                 = $request->input('detail') ?: 'Purchase Return';
            $ledger->pro_id                 = $products;
            $ledger->pro_qty                = $pro_qty1;
            $ledger->transaction_type       = 'Pur_return';
            $ledger->bank_id                = $request->input('bank_id');
            // Purchase return: amount paid back to us (positive pay)
            $ledger->pay                    = $request->input('refund_amount') ?: 0;
            // Pending/adjustment amount
            $ledger->pending                = $request->input('adjustment_amount') ?: 0;
            $ledger->bank                   = ($request->input('payment_type') === 'Bank' ? $request->input('refund_amount') : 0);
            $ledger->save();

            // Commit the transaction
            DB::commit();

            return response()->json(['status' => 200, 'message' => 'Purchase return updated successfully.']);

        } catch (\Exception $e) {
            // Rollback on error
            DB::rollBack();
            \Log::error('Error updating purchase return: ' . $e->getMessage());
            return response()->json(['status' => 500, 'message' => 'Error updating purchase return: ' . $e->getMessage()]);
        }
    }




    public function delete_purchase_return($id)
    {
        $data = PurchaseReturn::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Purchase return not found.'
            ]);
        }

        // If approved and refund was given, reverse the refund
        if ($data->status == 1 && $data->refund_amount > 0) {
            if ($data->payment_type === "Cash") {
                DB::table('hand_cashes')->decrement('cash_amount', $data->refund_amount);

            } elseif ($data->payment_type === "Bank" && $data->bank_id) {
                DB::table('banks')->where('id', $data->bank_id)
                    ->decrement('bank_balance', $data->refund_amount);
            }
        }

        // Delete related records
        PurchaseReturnDetail::where('purchase_return_id', $data->id)->delete();
        Stock::where('purchase_return_id', $data->id)->delete();
        SupplierLedger::where('purchase_return_id', $data->id)->delete();

        // Delete related Transactions
        Transaction::where('source_type', 'Purchase Return')
            ->where('source_id', $data->id)
            ->delete();

        // Delete the purchase return
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Purchase return and its details deleted successfully.'
        ]);
    }


    public function get_supplier_products($supplier_id)
    {
        // Get all products purchased from this supplier
        $products = DB::table('purchase_details')
            ->join('purchases', 'purchase_details.purchase_id', '=', 'purchases.id')
            ->join('products', 'purchase_details.pro_id', '=', 'products.id')
            ->select(
                'products.id',
                'products.pro_name',
                'products.sku',
                'purchase_details.batch_no',
                'purchase_details.purchase_price',
                DB::raw('(purchase_details.pro_qty - COALESCE(purchase_details.sale_qty, 0)) as available_qty')
            )
            ->where('purchases.supplier_id', $supplier_id)
            ->whereRaw('(purchase_details.pro_qty - COALESCE(purchase_details.sale_qty, 0)) > 0') // Only products with stock
            ->get();

        return response()->json([
            'status' => 200,
            'products' => $products
        ]);
    }

    public function getPurchaseProduct(Request $request)
    {
        $prodid = $request->prodid;
        $supplier_id = $request->supplier_id;

        $query = DB::table('purchase_details')
            ->join('purchases', 'purchase_details.purchase_id', '=', 'purchases.id')
            ->select(
                'purchase_details.batch_no as batch',
                DB::raw('(purchase_details.pro_qty - COALESCE(purchase_details.sale_qty, 0)) as stock_qty'),
                'purchase_details.purchase_price',
                'purchase_details.sale_price'
            )
            ->where('purchase_details.pro_id', $prodid)
            ->whereRaw('(purchase_details.pro_qty - COALESCE(purchase_details.sale_qty, 0)) > 0');

        // Filter by supplier if provided
        if ($supplier_id) {
            $query->where('purchases.supplier_id', $supplier_id);
        }

        $result = $query->get();

        if ($result->count() > 0) {
            return response()->json($result);
        } else {
            return response()->json(['message' => 'No Record Found'], 404);
        }
    }
}
