<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\Investment;
use App\Models\PurchaseDetail;
use App\Models\Quotation;
use App\Models\QuotationDetail;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class QuotationController extends Controller
{

    
    public function load_quotations(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        $query = DB::table('quotations')
        ->join('customers', 'quotations.cus_id', '=', 'customers.id')
        ->join('users', 'quotations.saleman_id', '=', 'users.id')
        ->select([
            'quotations.*', 
            'customers.name as cus_name', 
            'quotations.id as id',
            'users.name as saleman_name',
        ]);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('quotations.saleman_id', Auth::id());
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('customers.name', 'like', '%' . $search . '%');
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('quotations.is_active', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $sale = $query->get();

        $all_data = [];
        foreach ($sale as $data) {


            $btn = '<td class="text-right">';
            if (Auth::user()->can('view_quotation')) {
                $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }
            if (Auth::user()->can('update_quotation')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_quotation')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            $all_data[] = [
                'id'            => $data->id,
                'invoice_date'     => Carbon::parse($data->invoice_date)->format('d-m-Y'),
                'cus_name'      => $data->cus_name,
                'saleman_name'  => $data->saleman_name,
                'subtotal'      => number_format($data->subtotal, 0),
                'dis_percent'   => number_format($data->dis_percent, 0),
                'dis_amount'    => number_format($data->dis_amount, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'paid'          => number_format($data->paid, 0),
                'due'           => number_format($data->due, 0),
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_quotation(Request $request)
    {

        $sale = new Quotation();
        $sale->invoice_date             = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
        $sale->cus_id                   = $request->input('cus_id');
        $sale->bank_id                  = $request->input('bank_id');
        $sale->Payment_type             = $request->input('payment_type');
        $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $sale->detail                   = $request->input('detail');
        $sale->subtotal                 = $request->input('subtotal');
        $sale->dis_percent              = $request->input('total_dis_per');
        $sale->dis_amount               = $request->input('total_dis_amt');
        $sale->grand_total              = $request->input('grand_total');
        $sale->paid                     = $request->input('paid');
        $sale->due                      = $request->input('due');
        $sale->created_by               = Auth::id();
        $sale->save();


        // Get the last inserted Book Test ID
        $sale_id = $sale->id;
        // Extract data from the request
        $pro_ids    = $request->input('pro_id');
        $purchase_price   = $request->input('pro_cost');
        $pro_sku   = $request->input('pro_sku');
        $batch_no   = $request->input('batch_no');

        $pro_sale   = $request->input('pro_sale');
        $pro_qty    = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt    = $request->input('dis_amt');
        $total      = $request->input('total');

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $product_stock                  = new QuotationDetail();
            $product_stock->quotation_id    = $sale_id;
            $product_stock->pro_id          = $pro_id;
            $product_stock->purchase_price      = $purchase_price[$index];
            $product_stock->pro_sku      = $pro_sku[$index];
            $product_stock->batch_no      = $batch_no[$index];

            $product_stock->sale_price      = $pro_sale[$index];
            $product_stock->dis_per         = $dis_percent[$index];
            $product_stock->dis_amount      = $dis_amt[$index];
            $product_stock->pro_qty         = $pro_qty[$index];
            $product_stock->total           = $total[$index];
            $product_stock->save();

        }



        return response()->json([
            'status'=>200,
        ]);

        
    }


    public function edit_quotation($id, Request $request)
    {
        $salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();

        // Retrieve the sale with customer details
        $quotations = DB::table('quotations')
            ->leftJoin('customers', 'quotations.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'quotations.saleman_id', '=', 'users.id')
            ->leftJoin('banks', 'quotations.bank_id', '=', 'banks.id')
            ->select(
                'quotations.*',
                'customers.name as customer_name',
                'banks.bank_name',
                'users.name as salesman_name',
                'quotations.id as id'
            )
            ->where('quotations.id', $id)
            ->first();

        if (!$quotations) {
            return response()->json([
                'status' => 404,
                'message' => 'Quotation not found'
            ], 404);
        }

        // Retrieve quotation details for the specific quotation
        $quotation_details = DB::table('quotation_details')
            ->join('products', 'quotation_details.pro_id', '=', 'products.id')
            ->select(
                'quotation_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('quotation_details.quotation_id', '=', $id)
            ->get();

        // Calculate stock for each product (filtered by selected salesman)
        foreach ($quotation_details as $detail) {
            $stock = DB::table('saleman_stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->where('salesman_id', '=', $salesman_id) // <-- Filter by selected salesman
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'sale' => $quotations,
            'products' => $quotation_details
        ]);
    }
    


    public function view_quotation($id)
    {
        $quotation = DB::table('quotations')
            ->leftJoin('customers', 'quotations.cus_id', '=', 'customers.id')
            ->select(
                '*', 'customers.name', // Select specific fields
                'quotations.id as id'
            )
            ->where('quotations.id', $id)
            ->first();

        if (!$quotation) {
            return response()->json([
                'status' => 404,
                'message' => 'Quotation not found'
            ], 404);
        }

        $all_customers = DB::table('customers')->select('id', 'name')->get();

        $quotation_details = DB::table('quotation_details')
        ->select('*')->where('quotation_id', '=', $id)
        ->get();

        $products = [];

        foreach($quotation_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();
            
            $products[] = [
                'no'            => $index+1,
                'pro_id'        => $pro->id,
                'pro_name'      => $pro->pro_name,
                'sale_price'    => $pros->sale_price,
                'pro_qty'       => $pros->pro_qty,
                'dis_per'       => $pros->dis_per,
                'dis_amount'    => $pros->dis_amount,
                'total'         => $pros->total,
            ];
        }

        return response()->json([
            'status' => 200,
            'sale' => $quotation,
            'products' => $products,
            'all_customers' => $all_customers,  
        ]);
    }

    

    public function update_quotation(Request $request)
    {
        // Find the quotation record by ID
        $quotation = Quotation::find($request->input('quotation_id'));

        if (!$quotation) {
            return response()->json(['status' => 404, 'message' => 'Quotation not found.']);
        }

        // Update the main quotation record
        $quotation->invoice_date              = Carbon::parse($request->input('quotation_date'))->format('Y-m-d');
        $quotation->cus_id                      = $request->input('cus_id');
        $quotation->Payment_type                = $request->input('payment_type');
        $quotation->bank_id                     = $request->input('bank_id');
        $quotation->status                      = $request->input('status');
        $quotation->detail                    = $request->input('detail');
        $quotation->subtotal                  = $request->input('subtotal');
        $quotation->dis_percent               = $request->input('total_dis_per');
        $quotation->dis_amount                = $request->input('total_dis_amt');
        $quotation->grand_total               = $request->input('grand_total');
        $quotation->paid                      = $request->input('paid');
        $quotation->due                       = $request->input('due');
        $quotation->saleman_id                = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $quotation->updated_by = Auth::id();
        $quotation->save();

         

        // Delete old invoice details for this invoice
        QuotationDetail::where('quotation_id', $quotation->id)->delete();
        SalemanStock::where('quotation_id', $quotation->id)->delete();
        CustomerLedger::where('quotation_id', $quotation->id)->delete();
        SalemanLedger::where('quotation_id', $quotation->id)->delete();
        DB::table('investment_profits')->where('quotation_id', $quotation->id)->delete();
        Transaction::where('source_type', 'Quotation')
        ->where('source_id', $quotation->id)
        ->delete();

        if ($request->payment_type === "Cash") {
            // hand_cash column
            DB::table('hand_cashes')->increment('cash_amount', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Cash',
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Quotation',
                'source_id'         => $quotation->id,
                'description'       => 'Quotation Payment Received',
                'created_at'        => now(),
            ]);

        } elseif ($request->payment_type === "Bank") {
            
            DB::table('banks')->where('id', $request->bank_id)
            ->increment('bank_balance', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Bank',
                'bank_id'           => $request->input('bank_id'),
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Quotation',
                'source_id'         => $quotation->id,
                'description'      => 'Quotation Payment Received',
                'created_at'      => now(),
            ]);

        }



        // Reinsert the sale details
        $pro_ids = $request->input('pro_id');
        $purchase_price = $request->input('pro_cost');
        $pro_sku = $request->input('pro_sku');
        $batch_no = $request->input('batch_no');

        $pro_sale = $request->input('pro_sale');
        $pro_qty = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt = $request->input('dis_amt');
        $total = $request->input('total');

        foreach ($pro_ids as $index => $pro_id) {
            $product_stock = new QuotationDetail();
            $product_stock->quotation_id = $quotation->id;
            $product_stock->pro_id = $pro_id; 
            $product_stock->purchase_price = $purchase_price[$index];
            $product_stock->pro_sku = $pro_sku[$index];
            $product_stock->batch_no = $batch_no[$index];

            $product_stock->sale_price = $pro_sale[$index];
            $product_stock->dis_per = $dis_percent[$index];
            $product_stock->dis_amount = $dis_amt[$index];
            $product_stock->pro_qty = $pro_qty[$index];
            $product_stock->total = $total[$index];
            $product_stock->save();

            // 🟢 2. PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
            ->where('batch_no', $batch_no[$index])
            ->first()?->increment('sale_qty', $pro_qty[$index]);

            // Update stock
            $salemanstock                  = new SalemanStock();
            $salemanstock->pro_id          = $pro_id;
            $salemanstock->quotation_id         = $quotation->id;
            $salemanstock->sale_price      = $pro_sale[$index];
            $salemanstock->purchase_price  = $purchase_price[$index];
            $salemanstock->stock_out_qty   = $pro_qty[$index];
            $salemanstock->invoice_date    = Carbon::parse($request->input('quotation_date'))->format('Y-m-d');
            $salemanstock->salesman_id       = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $salemanstock->adjustment       = 'Sale';
            $salemanstock->save();

            // Sale & Purchase
            $saleAmount = $total[$index]; // qty * sale price
            $purchaseAmount = $purchase_price[$index] * $pro_qty[$index]; // qty * purchase price

            // Actual profit
            $actualProfit = $saleAmount - $purchaseAmount;

            // Investments fetch karo
            $investments = Investment::where('pro_id', $pro_id)
                ->where('remaining_amount', '>', 0)
                ->get();

            foreach ($investments as $investment) {

                // Investor ka share jitne profit se adjust ho sakta hai
                $adjustableProfit = min($actualProfit, $investment->remaining_amount);

                if ($adjustableProfit > 0) {
                    // Investor ka profit = adjustableProfit ka % according to investor's share
                    $profit = ($adjustableProfit * $investment->profit_percentage) / 100;

                    DB::table('investment_profits')->insert([
                        'investment_id' => $investment->id,
                        'investor_id'   => $investment->investor_id,
                        'quotation_id'  => $quotation->id,
                        'pro_id'        => $pro_id,
                        'profit_amount' => $profit,
                        'created_at'    => now(),
                    ]);

                    // ✅ Investment table update karo (remaining_amount minus karo by adjustableProfit)
                    $investment->remaining_amount -= $adjustableProfit;
                    if ($investment->remaining_amount < 0) {
                        $investment->remaining_amount = 0; // negative na ho
                    }
                    $investment->save();

                    // Actual profit bhi kam karna zaroori hai
                    $actualProfit -= $adjustableProfit;
                }
            }



        }

        // Extract customer  ledger data
        $products    = implode(",", $request->input('pro_id'));
        $pro_qty1     = implode(",", $request->input('pro_qty'));
        
        // Insert into SupplierLedger table
        $supplier_ledger = new CustomerLedger();
        $supplier_ledger->cus_id                    = $request->input('cus_id');
        $supplier_ledger->quotation_id                   = $quotation->id;
        $supplier_ledger->invoice_date              = Carbon::parse($request->input('quotation_date'))->format('Y-m-d');
        $supplier_ledger->detail                    = $request->input('detail');
        $supplier_ledger->pro_id                    = $products;
        $supplier_ledger->pro_qty                   = $pro_qty1;
        $supplier_ledger->transaction_type          = $request->input('payment_type');
        $supplier_ledger->total                     = $request->input('grand_total');
        $supplier_ledger->adjustment                 = 'Sale';
        $supplier_ledger->credit                    = $request->input('paid');
        $supplier_ledger->debit                     = $request->input('due');
        $supplier_ledger->created_by                = Auth::id();
        $supplier_ledger->updated_by                = Auth::id();
        $supplier_ledger->save();

        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id                    = $request->input('salesman_id');
        $saleman_ledger->cus_id                   = $request->input('cus_id');
        $saleman_ledger->quotation_id                   = $quotation->id;
        $saleman_ledger->invoice_date              = Carbon::parse($request->input('quotation_date'))->format('Y-m-d');
        $saleman_ledger->detail                    = $request->input('detail');
        $saleman_ledger->pro_id                    = $products;
        $saleman_ledger->pro_qty                   = $pro_qty1;
        $saleman_ledger->transaction_type          = $request->input('payment_type');
        $saleman_ledger->total                     = $request->input('grand_total');
        $saleman_ledger->credit                    = $request->input('paid');
        $saleman_ledger->debit                     = $request->input('due');
        $saleman_ledger->adjustment                     = 'Sale';
        $saleman_ledger->save();

        return response()->json(['status' => 200, 'message' => 'Quotation updated successfully.']);
    }



    public function delete_quotation($id)
    {
        $data = Quotation::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Quotation not found.'
            ]);
        }

        // Reverse entry using saved data
        if ($data->Payment_type === "Cash") {
            DB::table('hand_cashes')->decrement('cash_amount', $data->paid);

        } elseif ($data->Payment_type === "Bank" && $data->bank_id) {
            DB::table('banks')->where('id', $data->bank_id)
                ->decrement('bank_balance', $data->paid);
        }

        // Delete Quotation details first
        $data->quotationDetails()->delete();
        $data->salemanStock()->delete();
        $data->customerLedger()->delete();
        $data->salemanLedger()->delete();
        $data->investorLedger()->delete();
         // 3️⃣ Related Transactions delete
        Transaction::where('source_type', 'Quotation')
            ->where('source_id', $data->id)
            ->delete();
        // Delete the Quotation
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Quotation and its details deleted successfully.'
        ]);
    }

    public function convert_quotation_to_sale(Request $request)
    {
        try {
            // First, save the quotation
            $quotation = new Quotation();
            $quotation->invoice_date = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $quotation->cus_id = $request->input('cus_id');
            $quotation->bank_id = $request->input('bank_id');
            $quotation->Payment_type = $request->input('payment_type');
            $quotation->saleman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $quotation->detail = $request->input('detail');
            $quotation->subtotal = $request->input('subtotal');
            $quotation->dis_percent = $request->input('total_dis_per');
            $quotation->dis_amount = $request->input('total_dis_amt');
            $quotation->grand_total = $request->input('grand_total');
            $quotation->paid = $request->input('paid');
            $quotation->due = $request->input('due');
            $quotation->created_by = Auth::id();
            $quotation->save();

            // Get the quotation ID
            $quotation_id = $quotation->id;

            // Now create a sale from the same data
            $sale = new Sale();
            $sale->invoice_date = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
            $sale->cus_id = $request->input('cus_id');
            $sale->bank_id = $request->input('bank_id');
            $sale->Payment_type = $request->input('payment_type');
            $sale->saleman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $sale->detail = $request->input('detail');
            $sale->subtotal = $request->input('subtotal');
            $sale->dis_percent = $request->input('total_dis_per');
            $sale->dis_amount = $request->input('total_dis_amt');
            $sale->grand_total = $request->input('grand_total');
            $sale->paid = $request->input('paid');
            $sale->due = $request->input('due');
            $sale->created_by = Auth::id();
            $sale->save();

            // Get the sale ID
            $sale_id = $sale->id;

            // Extract product data
            $pro_ids = $request->input('pro_id');
            $purchase_price = $request->input('pro_cost');
            $pro_sku = $request->input('pro_sku');
            $batch_no = $request->input('batch_no');
            $pro_sale = $request->input('pro_sale');
            $pro_qty = $request->input('pro_qty');
            $dis_percent = $request->input('dis_percent');
            $dis_amt = $request->input('dis_amt');
            $total = $request->input('total');

            // Save quotation details
            foreach ($pro_ids as $index => $pro_id) {
                $quotation_detail = new QuotationDetail();
                $quotation_detail->quotation_id = $quotation_id;
                $quotation_detail->pro_id = $pro_id;
                $quotation_detail->purchase_price = $purchase_price[$index];
                $quotation_detail->pro_sku = $pro_sku[$index];
                $quotation_detail->batch_no = $batch_no[$index];
                $quotation_detail->sale_price = $pro_sale[$index];
                $quotation_detail->dis_per = $dis_percent[$index];
                $quotation_detail->dis_amount = $dis_amt[$index];
                $quotation_detail->pro_qty = $pro_qty[$index];
                $quotation_detail->total = $total[$index];
                $quotation_detail->save();

                // Save sale details
                $sale_detail = new SaleDetail();
                $sale_detail->sale_id = $sale_id;
                $sale_detail->pro_id = $pro_id;
                $sale_detail->purchase_price = $purchase_price[$index];
                $sale_detail->pro_sku = $pro_sku[$index];
                $sale_detail->batch_no = $batch_no[$index];
                $sale_detail->sale_price = $pro_sale[$index];
                $sale_detail->dis_per = $dis_percent[$index];
                $sale_detail->dis_amount = $dis_amt[$index];
                $sale_detail->pro_qty = $pro_qty[$index];
                $sale_detail->total = $total[$index];
                $sale_detail->date = Carbon::parse($request->input('invoice_date'))->format('Y-m-d');
                $sale_detail->created_by = Auth::id();
                $sale_detail->updated_by = Auth::id();
                $sale_detail->save();

                // Update stock
                PurchaseDetail::where('pro_id', $pro_id)
                    ->where('batch_no', $batch_no[$index])
                    ->first()?->increment('sale_qty', $pro_qty[$index]);
            }

            return response()->json([
                'status' => 200,
                'message' => 'Quotation converted to sale successfully!',
                'quotation_id' => $quotation_id,
                'sale_id' => $sale_id
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => 'Error converting quotation to sale: ' . $e->getMessage()
            ]);
        }
    }


}
