<?php

namespace App\Http\Controllers;

use App\Models\ContracterLedger;
use App\Models\RepairOrder;
use App\Models\RepairOrderDetail;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class RepairOrderController extends Controller
{

    public function load_repair_order(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");


       $query = DB::table('repair_orders')
        ->leftJoin('products', 'repair_orders.pro_id', '=', 'products.id')
        ->leftJoin('contractors', 'repair_orders.contractor_id', '=', 'contractors.id')
        ->select(
            'repair_orders.*',
            'products.pro_name',
            'contractors.name as contractor_name'
        );




        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('repair_orders.repair_date', [$start_date, $end_date]);
        }

     
        if ($search = $request->input('search')) { 
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%')
                  ->orWhere('contractors.name', 'like', '%' . $search . '%');
            });
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('repair_orders.status', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $purchases = $query->get();

        $all_data = [];

        
        foreach ($purchases as $data) {

           

            $btn = '<td >';


            if (Auth::user()->can('update_order') || Auth::user()->hasRole('Production Manager')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }

            // Repair button for Admin and Production Manager
            if (Auth::user()->hasRole(['Admin', 'Super Admin', 'Production Manager'])) {
                $btn .= '<button id="repair_btn" data-repairid="'.$data->id.'" class="btn btn-success btn-icon" data-toggle="tooltip" title="Repair" style="margin-right: 3px"><i class="fa-duotone fa-wrench"></i></button>';
            }

            if (Auth::user()->can('delete_order')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            // Status badge
            if ($data->status == "Pending") {
                $status = '<div class="label label-table label-warning" style="width:100%; max-width:130px;">Pending</div>';
            } else if ($data->status == "Completed") {
                $status = '<div class="label label-table label-success" style="width:100%; max-width:130px;">Completed</div>';
            } else {
                $status = '<div class="label label-table label-info" style="width:100%; max-width:130px;">'.$data->status.'</div>';
            }

            $all_data[] = [
                'id'            => $data->id,
                'contractor_name' => $data->contractor_name ?? '',
                'pro_name'      => $data->pro_name,
                'batch_number'  => $data->batch_number ?? '',
                'price'         => number_format($data->price ?? 0, 2),
                'labour_cost'   => number_format($data->labour_cost ?? 0, 2),
                'material_cost' => number_format($data->material_cost ?? 0, 2),
                'total_price'   => number_format($data->total_price ?? 0, 2),
                'pro_qty'       => $data->pro_qty ?? 0,
                'status'        => $status,
                'repair_date'   => $data->repair_date ? Carbon::parse($data->repair_date)->format('d-m-Y') : '',

                'btn'           => $btn
            ];
        };

      

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_repair_order(Request $request)
    {
        try {
            $RepairOrder = new RepairOrder();
            $RepairOrder->contractor_id          = $request->input('contractor_id');
            $RepairOrder->repair_date            = $request->input('date') ? Carbon::parse($request->input('date'))->format('Y-m-d') : now()->format('Y-m-d');
            $RepairOrder->pro_id                 = $request->input('nonservice_pro_id');
            $RepairOrder->pro_qty                = $request->input('pro_qty');
            $RepairOrder->advance                = $request->input('advance') ?? 0;
            $RepairOrder->batch_number           = $request->input('batch_number');
            $RepairOrder->price                  = $request->input('price') ?? 0;
            $RepairOrder->labour_cost            = $request->input('labour_cost') ?? 0;
            $RepairOrder->material_cost          = $request->input('material_cost') ?? 0;
            $RepairOrder->total_price            = $request->input('total_price') ?? 0;

            $RepairOrder->save();

            // Get the last inserted BOM ID
            $repair_order_id = $RepairOrder->id;
            // Extract data from the request
            $pro_ids    = $request->input('pro_id');
            $pro_sku    = $request->input('pro_sku');

            $item_qty    = $request->input('item_qty');

            // Loop through the data and save each entry
            if ($pro_ids && is_array($pro_ids)) {
                foreach ($pro_ids as $index => $pro_id) {
                    if ($pro_id) {
                        $product_stock                      = new RepairOrderDetail();
                        $product_stock->repair_order_id             = $repair_order_id;
                        $product_stock->pro_id              = $pro_id;
                        $product_stock->pro_sku             = $pro_sku[$index] ?? '';

                        $product_stock->qty             = $item_qty[$index] ?? 0;
                        $product_stock->save();
                    }
                }
            }


            return response()->json([
                'status'=>200
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e->getMessage()
            ], 500);
        }
    }


    public function edit_repair_order($id)
    {
        // Retrieve the repair order
        $data = DB::table('repair_orders')
            ->where('id', $id)
            ->first();

        // Agar order hi na mile
        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Repair Order not found.',
            ]);
        }

        // Retrieve repair order details with product names
        $repair_order_details = DB::table('repair_order_details')
            ->join('products', 'repair_order_details.pro_id', '=', 'products.id')
            ->select(
                'repair_order_details.*',
                'products.pro_name'
            )
            ->where('repair_order_details.repair_order_id', '=', $id) // foreign key use karo
            ->get();

        return response()->json([
            'status' => 200,
            'data' => $data,
            'repair_order_details' => $repair_order_details,
        ]);
    }

    

    public function update_repair_order(Request $request)
    {
        // Find the bom record by ID
        $data = RepairOrder::find($request->input('order_id'));

        if (!$data) {
            return response()->json(['status' => 404, 'message' => 'Data not found.']);
        }

        // Update the record
        $data->contractor_id            = $request->input('contractor_id');
        $data->repair_date              = Carbon::parse($request->input('date'))->format('Y-m-d');
        $data->pro_id                   = $request->input('nonservice_pro_id');
        $data->pro_qty                  = $request->input('pro_qty');
        $data->status                   = $request->input('status');
        $data->repair_cost              = $request->input('repair_cost');
        $data->advance                  = $request->input('advance');
        $data->batch_number             = $request->input('batch_number');
        $data->price                    = $request->input('price') ?? 0;
        $data->labour_cost              = $request->input('labour_cost') ?? 0; 
        $data->material_cost            = $request->input('material_cost') ?? 0;
        $data->total_price              = $request->input('total_price') ?? 0;




        $data->save();

        // Check if status is completed, only then stock adjustments should happen
        if ($request->input('status') !== 'Completed') {
            // Remove existing details
            RepairOrderDetail::where('repair_order_id', $data->id)->delete();
            
            // Reinsert the sale details
            $pro_ids = $request->input('pro_id'); 
            $pro_sku = $request->input('pro_sku'); 
            $item_qty = $request->input('item_qty'); 

            foreach ($pro_ids as $index => $pro_id) {
                if ($pro_id) {
                    $product_stock                      = new RepairOrderDetail();
                    $product_stock->repair_order_id     = $data->id;
                    $product_stock->pro_id              = $pro_id; 
                    $product_stock->pro_sku             = $pro_sku[$index] ?? ''; 
                    $product_stock->qty                 = $item_qty[$index] ?? 0; 
                    $product_stock->save();
                }
            }
            
            return response()->json(['status' => 200, 'message' => 'Data updated successfully.']);
        }

        // Status is Completed - do stock adjustments
        $nonServiceProId = $request->input('nonservice_pro_id');
        $nonServiceProQty = $request->input('pro_qty');

        if ($nonServiceProId && $nonServiceProQty > 0) {
            // Stock OUT from non_service_stocks
            DB::table('non_service_stocks')->insert([
                'repair_id'       => $data->id,
                'pro_id'        => $nonServiceProId,
                'stock_out_qty' => $nonServiceProQty,
                'stock_in_qty'  => 0,
                'adjustment'    => 'Repair',
                'created_at'    => now(),
                'updated_at'    => now(),
            ]);

            // Stock IN to normal stocks
            DB::table('stocks')->insert([
                'repair_id'       => $data->id,
                'pro_id'       => $nonServiceProId,
                'stock_in_qty' => $nonServiceProQty,
                'adjustment'    => 'Repair',
                'created_at'   => now(),
                'updated_at'   => now(),
            ]);
        }

        // Remove existing details
        RepairOrderDetail::where('repair_order_id', $data->id)->delete();

        // Reinsert the sale details
        $pro_ids = $request->input('pro_id'); 
        $pro_sku = $request->input('pro_sku'); 

        $item_qty = $request->input('item_qty'); 

        foreach ($pro_ids as $index => $pro_id) {
            if ($pro_id) {
                $product_stock                      = new RepairOrderDetail();
                $product_stock->repair_order_id     = $data->id;
                $product_stock->pro_id              = $pro_id; 
                $product_stock->pro_sku             = $pro_sku[$index] ?? ''; 
                $product_stock->qty                 = $item_qty[$index] ?? 0; 
                $product_stock->save();

                // Stock OUT from main stocks
                if (!empty($item_qty[$index]) && $item_qty[$index] > 0) {
                    DB::table('stocks')->insert([
                        'repair_id'       => $data->id,
                        'pro_id'        => $pro_id,
                        'stock_out_qty' => $item_qty[$index],
                        'adjustment'    => 'Repair Used',
                        'created_at'    => now(),
                        'updated_at'    => now(),
                    ]);
                }
            }
        }

        // Create contracter ledger entry for payment tracking
        $contracter_ledger                      = new ContracterLedger();
        $contracter_ledger->contractor_id       = $request->input('contractor_id');
        $contracter_ledger->date                = Carbon::parse($request->input('date'))->format('Y-m-d');
        $contracter_ledger->repair_order_id     = $data->id;
        $contracter_ledger->pro_id              = $request->input('nonservice_pro_id');
        $contracter_ledger->pro_qty             = $request->input('pro_qty');
        $contracter_ledger->cost                = $request->input('repair_cost');
        $contracter_ledger->advance             = $request->input('advance');
        $contracter_ledger->status              = $request->input('status');
        $contracter_ledger->batch_number        = $request->input('batch_number');
        $contracter_ledger->price               = $request->input('price') ?? 0;
        $contracter_ledger->labour_cost         = $request->input('labour_cost') ?? 0;
        $contracter_ledger->material_cost       = $request->input('material_cost') ?? 0;
        $contracter_ledger->total_price         = $request->input('total_price') ?? 0;
        $contracter_ledger->payment_type        = 'Cash';
        $contracter_ledger->payment_status      = 'Pending';
        $contracter_ledger->created_by          = Auth::id();
        $contracter_ledger->save();

     

        return response()->json(['status' => 200, 'message' => 'Data updated successfully.']);
    }

    

    public function delete_repair_order($id)
    {
        $data = RepairOrder::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Data not found.'
            ]);
        }

        // پہلے BOMDetails delete کرو
        $data->repairOrderDetails()->delete(); 
        $data->stocks()->delete(); 
        $data->nonServiceStock()->delete(); 
        $data->contracterLedger()->delete(); 


        // پھر BOM delete کرو
        $data->delete();


        return response()->json([
            'status' => 200,
            'message' => 'Bom and its details deleted successfully.'
        ]);
    }


    public function non_service_products()
    {
        $products = DB::table('products as p')
            ->join('non_service_stocks as s', 'p.id', '=', 's.pro_id')
            ->select(
                'p.id',
                'p.pro_name',
                DB::raw('SUM(s.stock_in_qty - s.stock_out_qty) as total_qty')
            )
            ->groupBy('p.id', 'p.pro_name')
            ->havingRaw('SUM(s.stock_in_qty - s.stock_out_qty) > 0') // sirf stock > 0
            ->get();

        return response()->json($products);
    }

    public function get_production_price_by_batch($batch_number)
    {
        try {
            $production = DB::table('productions')
                ->where('batch_number', $batch_number)
                ->select('price as single_product_price')
                ->first();

            if ($production) {
                return response()->json([
                    'status' => 200,
                    'price' => $production->single_product_price
                ]);
            } else {
                return response()->json([
                    'status' => 404,
                    'message' => 'Production not found for this batch number',
                    'price' => 0
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e->getMessage(),
                'price' => 0
            ]);
        }
    }

    public function get_production_price_by_product($product_id)
    {
        try {
            // Get the latest production price for the selected product
            $production = DB::table('productions')
                ->where('pro_id', $product_id)
                ->whereNotNull('price')
                ->where('price', '>', 0)
                ->orderBy('id', 'desc')
                ->select('price as single_product_price', 'batch_number')
                ->first();

            if ($production) {
                return response()->json([
                    'status' => 200,
                    'price' => $production->single_product_price,
                    'batch_number' => $production->batch_number
                ]);
            } else {
                // Try to get from purchase_details if no production price
                $purchase = DB::table('purchase_details')
                    ->where('pro_id', $product_id)
                    ->orderBy('id', 'desc')
                    ->select('single_price as purchase_price')
                    ->first();

                if ($purchase) {
                    return response()->json([
                        'status' => 200,
                        'price' => $purchase->purchase_price,
                        'batch_number' => null
                    ]);
                }

                return response()->json([
                    'status' => 404,
                    'message' => 'Production price not found for this product',
                    'price' => 0,
                    'batch_number' => null
                ]);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e->getMessage(),
                'price' => 0
            ]);
        }
    }

    public function get_all_products_with_price()
    {
        try {
            $products = DB::table('products')
                ->leftJoin(
                    DB::raw('(SELECT pro_id, single_price as purchase_price, id FROM purchase_details pd1 WHERE id = (SELECT MAX(id) FROM purchase_details pd2 WHERE pd2.pro_id = pd1.pro_id)) as latest_purchase'),
                    'products.id',
                    '=',
                    'latest_purchase.pro_id'
                )
                ->leftJoin(
                    DB::raw('(SELECT pro_id, SUM(stock_in_qty - stock_out_qty) as total_qty FROM stocks GROUP BY pro_id) as stock_summary'),
                    'products.id',
                    '=',
                    'stock_summary.pro_id'
                )
                ->select(
                    'products.id',
                    'products.pro_name',
                    'products.sku',
                    DB::raw('COALESCE(latest_purchase.purchase_price, products.purchase_price, 0) as purchase_price'),
                    DB::raw('COALESCE(stock_summary.total_qty, 0) as available_qty')
                )
                ->get();

            return response()->json([
                'status' => 200,
                'products' => $products
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 500,
                'message' => $e->getMessage(),
                'products' => []
            ]);
        }
    }



}