<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Sale Report - View
     */
    public function sale_report()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.sales_report', compact('salesmen'));
    }

    /**
     * Sale Report - AJAX Data
     */
    public function sale_report_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');

        // Build query
        $query = DB::table('sale_details')
            ->join('sales', 'sale_details.sale_id', '=', 'sales.id')
            ->join('customers', 'sales.cus_id', '=', 'customers.id')
            ->join('products', 'sale_details.pro_id', '=', 'products.id')
            ->join('users', 'sales.saleman_id', '=', 'users.id')
            ->select([
                'sale_details.id',
                'sales.id as sale_id',
                'sales.invoice_date',
                DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                'products.pro_name',
                'sale_details.batch_no',
                'sale_details.pro_qty',
                'sale_details.purchase_price',
                'sale_details.sale_price',
                'sale_details.total as total_sale',
                'users.name as salesman_name'
            ])
            ->where('sales.is_deleted', '!=', 1);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            // Salesman can only see their own sales
            $query->where('sales.saleman_id', Auth::id());
        } else {
            // Admin can filter by specific salesman
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('sales.saleman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }

        // Get all data
        $sales = $query->orderBy('sales.invoice_date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_purchase_amount = 0;
        $total_sale_amount = 0;
        $total_profit = 0;

        $data = [];
        foreach ($sales as $index => $sale) {
            $purchase_amount = $sale->purchase_price * $sale->pro_qty;
            $profit = $sale->total_sale - $purchase_amount;

            $total_quantity += $sale->pro_qty;
            $total_purchase_amount += $purchase_amount;
            $total_sale_amount += $sale->total_sale;
            $total_profit += $profit;

            $row = [
                'sno' => $index + 1,
                'sale_id' => $sale->sale_id,
                'invoice_date' => Carbon::parse($sale->invoice_date)->format('d-m-Y'),
                'customer_name' => $sale->customer_name,
                'salesman_name' => $sale->salesman_name,
                'pro_name' => $sale->pro_name,
                'batch_no' => $sale->batch_no ?? 'N/A',
                'pro_qty' => number_format($sale->pro_qty, 0),
                'purchase_price' => number_format($sale->purchase_price, 2),
                'sale_price' => number_format($sale->sale_price, 2),
                'total_sale' => number_format($sale->total_sale, 2),
            ];

            if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                $row['profit'] = number_format($profit, 2);
            }
            
            $data[] = $row;
        }

        $totals = [
            'total_quantity' => number_format($total_quantity, 0),
            'total_purchase_amount' => number_format($total_purchase_amount, 2),
            'total_sale_amount' => number_format($total_sale_amount, 2),
        ];

        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $totals['total_profit'] = number_format($total_profit, 2);
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => $totals
        ]);
    }

    /**
     * Sale Summary - View
     */
    public function sale_summary()
    {
        return view('admin.reports.sale_summary');
    }

    /**
     * Sale Summary - AJAX Data
     */
    public function sale_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $view_type = $request->input('view_type', 'daily'); // daily or monthly

        // Product-wise sale summary
        $product_query = DB::table('sale_details')
            ->join('sales', 'sale_details.sale_id', '=', 'sales.id')
            ->join('products', 'sale_details.pro_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                DB::raw('SUM(sale_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_details.total) as total_amount'),
                DB::raw('SUM(sale_details.total - (sale_details.purchase_price * sale_details.pro_qty)) as total_profit')
            ])
            ->where('sales.is_deleted', '!=', 1)
            ->groupBy('products.id', 'products.pro_name');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $product_query->where('sales.saleman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $product_query->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }

        $product_summary = $product_query->orderBy('total_amount', 'desc')->get();

        // Date-wise summary
        $date_query = DB::table('sale_details')
            ->join('sales', 'sale_details.sale_id', '=', 'sales.id')
            ->where('sales.is_deleted', '!=', 1);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $date_query->where('sales.saleman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $date_query->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }

        if ($view_type === 'monthly') {
            $date_query->select([
                DB::raw('DATE_FORMAT(sales.invoice_date, "%Y-%m") as period'),
                DB::raw('DATE_FORMAT(sales.invoice_date, "%M %Y") as period_label'),
                DB::raw('COUNT(DISTINCT sales.id) as total_sales'),
                DB::raw('SUM(sale_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_details.total) as total_amount')
            ])
            ->groupBy('period', 'period_label')
            ->orderBy('period', 'desc');
        } else {
            $date_query->select([
                'sales.invoice_date as period',
                DB::raw('sales.invoice_date as period_label'),
                DB::raw('COUNT(DISTINCT sales.id) as total_sales'),
                DB::raw('SUM(sale_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_details.total) as total_amount')
            ])
            ->groupBy('sales.invoice_date')
            ->orderBy('sales.invoice_date', 'desc');
        }

        $date_summary = $date_query->get();

        // Format date summary
        $formatted_date_summary = $date_summary->map(function($item) use ($view_type) {
            return [
                'period' => $view_type === 'daily' ? Carbon::parse($item->period_label)->format('d-m-Y') : $item->period_label,
                'total_sales' => $item->total_sales,
                'total_qty' => number_format($item->total_qty, 0),
                'total_amount' => number_format($item->total_amount, 2)
            ];
        });

        return response()->json([
            'status' => 200,
            'product_summary' => $product_summary,
            'date_summary' => $formatted_date_summary
        ]);
    }

    /**
     * Sale Return Report - View
     */
    public function sale_return_report()
    {
        return view('admin.reports.sale_return_report');
    }

    /**
     * Sale Return Report - AJAX Data
     */
    public function sale_return_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query
        $query = DB::table('sale_return_details')
            ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id')
            ->join('customers', 'sale_returns.cus_id', '=', 'customers.id')
            ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
            ->select([
                'sale_return_details.id',
                'sale_returns.id as sale_return_id',
                'sale_returns.invoice_date',
                DB::raw('COALESCE(customers.shop_name, customers.name) as customer_shop_name'),
                'sale_return_details.batch_no',
                'products.pro_name',
                'sale_return_details.pro_qty',
                'sale_return_details.sale_return_price'
            ]);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            // Salesman can only see their own sale returns
            $query->where('sale_returns.saleman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }

        // Get all data
        $returns = $query->orderBy('sale_returns.invoice_date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_amount = 0;

        $data = [];
        foreach ($returns as $index => $return) {
            $amount = $return->sale_return_price * $return->pro_qty;

            $total_quantity += $return->pro_qty;
            $total_amount += $amount;

            $data[] = [
                'sno' => $index + 1,
                'invoice_date' => Carbon::parse($return->invoice_date)->format('d-m-Y'),
                'sale_return_id' => $return->sale_return_id,
                'customer_shop_name' => $return->customer_shop_name,
                'batch_no' => $return->batch_no ?? 'N/A',
                'pro_name' => $return->pro_name,
                'pro_qty' => number_format($return->pro_qty, 0),
                'sale_return_price' => number_format($return->sale_return_price, 2),
                'amount' => number_format($amount, 2)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_quantity' => number_format($total_quantity, 0),
                'total_amount' => number_format($total_amount, 2)
            ]
        ]);
    }

    /**
     * Sale Return Summary - View
     */
    public function sale_return_summary()
    {
        return view('admin.reports.sale_return_summary');
    }

    /**
     * Sale Return Summary - AJAX Data
     */
    public function sale_return_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $view_type = $request->input('view_type', 'daily'); // daily or monthly

        // Product-wise return summary
        $product_query = DB::table('sale_return_details')
            ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id')
            ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
            ])
            ->groupBy('products.id', 'products.pro_name');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $product_query->where('sale_returns.saleman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $product_query->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }

        $product_summary = $product_query->orderBy('total_amount', 'desc')->get();

        // Date-wise summary
        $date_query = DB::table('sale_return_details')
            ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $date_query->where('sale_returns.saleman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $date_query->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }

        if ($view_type === 'monthly') {
            $date_query->select([
                DB::raw('DATE_FORMAT(sale_returns.invoice_date, "%Y-%m") as period'),
                DB::raw('DATE_FORMAT(sale_returns.invoice_date, "%M %Y") as period_label'),
                DB::raw('COUNT(DISTINCT sale_returns.id) as total_returns'),
                DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
            ])
            ->groupBy('period', 'period_label')
            ->orderBy('period', 'desc');
        } else {
            $date_query->select([
                'sale_returns.invoice_date as period',
                DB::raw('sale_returns.invoice_date as period_label'),
                DB::raw('COUNT(DISTINCT sale_returns.id) as total_returns'),
                DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
            ])
            ->groupBy('sale_returns.invoice_date')
            ->orderBy('sale_returns.invoice_date', 'desc');
        }

        $date_summary = $date_query->get();

        // Format date summary
        $formatted_date_summary = $date_summary->map(function($item) use ($view_type) {
            return [
                'period' => $view_type === 'daily' ? Carbon::parse($item->period_label)->format('d-m-Y') : $item->period_label,
                'total_returns' => $item->total_returns,
                'total_qty' => number_format($item->total_qty, 0),
                'total_amount' => number_format($item->total_amount, 2)
            ];
        });

        return response()->json([
            'status' => 200,
            'product_summary' => $product_summary,
            'date_summary' => $formatted_date_summary
        ]);
    }

    /**
     * Salesman Order Report - View
     */
    public function salesman_order_report()
    {
        return view('admin.reports.salesman_order_report');
    }

    /**
     * Salesman Order Report - AJAX Data
     */
    public function salesman_order_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $status_filter = $request->input('status_filter');

        // Build query
        $query = DB::table('order_details')
            ->join('orders', 'order_details.order_id', '=', 'orders.id')
            ->join('products', 'order_details.product_id', '=', 'products.id')
            ->select([
                'order_details.id',
                'orders.id as order_id',
                'orders.created_at as order_date',
                'products.pro_name',
                'order_details.quantity',
                DB::raw('COALESCE(orders.status, "pending") as status')
            ]);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            // Salesman can only see their own orders
            $query->where('orders.salesman_id', Auth::id());
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(orders.created_at)'), [$start_date, $end_date]);
        }

        // Status filter
        if (!empty($status_filter) && $status_filter != 'all') {
            $query->where('orders.status', $status_filter);
        }

        // Get all data
        $orders = $query->orderBy('orders.created_at', 'desc')->get();

        // Calculate totals
        $total_orders = $orders->unique('order_id')->count();
        $total_quantity = 0;
        $status_counts = [
            'pending' => 0,
            'completed' => 0,
            'cancelled' => 0
        ];

        $data = [];
        foreach ($orders as $index => $order) {
            $total_quantity += $order->quantity;
            $status_counts[$order->status] = ($status_counts[$order->status] ?? 0) + 1;

            $data[] = [
                'sno' => $index + 1,
                'order_date' => Carbon::parse($order->order_date)->format('d-m-Y'),
                'order_id' => $order->order_id,
                'pro_name' => $order->pro_name,
                'quantity' => number_format($order->quantity, 0),
                'status' => ucfirst($order->status),
                'status_badge' => $this->getStatusBadge($order->status)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_orders' => $total_orders,
                'total_quantity' => number_format($total_quantity, 0),
                'pending_count' => $status_counts['pending'],
                'completed_count' => $status_counts['completed'],
                'cancelled_count' => $status_counts['cancelled']
            ]
        ]);
    }

    /**
     * Get status badge HTML
     */
    private function getStatusBadge($status)
    {
        switch ($status) {
            case 'completed':
                return '<span class="label label-success">Completed</span>';
            case 'cancelled':
                return '<span class="label label-danger">Cancelled</span>';
            default:
                return '<span class="label label-warning">Pending</span>';
        }
    }

    /**
     * Salesman Order Summary - View
     */
    public function salesman_order_summary()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.salesman_order_summary', compact('salesmen'));
    }

    /**
     * Salesman Order Summary - AJAX Data
     */
    public function salesman_order_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $view_type = $request->input('view_type', 'daily');
        $salesman_id = $request->input('salesman_id');

        // Product-wise summary
        $product_query = DB::table('order_booker_details')
            ->join('order_bookers', 'order_booker_details.order_booker_id', '=', 'order_bookers.id')
            ->join('products', 'order_booker_details.pro_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                DB::raw('SUM(order_booker_details.pro_qty) as total_qty'),
                DB::raw('SUM(order_booker_details.total) as total_amount'),
                DB::raw('COUNT(DISTINCT order_bookers.id) as total_orders')
            ])
            ->groupBy('products.id', 'products.pro_name');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $product_query->where('order_bookers.saleman_id', Auth::id());
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $product_query->where('order_bookers.saleman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $product_query->whereBetween('order_bookers.invoice_date', [$start_date, $end_date]);
        }

        $product_summary = $product_query->orderBy('total_amount', 'desc')->get();

        // Date-wise summary
        $date_query = DB::table('order_booker_details')
            ->join('order_bookers', 'order_booker_details.order_booker_id', '=', 'order_bookers.id');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $date_query->where('order_bookers.saleman_id', Auth::id());
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $date_query->where('order_bookers.saleman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $date_query->whereBetween('order_bookers.invoice_date', [$start_date, $end_date]);
        }

        if ($view_type === 'monthly') {
            $date_query->select([
                DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m") as period'),
                DB::raw('COUNT(DISTINCT order_bookers.id) as total_orders'),
                DB::raw('SUM(order_booker_details.pro_qty) as total_qty'),
                DB::raw('SUM(order_booker_details.total) as total_amount')
            ])
            ->groupBy(DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m")'))
            ->orderBy(DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m")'), 'desc');
        } else {
            $date_query->select([
                DB::raw('DATE(order_bookers.invoice_date) as period'),
                DB::raw('COUNT(DISTINCT order_bookers.id) as total_orders'),
                DB::raw('SUM(order_booker_details.pro_qty) as total_qty'),
                DB::raw('SUM(order_booker_details.total) as total_amount')
            ])
            ->groupBy(DB::raw('DATE(order_bookers.invoice_date)'))
            ->orderBy(DB::raw('DATE(order_bookers.invoice_date)'), 'desc');
        }

        $date_summary = $date_query->get();

        // Format date summary
        $formatted_date_summary = $date_summary->map(function($item) use ($view_type) {
            // Format the period for display
            $formatted_period = $item->period;
            if ($view_type === 'daily') {
                $formatted_period = Carbon::createFromFormat('Y-m-d', $item->period)->format('d-m-Y');
            } else {
                $formatted_period = Carbon::createFromFormat('Y-m', $item->period)->format('F Y');
            }

            return [
                'period' => $formatted_period,
                'total_orders' => $item->total_orders,
                'total_qty' => number_format($item->total_qty, 0),
                'total_amount' => number_format($item->total_amount, 2)
            ];
        });

        return response()->json([
            'status' => 200,
            'product_summary' => $product_summary,
            'date_summary' => $formatted_date_summary
        ]);
    }

    /**
     * Salesman Order Return Report - View
     */
    public function salesman_order_return_report()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.salesman_order_return_report', compact('salesmen'));
    }

    /**
     * Salesman Order Return Report - AJAX Data
     */
    public function salesman_order_return_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');

        // Build query
        $query = DB::table('order_return_details')
            ->join('order_returns', 'order_return_details.order_return_id', '=', 'order_returns.id')
            ->join('users', 'order_returns.salesman_id', '=', 'users.id')
            ->leftJoin('customers', 'order_returns.customer_id', '=', 'customers.id')
            ->join('products', 'order_return_details.product_id', '=', 'products.id')
            ->select([
                'order_return_details.id',
                'order_returns.id as return_id',
                'order_returns.created_at as return_date',
                'users.name as salesman_name',
                DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                'products.pro_name',
                'order_return_details.quantity',
                'order_return_details.price',
                'order_returns.status',
                'order_returns.reason'
            ]);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('order_returns.salesman_id', Auth::id());
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('order_returns.salesman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween(DB::raw('DATE(order_returns.created_at)'), [$start_date, $end_date]);
        }

        // Get all data
        $returns = $query->orderBy('order_returns.created_at', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_amount = 0;

        $data = [];
        foreach ($returns as $index => $return) {
            $amount = $return->price * $return->quantity;

            $total_quantity += $return->quantity;
            $total_amount += $amount;

            $data[] = [
                'sno' => $index + 1,
                'return_date' => Carbon::parse($return->return_date)->format('d-m-Y'),
                'return_id' => $return->return_id,
                'salesman_name' => $return->salesman_name,
                'customer_name' => $return->customer_name ?? 'N/A',
                'product_name' => $return->pro_name,
                'quantity' => number_format($return->quantity, 0),
                'price' => number_format($return->price, 2),
                'total_amount' => number_format($amount, 2),
                'status' => ucfirst($return->status),
                'reason' => $return->reason ?? 'N/A'
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_quantity' => number_format($total_quantity, 0),
                'total_amount' => number_format($total_amount, 2)
            ]
        ]);
    }

    /**
     * Salesman Order Return Summary - View
     */
    public function salesman_order_return_summary()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.salesman_order_return_summary', compact('salesmen'));
    }

    /**
     * Salesman Order Return Summary - AJAX Data
     */
    public function salesman_order_return_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $view_type = $request->input('view_type', 'daily');
        $salesman_id = $request->input('salesman_id');

        // Product-wise return summary
        $product_query = DB::table('order_return_details')
            ->join('order_returns', 'order_return_details.order_return_id', '=', 'order_returns.id')
            ->join('products', 'order_return_details.product_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                DB::raw('SUM(order_return_details.quantity) as total_qty'),
                DB::raw('COUNT(DISTINCT order_returns.id) as total_returns')
            ])
            ->groupBy('products.id', 'products.pro_name');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $product_query->where('order_returns.salesman_id', Auth::id());
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $product_query->where('order_returns.salesman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $product_query->whereBetween(DB::raw('DATE(order_returns.created_at)'), [$start_date, $end_date]);
        }

        $product_summary = $product_query->orderBy('total_qty', 'desc')->get();

        // Date-wise summary
        $date_query = DB::table('order_return_details')
            ->join('order_returns', 'order_return_details.order_return_id', '=', 'order_returns.id');

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $date_query->where('order_returns.salesman_id', Auth::id());
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $date_query->where('order_returns.salesman_id', $salesman_id);
            }
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $date_query->whereBetween(DB::raw('DATE(order_returns.created_at)'), [$start_date, $end_date]);
        }

        if ($view_type === 'monthly') {
            $date_query->select([
                DB::raw('DATE_FORMAT(order_returns.created_at, "%Y-%m") as period'),
                DB::raw('COUNT(DISTINCT order_returns.id) as total_returns'),
                DB::raw('SUM(order_return_details.quantity) as total_qty')
            ])
            ->groupBy(DB::raw('DATE_FORMAT(order_returns.created_at, "%Y-%m")'))
            ->orderBy(DB::raw('DATE_FORMAT(order_returns.created_at, "%Y-%m")'), 'desc');
        } else {
            $date_query->select([
                DB::raw('DATE(order_returns.created_at) as period'),
                DB::raw('COUNT(DISTINCT order_returns.id) as total_returns'),
                DB::raw('SUM(order_return_details.quantity) as total_qty')
            ])
            ->groupBy(DB::raw('DATE(order_returns.created_at)'))
            ->orderBy(DB::raw('DATE(order_returns.created_at)'), 'desc');
        }

        $date_summary = $date_query->get();

        // Format date summary
        $formatted_date_summary = $date_summary->map(function($item) use ($view_type) {
            // Format the period for display
            $formatted_period = $item->period;
            if ($view_type === 'daily') {
                $formatted_period = Carbon::createFromFormat('Y-m-d', $item->period)->format('d-m-Y');
            } else {
                $formatted_period = Carbon::createFromFormat('Y-m', $item->period)->format('F Y');
            }

            return [
                'period' => $formatted_period,
                'total_returns' => $item->total_returns,
                'total_qty' => number_format($item->total_qty, 0)
            ];
        });

        return response()->json([
            'status' => 200,
            'product_summary' => $product_summary,
            'date_summary' => $formatted_date_summary
        ]);
    }

    /**
     * Order Booker Report - View
     */
    public function order_booker_report()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.order_booker_report', compact('salesmen'));
    }

    /**
     * Order Booker Report - AJAX Data
     */
    public function order_booker_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $status_filter = $request->input('status_filter');
        $view_type = $request->input('view_type', 'detailed');

        // Build query for detailed view
        if ($view_type === 'detailed') {
            $query = DB::table('order_bookers')
                ->join('users', 'order_bookers.saleman_id', '=', 'users.id')
                ->join('customers', 'order_bookers.cus_id', '=', 'customers.id')
                ->select([
                    'order_bookers.id',
                    'order_bookers.invoice_date',
                    'users.name as salesman_name',
                    DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                    'order_bookers.subtotal',
                    'order_bookers.dis_amount',
                    'order_bookers.grand_total',
                    'order_bookers.paid',
                    'order_bookers.due',
                    'order_bookers.Payment_type',
                    'order_bookers.status'
                ]);

            // Role-based filter
            if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                $query->where('order_bookers.saleman_id', Auth::id());
            } else {
                if (!empty($salesman_id) && $salesman_id != 'all') {
                    $query->where('order_bookers.saleman_id', $salesman_id);
                }
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('order_bookers.invoice_date', [$start_date, $end_date]);
            }

            // Status filter
            if (!empty($status_filter) && $status_filter != 'all') {
                $query->where('order_bookers.status', $status_filter);
            }

            // Get all data
            $orders = $query->orderBy('order_bookers.invoice_date', 'desc')->get();

            // Calculate totals
            $total_subtotal = 0;
            $total_discount = 0;
            $total_grand = 0;
            $total_paid = 0;
            $total_due = 0;

            $data = [];
            foreach ($orders as $index => $order) {
                $total_subtotal += $order->subtotal;
                $total_discount += $order->dis_amount;
                $total_grand += $order->grand_total;
                $total_paid += $order->paid;
                $total_due += $order->due;

                $data[] = [
                    'sno' => $index + 1,
                    'order_date' => Carbon::parse($order->invoice_date)->format('d-m-Y'),
                    'order_id' => $order->id,
                    'salesman_name' => $order->salesman_name,
                    'customer_name' => $order->customer_name,
                    'subtotal' => number_format($order->subtotal, 2),
                    'discount' => number_format($order->dis_amount, 2),
                    'grand_total' => number_format($order->grand_total, 2),
                    'paid' => number_format($order->paid, 2),
                    'due' => number_format($order->due, 2),
                    'payment_type' => $order->Payment_type ?? 'N/A',
                    'status' => ucfirst($order->status),
                    'status_badge' => $this->getStatusBadge(strtolower($order->status))
                ];
            }

            return response()->json([
                'status' => 200,
                'data' => $data,
                'totals' => [
                    'total_records' => count($data),
                    'total_subtotal' => number_format($total_subtotal, 2),
                    'total_discount' => number_format($total_discount, 2),
                    'total_grand' => number_format($total_grand, 2),
                    'total_paid' => number_format($total_paid, 2),
                    'total_due' => number_format($total_due, 2)
                ]
            ]);
        } else {
            // Summary view (daily or monthly)
            $summary_query = DB::table('order_bookers');

            // Role-based filter
            if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                $summary_query->where('order_bookers.saleman_id', Auth::id());
            } else {
                if (!empty($salesman_id) && $salesman_id != 'all') {
                    $summary_query->where('order_bookers.saleman_id', $salesman_id);
                }
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $summary_query->whereBetween('order_bookers.invoice_date', [$start_date, $end_date]);
            }

            // Status filter
            if (!empty($status_filter) && $status_filter != 'all') {
                $summary_query->where('order_bookers.status', $status_filter);
            }

            if ($view_type === 'monthly') {
                $summary_query->select([
                    DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m") as period'),
                    DB::raw('COUNT(*) as total_orders'),
                    DB::raw('SUM(order_bookers.subtotal) as total_subtotal'),
                    DB::raw('SUM(order_bookers.dis_amount) as total_discount'),
                    DB::raw('SUM(order_bookers.grand_total) as total_grand'),
                    DB::raw('SUM(order_bookers.paid) as total_paid'),
                    DB::raw('SUM(order_bookers.due) as total_due')
                ])
                ->groupBy(DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m")'))
                ->orderBy(DB::raw('DATE_FORMAT(order_bookers.invoice_date, "%Y-%m")'), 'desc');
            } else {
                $summary_query->select([
                    DB::raw('DATE(order_bookers.invoice_date) as period'),
                    DB::raw('COUNT(*) as total_orders'),
                    DB::raw('SUM(order_bookers.subtotal) as total_subtotal'),
                    DB::raw('SUM(order_bookers.dis_amount) as total_discount'),
                    DB::raw('SUM(order_bookers.grand_total) as total_grand'),
                    DB::raw('SUM(order_bookers.paid) as total_paid'),
                    DB::raw('SUM(order_bookers.due) as total_due')
                ])
                ->groupBy(DB::raw('DATE(order_bookers.invoice_date)'))
                ->orderBy(DB::raw('DATE(order_bookers.invoice_date)'), 'desc');
            }

            $summary_data = $summary_query->get();

            // Format summary
            $formatted_summary = $summary_data->map(function($item) use ($view_type) {
                // Format the period for display
                $formatted_period = $item->period;
                if ($view_type === 'daily') {
                    $formatted_period = Carbon::createFromFormat('Y-m-d', $item->period)->format('d-m-Y');
                } else {
                    $formatted_period = Carbon::createFromFormat('Y-m', $item->period)->format('F Y');
                }

                return [
                    'period' => $formatted_period,
                    'total_orders' => $item->total_orders,
                    'subtotal' => number_format($item->total_subtotal, 2),
                    'discount' => number_format($item->total_discount, 2),
                    'grand_total' => number_format($item->total_grand, 2),
                    'paid' => number_format($item->total_paid, 2),
                    'due' => number_format($item->total_due, 2)
                ];
            });

            return response()->json([
                'status' => 200,
                'data' => $formatted_summary,
                'totals' => [
                    'total_records' => count($formatted_summary),
                    'total_subtotal' => number_format($summary_data->sum('total_subtotal'), 2),
                    'total_discount' => number_format($summary_data->sum('total_discount'), 2),
                    'total_grand' => number_format($summary_data->sum('total_grand'), 2),
                    'total_paid' => number_format($summary_data->sum('total_paid'), 2),
                    'total_due' => number_format($summary_data->sum('total_due'), 2)
                ]
            ]);
        }
    }

    /**
     * Salesman Ledger Report - View
     */
    public function salesman_ledger_report()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.salesman_ledger_report', compact('salesmen'));
    }

    /**
     * Salesman Ledger Report - AJAX Data
     */
    public function salesman_ledger_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $transaction_type = $request->input('transaction_type');

        // Determine salesman to query
        $target_salesman_id = null;
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $target_salesman_id = Auth::id();
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $target_salesman_id = $salesman_id;
            }
        }

        // Build query
        $query = DB::table('saleman_ledgers')
            ->leftJoin('customers', 'saleman_ledgers.cus_id', '=', 'customers.id')
            ->leftJoin('products', 'saleman_ledgers.pro_id', '=', 'products.id')
            ->leftJoin('banks', 'saleman_ledgers.bank_id', '=', 'banks.id')
            ->select([
                'saleman_ledgers.id',
                'saleman_ledgers.invoice_date',
                'saleman_ledgers.transaction_type',
                'saleman_ledgers.detail',
                DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                'products.pro_name',
                'saleman_ledgers.pro_qty',
                'saleman_ledgers.debit',
                'saleman_ledgers.credit',
                'saleman_ledgers.bank',
                'saleman_ledgers.saleman_cash',
                'banks.bank_name',
                'saleman_ledgers.status'
            ])
            ->whereNull('saleman_ledgers.deleted_at');

        // Salesman filter
        if ($target_salesman_id) {
            $query->where('saleman_ledgers.salesman_id', $target_salesman_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
        }

        // Transaction type filter
        if (!empty($transaction_type) && $transaction_type != 'all') {
            $query->where('saleman_ledgers.transaction_type', $transaction_type);
        }

        // Get all data ordered by date
        $ledgers = $query->orderBy('saleman_ledgers.invoice_date', 'desc')
                         ->orderBy('saleman_ledgers.id', 'desc')
                         ->get();

        // Calculate totals and running balance
        $total_debit = 0;
        $total_credit = 0;
        $running_balance = 0;

        $data = [];
        foreach ($ledgers as $index => $ledger) {
            $total_debit += $ledger->debit ?? 0;
            $total_credit += $ledger->credit ?? 0;
            $running_balance = $total_credit - $total_debit;

            $data[] = [
                'sno' => $index + 1,
                'transaction_date' => Carbon::parse($ledger->invoice_date)->format('d-m-Y'),
                'transaction_type' => str_replace('_', ' ', ucfirst($ledger->transaction_type)),
                'detail' => $ledger->detail ?? 'N/A',
                'customer' => $ledger->customer_name ?? 'N/A',
                'product' => $ledger->pro_name ?? 'N/A',
                'qty' => $ledger->pro_qty ? number_format($ledger->pro_qty, 0) : '-',
                'debit' => number_format($ledger->debit ?? 0, 2),
                'credit' => number_format($ledger->credit ?? 0, 2),
                'bank_amount' => number_format($ledger->bank ?? 0, 2),
                'cash_amount' => number_format($ledger->saleman_cash ?? 0, 2),
                'bank_name' => $ledger->bank_name ?? 'N/A',
                'status' => $ledger->status ?? 'N/A',
                'balance' => number_format($running_balance, 2)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_records' => count($data),
                'total_debit' => number_format($total_debit, 2),
                'total_credit' => number_format($total_credit, 2),
                'final_balance' => number_format($running_balance, 2)
            ]
        ]);
    }

    /**
     * Salesman Expense Report - View
     */
    public function salesman_expense_report()
    {
        // Get all salesmen for filter dropdown (Admin only)
        $salesmen = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $salesmen = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Sales Man')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.salesman_expense_report', compact('salesmen'));
    }

    /**
     * Salesman Expense Report - AJAX Data
     */
    public function salesman_expense_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $view_type = $request->input('view_type', 'detailed');

        // Determine salesman to query
        $target_salesman_id = null;
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $target_salesman_id = Auth::id();
        } else {
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $target_salesman_id = $salesman_id;
            }
        }

        if ($view_type === 'detailed') {
            // Detailed expense view
            $query = DB::table('saleman_ledgers')
                ->leftJoin('banks', 'saleman_ledgers.bank_id', '=', 'banks.id')
                ->select([
                    'saleman_ledgers.id',
                    'saleman_ledgers.invoice_date',
                    'saleman_ledgers.detail',
                    'saleman_ledgers.reason',
                    'saleman_ledgers.debit as expense_amount',
                    'saleman_ledgers.bank',
                    'saleman_ledgers.saleman_cash',
                    'banks.bank_name',
                    'saleman_ledgers.status'
                ])
                ->where('saleman_ledgers.transaction_type', 'expense')
                ->whereNull('saleman_ledgers.deleted_at');

            // Salesman filter
            if ($target_salesman_id) {
                $query->where('saleman_ledgers.salesman_id', $target_salesman_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $expenses = $query->orderBy('saleman_ledgers.invoice_date', 'desc')->get();

            // Calculate totals
            $total_amount = 0;
            $total_bank = 0;
            $total_cash = 0;

            $data = [];
            foreach ($expenses as $index => $expense) {
                $total_amount += $expense->expense_amount ?? 0;
                $total_bank += $expense->bank ?? 0;
                $total_cash += $expense->saleman_cash ?? 0;

                $data[] = [
                    'sno' => $index + 1,
                    'invoice_date' => Carbon::parse($expense->invoice_date)->format('d-m-Y'),
                    'detail' => $expense->detail ?? 'N/A',
                    'reason' => $expense->reason ?? 'N/A',
                    'expense_amount' => number_format($expense->expense_amount ?? 0, 2),
                    'bank' => number_format($expense->bank ?? 0, 2),
                    'cash' => number_format($expense->saleman_cash ?? 0, 2),
                    'bank_name' => $expense->bank_name ?? 'N/A',
                    'status' => $expense->status ?? 'N/A'
                ];
            }

            return response()->json([
                'status' => 200,
                'data' => $data,
                'totals' => [
                    'total_expenses' => count($data),
                    'total_amount' => number_format($total_amount, 2),
                    'total_bank' => number_format($total_bank, 2),
                    'total_cash' => number_format($total_cash, 2)
                ]
            ]);
        } else {
            // Summary view (daily or monthly)
            $summary_query = DB::table('saleman_ledgers')
                ->where('transaction_type', 'expense')
                ->whereNull('deleted_at');

            // Salesman filter
            if ($target_salesman_id) {
                $summary_query->where('salesman_id', $target_salesman_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $summary_query->whereBetween('invoice_date', [$start_date, $end_date]);
            }

            if ($view_type === 'monthly') {
                $summary_query->select([
                    DB::raw('DATE_FORMAT(invoice_date, "%Y-%m") as period'),
                    DB::raw('DATE_FORMAT(invoice_date, "%M %Y") as period_label'),
                    DB::raw('COUNT(*) as total_expenses'),
                    DB::raw('SUM(debit) as total_amount'),
                    DB::raw('SUM(bank) as total_bank'),
                    DB::raw('SUM(saleman_cash) as total_cash')
                ])
                ->groupBy('period', 'period_label')
                ->orderBy('period', 'desc');
            } else {
                $summary_query->select([
                    'invoice_date as period',
                    DB::raw('invoice_date as period_label'),
                    DB::raw('COUNT(*) as total_expenses'),
                    DB::raw('SUM(debit) as total_amount'),
                    DB::raw('SUM(bank) as total_bank'),
                    DB::raw('SUM(saleman_cash) as total_cash')
                ])
                ->groupBy('invoice_date')
                ->orderBy('invoice_date', 'desc');
            }

            $summary_data = $summary_query->get();

            // Format summary
            $formatted_summary = $summary_data->map(function($item) use ($view_type) {
                return [
                    'period' => $view_type === 'daily' ? Carbon::parse($item->period_label)->format('d-m-Y') : $item->period_label,
                    'total_expenses' => $item->total_expenses,
                    'total_amount' => number_format($item->total_amount, 2),
                    'total_bank' => number_format($item->total_bank, 2),
                    'total_cash' => number_format($item->total_cash, 2)
                ];
            });

            return response()->json([
                'status' => 200,
                'summary' => $formatted_summary
            ]);
        }
    }

    /**
     * Order Booker Cashier Report - View
     */
    public function order_booker_cashier_report()
    {
        // Get all order bookers for filter dropdown (Admin only)
        $order_bookers = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $order_bookers = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Order Booker')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.order_booker_cashier_report', compact('order_bookers'));
    }

    /**
     * Order Booker Cashier Report - AJAX Data
     */
    public function order_booker_cashier_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $order_booker_id = $request->input('order_booker_id');
        $view_type = $request->input('view_type', 'detailed'); // detailed, daily, monthly

        if ($view_type === 'detailed') {
            // Detailed cashier transactions
            $query = DB::table('order_booker_ledgers')
                ->join('users as ob', 'order_booker_ledgers.order_booker_id', '=', 'ob.id')
                ->leftJoin('banks', 'order_booker_ledgers.bank_id', '=', 'banks.id')
                ->select([
                    'order_booker_ledgers.id',
                    'order_booker_ledgers.invoice_date',
                    'ob.name as order_booker_name',
                    'order_booker_ledgers.transaction_type',
                    'order_booker_ledgers.detail',
                    'order_booker_ledgers.paid_amount',
                    'order_booker_ledgers.expense_amount',
                    'banks.bank_name',
                    'order_booker_ledgers.status'
                ])
                ->whereIn('order_booker_ledgers.transaction_type', ['cash_received', 'payment'])
                ->whereNull('order_booker_ledgers.deleted_at');

            // Filter by order booker (Admin only)
            if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                if (!empty($order_booker_id) && $order_booker_id != 'all') {
                    $query->where('order_booker_ledgers.order_booker_id', $order_booker_id);
                }
            } else {
                $query->where('order_booker_ledgers.order_booker_id', Auth::id());
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('order_booker_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $transactions = $query->orderBy('order_booker_ledgers.invoice_date', 'desc')->get();

            // Calculate totals
            $total_cash_received = 0;
            $total_expenses = 0;

            $data = [];
            foreach ($transactions as $index => $transaction) {
                if ($transaction->transaction_type === 'cash_received' || $transaction->transaction_type === 'payment') {
                    $total_cash_received += $transaction->paid_amount ?? 0;
                }
                $total_expenses += $transaction->expense_amount ?? 0;

                $data[] = [
                    'sno' => $index + 1,
                    'invoice_date' => Carbon::parse($transaction->invoice_date)->format('d-m-Y'),
                    'order_booker_name' => $transaction->order_booker_name,
                    'transaction_type' => ucfirst(str_replace('_', ' ', $transaction->transaction_type)),
                    'detail' => $transaction->detail ?? 'N/A',
                    'cash_received' => number_format($transaction->paid_amount ?? 0, 2),
                    'expense' => number_format($transaction->expense_amount ?? 0, 2),
                    'bank_name' => $transaction->bank_name ?? 'N/A',
                    'status' => ucfirst($transaction->status)
                ];
            }

            return response()->json([
                'status' => 200,
                'data' => $data,
                'totals' => [
                    'total_records' => count($data),
                    'total_cash_received' => number_format($total_cash_received, 2),
                    'total_expenses' => number_format($total_expenses, 2),
                    'net_amount' => number_format($total_cash_received - $total_expenses, 2)
                ]
            ]);
        } else {
            // Summary view (daily or monthly)
            $summary_query = DB::table('order_booker_ledgers')
                ->whereIn('transaction_type', ['cash_received', 'payment'])
                ->whereNull('deleted_at');

            // Filter by order booker
            if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                if (!empty($order_booker_id) && $order_booker_id != 'all') {
                    $summary_query->where('order_booker_id', $order_booker_id);
                }
            } else {
                $summary_query->where('order_booker_id', Auth::id());
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $summary_query->whereBetween('invoice_date', [$start_date, $end_date]);
            }

            if ($view_type === 'monthly') {
                $summary_query->select([
                    DB::raw('DATE_FORMAT(invoice_date, "%Y-%m") as period'),
                    DB::raw('DATE_FORMAT(invoice_date, "%M %Y") as period_label'),
                    DB::raw('COUNT(*) as total_transactions'),
                    DB::raw('SUM(paid_amount) as total_cash_received'),
                    DB::raw('SUM(expense_amount) as total_expenses')
                ])
                ->groupBy('period', 'period_label')
                ->orderBy('period', 'desc');
            } else {
                $summary_query->select([
                    'invoice_date as period',
                    DB::raw('invoice_date as period_label'),
                    DB::raw('COUNT(*) as total_transactions'),
                    DB::raw('SUM(paid_amount) as total_cash_received'),
                    DB::raw('SUM(expense_amount) as total_expenses')
                ])
                ->groupBy('invoice_date')
                ->orderBy('invoice_date', 'desc');
            }

            $summary_data = $summary_query->get();

            // Format summary
            $formatted_summary = $summary_data->map(function($item) use ($view_type) {
                $net_amount = $item->total_cash_received - $item->total_expenses;
                return [
                    'period' => $view_type === 'daily' ? Carbon::parse($item->period_label)->format('d-m-Y') : $item->period_label,
                    'total_transactions' => $item->total_transactions,
                    'total_cash_received' => number_format($item->total_cash_received, 2),
                    'total_expenses' => number_format($item->total_expenses, 2),
                    'net_amount' => number_format($net_amount, 2)
                ];
            });

            return response()->json([
                'status' => 200,
                'summary' => $formatted_summary
            ]);
        }
    }

    /**
     * Order Booker Order Report - View
     */
    public function order_booker_order_report()
    {
        // Get all order bookers for filter dropdown (Admin only)
        $order_bookers = [];
        if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $order_bookers = DB::table('users')
                ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
                ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
                ->where('roles.name', 'Order Booker')
                ->select('users.id', 'users.name')
                ->get();
        }

        return view('admin.reports.order_booker_order_report', compact('order_bookers'));
    }

    /**
     * Order Booker Order Report - AJAX Data
     */
    public function order_booker_order_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $order_booker_id = $request->input('order_booker_id');
        $view_type = $request->input('view_type', 'detailed'); // detailed, daily, monthly

        if ($view_type === 'detailed') {
            // Detailed order view
            $query = DB::table('order_bookers')
                ->join('users as ob', 'order_bookers.saleman_id', '=', 'ob.id')
                ->join('customers', 'order_bookers.cus_id', '=', 'customers.id')
                ->select([
                    'order_bookers.id',
                    'order_bookers.invoice_date',
                    'ob.name as order_booker_name',
                    DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                    'order_bookers.subtotal',
                    'order_bookers.dis_amount',
                    'order_bookers.grand_total',
                    'order_bookers.paid',
                    'order_bookers.due',
                    'order_bookers.status'
                ])
                ->where('order_bookers.is_active', 1);

            // Filter by order booker (Admin only)
            if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                if (!empty($order_booker_id) && $order_booker_id != 'all') {
                    $query->where('order_bookers.saleman_id', $order_booker_id);
                }
            } else {
                $query->where('order_bookers.saleman_id', Auth::id());
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('order_bookers.invoice_date', [$start_date, $end_date]);
            }

            $orders = $query->orderBy('order_bookers.invoice_date', 'desc')->get();

            // Calculate totals
            $total_orders = count($orders);
            $total_subtotal = 0;
            $total_discount = 0;
            $total_grand = 0;
            $total_paid = 0;
            $total_due = 0;

            $data = [];
            foreach ($orders as $index => $order) {
                $total_subtotal += $order->subtotal;
                $total_discount += $order->dis_amount;
                $total_grand += $order->grand_total;
                $total_paid += $order->paid;
                $total_due += $order->due;

                $data[] = [
                    'sno' => $index + 1,
                    'invoice_date' => Carbon::parse($order->invoice_date)->format('d-m-Y'),
                    'order_id' => $order->id,
                    'order_booker_name' => $order->order_booker_name,
                    'customer_name' => $order->customer_name,
                    'subtotal' => number_format($order->subtotal, 2),
                    'discount' => number_format($order->dis_amount, 2),
                    'grand_total' => number_format($order->grand_total, 2),
                    'paid' => number_format($order->paid, 2),
                    'due' => number_format($order->due, 2),
                    'status' => ucfirst($order->status),
                    'status_badge' => $this->getStatusBadge(strtolower($order->status))
                ];
            }

            return response()->json([
                'status' => 200,
                'data' => $data,
                'totals' => [
                    'total_orders' => $total_orders,
                    'total_subtotal' => number_format($total_subtotal, 2),
                    'total_discount' => number_format($total_discount, 2),
                    'total_grand' => number_format($total_grand, 2),
                    'total_paid' => number_format($total_paid, 2),
                    'total_due' => number_format($total_due, 2)
                ]
            ]);
        } else {
            // Summary view (daily or monthly)
            $summary_query = DB::table('order_bookers')
                ->where('is_active', 1);

            // Filter by order booker
            if (Auth::user()->hasRole(['Admin', 'Super Admin'])) {
                if (!empty($order_booker_id) && $order_booker_id != 'all') {
                    $summary_query->where('saleman_id', $order_booker_id);
                }
            } else {
                $summary_query->where('saleman_id', Auth::id());
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $summary_query->whereBetween('invoice_date', [$start_date, $end_date]);
            }

            if ($view_type === 'monthly') {
                $summary_query->select([
                    DB::raw('DATE_FORMAT(invoice_date, "%Y-%m") as period'),
                    DB::raw('DATE_FORMAT(invoice_date, "%M %Y") as period_label'),
                    DB::raw('COUNT(*) as total_orders'),
                    DB::raw('SUM(subtotal) as total_subtotal'),
                    DB::raw('SUM(grand_total) as total_amount')
                ])
                ->groupBy('period', 'period_label')
                ->orderBy('period', 'desc');
            } else {
                $summary_query->select([
                    'invoice_date as period',
                    DB::raw('invoice_date as period_label'),
                    DB::raw('COUNT(*) as total_orders'),
                    DB::raw('SUM(subtotal) as total_subtotal'),
                    DB::raw('SUM(grand_total) as total_amount')
                ])
                ->groupBy('invoice_date')
                ->orderBy('invoice_date', 'desc');
            }

            $summary_data = $summary_query->get();

            // Format summary
            $formatted_summary = $summary_data->map(function($item) use ($view_type) {
                return [
                    'period' => $view_type === 'daily' ? Carbon::parse($item->period_label)->format('d-m-Y') : $item->period_label,
                    'total_orders' => $item->total_orders,
                    'total_subtotal' => number_format($item->total_subtotal, 2),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });

            return response()->json([
                'status' => 200,
                'summary' => $formatted_summary
            ]);
        }
    }

    /**
     * Storeman Order for Stock Report - View
     */
    public function storeman_order_stock_report()
    {
        // Get all salesmen for filter dropdown
        $salesmen = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'Sales Man')
            ->select('users.id', 'users.name')
            ->get();

        // Get all products
        $products = DB::table('products')
            ->select('id', 'pro_name')
            ->get();

        return view('admin.reports.storeman_order_stock_report', compact('salesmen', 'products'));
    }

    /**
     * Storeman Order for Stock Report - AJAX Data
     */
    public function storeman_order_stock_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $product_id = $request->input('product_id');

        // Build query for salesman stock orders
        $query = DB::table('saleman_ledgers')
            ->join('users as salesman', 'saleman_ledgers.salesman_id', '=', 'salesman.id')
            ->join('products', 'saleman_ledgers.pro_id', '=', 'products.id')
            ->select([
                'saleman_ledgers.id',
                'saleman_ledgers.invoice_date',
                'salesman.name as salesman_name',
                'products.pro_name',
                'saleman_ledgers.pro_qty',
                'saleman_ledgers.transaction_type',
                'saleman_ledgers.detail'
            ])
            ->where('saleman_ledgers.transaction_type', 'order')
            ->whereNotNull('saleman_ledgers.pro_id')
            ->whereNull('saleman_ledgers.deleted_at');

        // Salesman filter
        if (!empty($salesman_id) && $salesman_id != 'all') {
            $query->where('saleman_ledgers.salesman_id', $salesman_id);
        }

        // Product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('saleman_ledgers.pro_id', $product_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
        }

        $orders = $query->orderBy('saleman_ledgers.invoice_date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;

        $data = [];
        foreach ($orders as $index => $order) {
            $total_quantity += $order->pro_qty ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'invoice_date' => Carbon::parse($order->invoice_date)->format('d-m-Y'),
                'salesman_name' => $order->salesman_name,
                'pro_name' => $order->pro_name,
                'quantity' => number_format($order->pro_qty ?? 0, 0),
                'detail' => $order->detail ?? 'N/A'
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_records' => count($data),
                'total_quantity' => number_format($total_quantity, 0)
            ]
        ]);
    }

    /**
     * Storeman Order for Stock Summary - View
     */
    public function storeman_order_stock_summary()
    {
        // Get all salesmen for filter dropdown
        $salesmen = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'Sales Man')
            ->select('users.id', 'users.name')
            ->get();

        // Get all products
        $products = DB::table('products')
            ->select('id', 'pro_name')
            ->get();

        return view('admin.reports.storeman_order_stock_summary', compact('salesmen', 'products'));
    }

    /**
     * Storeman Order for Stock Summary - AJAX Data
     */
    public function storeman_order_stock_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $product_id = $request->input('product_id');
        $view_type = $request->input('view_type', 'product'); // product, salesman, daily, monthly

        if ($view_type === 'product') {
            // Product-wise summary
            $query = DB::table('saleman_ledgers')
                ->join('products', 'saleman_ledgers.pro_id', '=', 'products.id')
                ->select([
                    'products.pro_name',
                    DB::raw('SUM(saleman_ledgers.pro_qty) as total_qty')
                ])
                ->where('saleman_ledgers.transaction_type', 'order')
                ->whereNotNull('saleman_ledgers.pro_id')
                ->whereNull('saleman_ledgers.deleted_at')
                ->groupBy('products.id', 'products.pro_name');

            // Salesman filter
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('saleman_ledgers.salesman_id', $salesman_id);
            }

            // Product filter
            if (!empty($product_id) && $product_id != 'all') {
                $query->where('saleman_ledgers.pro_id', $product_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $summary = $query->orderBy('total_qty', 'desc')->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->pro_name,
                    'total_qty' => number_format($item->total_qty, 0)
                ];
            });
        } elseif ($view_type === 'salesman') {
            // Salesman-wise summary
            $query = DB::table('saleman_ledgers')
                ->join('users as salesman', 'saleman_ledgers.salesman_id', '=', 'salesman.id')
                ->select([
                    'salesman.name as salesman_name',
                    DB::raw('SUM(saleman_ledgers.pro_qty) as total_qty')
                ])
                ->where('saleman_ledgers.transaction_type', 'order')
                ->whereNotNull('saleman_ledgers.pro_id')
                ->whereNull('saleman_ledgers.deleted_at')
                ->groupBy('salesman.id', 'salesman.name');

            // Salesman filter
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('saleman_ledgers.salesman_id', $salesman_id);
            }

            // Product filter
            if (!empty($product_id) && $product_id != 'all') {
                $query->where('saleman_ledgers.pro_id', $product_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $summary = $query->orderBy('total_qty', 'desc')->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->salesman_name,
                    'total_qty' => number_format($item->total_qty, 0)
                ];
            });
        } elseif ($view_type === 'monthly') {
            // Monthly summary
            $query = DB::table('saleman_ledgers')
                ->select([
                    DB::raw('DATE_FORMAT(saleman_ledgers.invoice_date, "%Y-%m") as period'),
                    DB::raw('DATE_FORMAT(saleman_ledgers.invoice_date, "%M %Y") as period_label'),
                    DB::raw('SUM(saleman_ledgers.pro_qty) as total_qty')
                ])
                ->where('saleman_ledgers.transaction_type', 'order')
                ->whereNotNull('saleman_ledgers.pro_id')
                ->whereNull('saleman_ledgers.deleted_at')
                ->groupBy('period', 'period_label');

            // Salesman filter
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('saleman_ledgers.salesman_id', $salesman_id);
            }

            // Product filter
            if (!empty($product_id) && $product_id != 'all') {
                $query->where('saleman_ledgers.pro_id', $product_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $summary = $query->orderBy('period', 'desc')->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->period_label,
                    'total_qty' => number_format($item->total_qty, 0)
                ];
            });
        } else {
            // Daily summary
            $query = DB::table('saleman_ledgers')
                ->select([
                    'saleman_ledgers.invoice_date as period',
                    DB::raw('saleman_ledgers.invoice_date as period_label'),
                    DB::raw('SUM(saleman_ledgers.pro_qty) as total_qty')
                ])
                ->where('saleman_ledgers.transaction_type', 'order')
                ->whereNotNull('saleman_ledgers.pro_id')
                ->whereNull('saleman_ledgers.deleted_at')
                ->groupBy('saleman_ledgers.invoice_date');

            // Salesman filter
            if (!empty($salesman_id) && $salesman_id != 'all') {
                $query->where('saleman_ledgers.salesman_id', $salesman_id);
            }

            // Product filter
            if (!empty($product_id) && $product_id != 'all') {
                $query->where('saleman_ledgers.pro_id', $product_id);
            }

            // Date range filter
            if (!empty($start_date) && !empty($end_date)) {
                $query->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
            }

            $summary = $query->orderBy('period', 'desc')->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => Carbon::parse($item->period_label)->format('d-m-Y'),
                    'total_qty' => number_format($item->total_qty, 0)
                ];
            });
        }

        return response()->json([
            'status' => 200,
            'summary' => $formatted_summary
        ]);
    }

    /**
     * Storeman Sale Return Report - View
     */
    public function storeman_sale_return_report()
    {
        // Get all salesmen for filter dropdown
        $salesmen = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'Sales Man')
            ->select('users.id', 'users.name')
            ->get();

        // Get all customers
        $customers = DB::table('customers')
            ->select('id', DB::raw('COALESCE(shop_name, name) as display_name'))
            ->get();

        // Get all products
        $products = DB::table('products')
            ->select('id', 'pro_name')
            ->get();

        return view('admin.reports.storeman_sale_return_report', compact('salesmen', 'customers', 'products'));
    }

    /**
     * Storeman Sale Return Report - AJAX Data
     */
    public function storeman_sale_return_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $customer_id = $request->input('customer_id');
        $product_id = $request->input('product_id');

        // Build query
        $query = DB::table('sale_return_details')
            ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id')
            ->join('customers', 'sale_returns.cus_id', '=', 'customers.id')
            ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
            ->join('users as salesman', 'sale_returns.saleman_id', '=', 'salesman.id')
            ->select([
                'sale_return_details.id',
                'sale_returns.id as sale_return_id',
                'sale_returns.invoice_date',
                DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                'salesman.name as salesman_name',
                'products.pro_name',
                'sale_return_details.batch_no',
                'sale_return_details.pro_qty',
                'sale_return_details.sale_return_price'
            ]);

        // Salesman filter
        if (!empty($salesman_id) && $salesman_id != 'all') {
            $query->where('sale_returns.saleman_id', $salesman_id);
        }

        // Customer filter
        if (!empty($customer_id) && $customer_id != 'all') {
            $query->where('sale_returns.cus_id', $customer_id);
        }

        // Product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('sale_return_details.pro_id', $product_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }

        $returns = $query->orderBy('sale_returns.invoice_date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_amount = 0;

        $data = [];
        foreach ($returns as $index => $return) {
            $amount = $return->sale_return_price * $return->pro_qty;
            $total_quantity += $return->pro_qty;
            $total_amount += $amount;

            $data[] = [
                'sno' => $index + 1,
                'invoice_date' => Carbon::parse($return->invoice_date)->format('d-m-Y'),
                'sale_return_id' => $return->sale_return_id,
                'customer_name' => $return->customer_name,
                'salesman_name' => $return->salesman_name,
                'pro_name' => $return->pro_name,
                'batch_no' => $return->batch_no ?? 'N/A',
                'quantity' => number_format($return->pro_qty, 0),
                'price' => number_format($return->sale_return_price, 2),
                'amount' => number_format($amount, 2)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_records' => count($data),
                'total_quantity' => number_format($total_quantity, 0),
                'total_amount' => number_format($total_amount, 2)
            ]
        ]);
    }

    /**
     * Storeman Sale Return Summary - View
     */
    public function storeman_sale_return_summary()
    {
        // Get all salesmen for filter dropdown
        $salesmen = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'Sales Man')
            ->select('users.id', 'users.name')
            ->get();

        // Get all customers
        $customers = DB::table('customers')
            ->select('id', DB::raw('COALESCE(shop_name, name) as display_name'))
            ->get();

        // Get all products
        $products = DB::table('products')
            ->select('id', 'pro_name')
            ->get();

        return view('admin.reports.storeman_sale_return_summary', compact('salesmen', 'customers', 'products'));
    }

    /**
     * Storeman Sale Return Summary - AJAX Data
     */
    public function storeman_sale_return_summary_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;
        $salesman_id = $request->input('salesman_id');
        $customer_id = $request->input('customer_id');
        $product_id = $request->input('product_id');
        $view_type = $request->input('view_type', 'product'); // product, customer, salesman, daily, monthly

        $base_query = DB::table('sale_return_details')
            ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id');

        // Apply filters
        if (!empty($salesman_id) && $salesman_id != 'all') {
            $base_query->where('sale_returns.saleman_id', $salesman_id);
        }

        if (!empty($customer_id) && $customer_id != 'all') {
            $base_query->where('sale_returns.cus_id', $customer_id);
        }

        if (!empty($product_id) && $product_id != 'all') {
            $base_query->where('sale_return_details.pro_id', $product_id);
        }

        if (!empty($start_date) && !empty($end_date)) {
            $base_query->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }

        if ($view_type === 'product') {
            // Product-wise summary
            $summary = $base_query
                ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
                ->select([
                    'products.pro_name',
                    DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                    DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
                ])
                ->groupBy('products.id', 'products.pro_name')
                ->orderBy('total_amount', 'desc')
                ->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->pro_name,
                    'total_qty' => number_format($item->total_qty, 0),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });
        } elseif ($view_type === 'customer') {
            // Customer-wise summary
            $summary = $base_query
                ->join('customers', 'sale_returns.cus_id', '=', 'customers.id')
                ->select([
                    DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'),
                    DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                    DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
                ])
                ->groupBy('customers.id', DB::raw('COALESCE(customers.shop_name, customers.name)'))
                ->orderBy('total_amount', 'desc')
                ->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->customer_name,
                    'total_qty' => number_format($item->total_qty, 0),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });
        } elseif ($view_type === 'salesman') {
            // Salesman-wise summary
            $summary = $base_query
                ->join('users as salesman', 'sale_returns.saleman_id', '=', 'salesman.id')
                ->select([
                    'salesman.name as salesman_name',
                    DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                    DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
                ])
                ->groupBy('salesman.id', 'salesman.name')
                ->orderBy('total_amount', 'desc')
                ->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->salesman_name,
                    'total_qty' => number_format($item->total_qty, 0),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });
        } elseif ($view_type === 'monthly') {
            // Monthly summary
            $summary = $base_query
                ->select([
                    DB::raw('DATE_FORMAT(sale_returns.invoice_date, "%Y-%m") as period'),
                    DB::raw('DATE_FORMAT(sale_returns.invoice_date, "%M %Y") as period_label'),
                    DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                    DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
                ])
                ->groupBy('period', 'period_label')
                ->orderBy('period', 'desc')
                ->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => $item->period_label,
                    'total_qty' => number_format($item->total_qty, 0),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });
        } else {
            // Daily summary
            $summary = $base_query
                ->select([
                    'sale_returns.invoice_date as period',
                    DB::raw('sale_returns.invoice_date as period_label'),
                    DB::raw('SUM(sale_return_details.pro_qty) as total_qty'),
                    DB::raw('SUM(sale_return_details.sale_return_price * sale_return_details.pro_qty) as total_amount')
                ])
                ->groupBy('sale_returns.invoice_date')
                ->orderBy('sale_returns.invoice_date', 'desc')
                ->get();

            $formatted_summary = $summary->map(function($item) {
                return [
                    'group_name' => Carbon::parse($item->period_label)->format('d-m-Y'),
                    'total_qty' => number_format($item->total_qty, 0),
                    'total_amount' => number_format($item->total_amount, 2)
                ];
            });
        }

        return response()->json([
            'status' => 200,
            'summary' => $formatted_summary
        ]);
    }

    /**
     * Production Report - View
     */
    public function production_report()
    {
        // Get all contractors for filter dropdown
        $contractors = DB::table('contractors')->select('id', 'name')->orderBy('name')->get();

        return view('admin.reports.production_report', compact('contractors'));
    }

    /**
     * Production Report - AJAX Data
     */
    public function production_report_data(Request $request)
    {
        $contractor_id = $request->input('contractor_id');
        $view_type = $request->input('view_type', 'all'); // all, daily, monthly
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query
        $query = DB::table('productions')
            ->leftJoin('contractors', 'productions.contractor_id', '=', 'contractors.id')
            ->leftJoin('products', 'productions.pro_id', '=', 'products.id')
            ->select([
                'productions.id',
                DB::raw('COALESCE(contractors.name, "N/A") as contractor_name'),
                DB::raw('COALESCE(products.pro_name, "N/A") as product_name'),
                'productions.order_number as batch_no',
                'productions.pro_qty as quantity',
                'productions.labour_cost',
                DB::raw('(productions.total_cost - COALESCE(productions.labour_cost, 0)) as material_cost'),
                'productions.total_cost',
                'productions.start_date',
                'productions.status'
            ]);

        // Contractor filter
        if (!empty($contractor_id) && $contractor_id != 'all') {
            $query->where('productions.contractor_id', $contractor_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('productions.start_date', [$start_date, $end_date]);
        }

        // Get all data
        $productions = $query->orderBy('productions.start_date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_labour_cost = 0;
        $total_material_cost = 0;
        $total_cost = 0;

        $data = [];
        foreach ($productions as $index => $production) {
            $total_quantity += $production->quantity;
            $total_labour_cost += $production->labour_cost ?? 0;
            $total_material_cost += $production->material_cost ?? 0;
            $total_cost += $production->total_cost;

            $data[] = [
                'sno' => $index + 1,
                'id' => $production->id,
                'contractor_name' => $production->contractor_name,
                'product_name' => $production->product_name,
                'batch_no' => $production->batch_no ?? 'N/A',
                'quantity' => number_format($production->quantity, 0),
                'labour_cost' => number_format($production->labour_cost ?? 0, 2),
                'material_cost' => number_format($production->material_cost ?? 0, 2),
                'total_cost' => number_format($production->total_cost, 2),
                'date' => !empty($production->start_date) ? Carbon::parse($production->start_date)->format('d-m-Y') : 'N/A',
                'status' => ucfirst($production->status)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_quantity' => number_format($total_quantity, 0),
                'total_labour_cost' => number_format($total_labour_cost, 2),
                'total_material_cost' => number_format($total_material_cost, 2),
                'total_cost' => number_format($total_cost, 2)
            ]
        ]);
    }

    /**
     * Contractor Order Report - View
     */
    public function contractor_order_report()
    {
        // Get all contractors for filter dropdown
        $contractors = DB::table('contractors')->select('id', 'name')->orderBy('name')->get();

        return view('admin.reports.contractor_order_report', compact('contractors'));
    }

    /**
     * Contractor Order Report - AJAX Data
     */
    public function contractor_order_report_data(Request $request)
    {
        $contractor_id = $request->input('contractor_id');
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query - from contracter_ledgers table which stores contractor orders
        $query = DB::table('contracter_ledgers')
            ->join('contractors', 'contracter_ledgers.contractor_id', '=', 'contractors.id')
            ->leftJoin('products', 'contracter_ledgers.pro_id', '=', 'products.id')
            ->select([
                'contracter_ledgers.id',
                'contractors.name as contractor_name',
                DB::raw('COALESCE(products.pro_name, "N/A") as product_name'),
                'contracter_ledgers.batch_number as batch_no',
                'contracter_ledgers.price as prod_price',
                'contracter_ledgers.pro_qty as quantity',
                'contracter_ledgers.labour_cost',
                'contracter_ledgers.material_cost',
                'contracter_ledgers.total_price',
                'contracter_ledgers.date',
                'contracter_ledgers.status',
                'contracter_ledgers.payment_status'
            ])
            ->whereNotNull('contracter_ledgers.batch_number'); // Only records that are orders

        // Contractor filter
        if (!empty($contractor_id) && $contractor_id != 'all') {
            $query->where('contracter_ledgers.contractor_id', $contractor_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('contracter_ledgers.date', [$start_date, $end_date]);
        }

        // Get all data
        $orders = $query->orderBy('contracter_ledgers.date', 'desc')->get();

        // Calculate totals
        $total_quantity = 0;
        $total_labour_cost = 0;
        $total_material_cost = 0;
        $total_price = 0;

        $data = [];
        foreach ($orders as $index => $order) {
            $total_quantity += $order->quantity ?? 0;
            $total_labour_cost += $order->labour_cost ?? 0;
            $total_material_cost += $order->material_cost ?? 0;
            $total_price += $order->total_price ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'id' => $order->id,
                'contractor_name' => $order->contractor_name,
                'product_name' => $order->product_name,
                'batch_no' => $order->batch_no ?? 'N/A',
                'prod_price' => number_format($order->prod_price ?? 0, 2),
                'quantity' => number_format($order->quantity ?? 0, 0),
                'labour_cost' => number_format($order->labour_cost ?? 0, 2),
                'material_cost' => number_format($order->material_cost ?? 0, 2),
                'total_price' => number_format($order->total_price ?? 0, 2),
                'date' => !empty($order->date) ? Carbon::parse($order->date)->format('d-m-Y') : 'N/A',
                'status' => $order->status ?? 'N/A',
                'payment_status' => $order->payment_status ?? 'N/A'
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_quantity' => number_format($total_quantity, 0),
                'total_labour_cost' => number_format($total_labour_cost, 2),
                'total_material_cost' => number_format($total_material_cost, 2),
                'total_price' => number_format($total_price, 2)
            ]
        ]);
    }

    /**
     * Non-Serviceable Stock Report - View
     */
    public function non_serviceable_stock_report_new()
    {
        // Get all products, salesmen, and contractors for filters
        $products = DB::table('products')->select('id', 'pro_name')->orderBy('pro_name')->get();
        $salesmen = DB::table('users')
            ->join('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
            ->where('roles.name', 'Sales Man')
            ->select('users.id', 'users.name')
            ->get();
        $contractors = DB::table('contractors')->select('id', 'name')->orderBy('name')->get();

        return view('admin.reports.non_serviceable_stock_report', compact('products', 'salesmen', 'contractors'));
    }

    /**
     * Non-Serviceable Stock Report - AJAX Data
     */
    public function non_serviceable_stock_report_data(Request $request)
    {
        $filter_type = $request->input('filter_type', 'all'); // all, product_sale_return, salesman_sale_return, product_repair, contractor_repair
        $product_id = $request->input('product_id');
        $salesman_id = $request->input('salesman_id');
        $contractor_id = $request->input('contractor_id');

        $data = [];

        if ($filter_type == 'all' || $filter_type == 'product_sale_return' || $filter_type == 'salesman_sale_return') {
            // Get data from sale returns
            $query = DB::table('sale_return_details')
                ->join('sale_returns', 'sale_return_details.sale_return_id', '=', 'sale_returns.id')
                ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
                ->leftJoin('users', 'sale_returns.saleman_id', '=', 'users.id')
                ->select([
                    'sale_return_details.id',
                    'products.pro_name',
                    'products.sku',
                    'sale_return_details.pro_qty',
                    'sale_return_details.batch_no',
                    'sale_returns.invoice_date',
                    DB::raw('"Sale Return" as source'),
                    DB::raw('COALESCE(users.name, "N/A") as related_person')
                ]);

            if ($filter_type == 'product_sale_return' && !empty($product_id)) {
                $query->where('sale_return_details.pro_id', $product_id);
            }

            if ($filter_type == 'salesman_sale_return' && !empty($salesman_id)) {
                $query->where('sale_returns.saleman_id', $salesman_id);
            }

            $sale_returns = $query->get();

            foreach ($sale_returns as $index => $item) {
                $data[] = [
                    'sno' => count($data) + 1,
                    'product_name' => $item->pro_name,
                    'sku' => $item->sku ?? 'N/A',
                    'batch_no' => $item->batch_no ?? 'N/A',
                    'quantity' => number_format($item->pro_qty, 0),
                    'date' => Carbon::parse($item->invoice_date)->format('d-m-Y'),
                    'source' => $item->source,
                    'related_person' => $item->related_person
                ];
            }
        }

        if ($filter_type == 'all' || $filter_type == 'product_repair' || $filter_type == 'contractor_repair') {
            // Get data from repair orders
            $query = DB::table('repair_orders')
                ->leftJoin('products', 'repair_orders.pro_id', '=', 'products.id')
                ->leftJoin('contractors', 'repair_orders.contractor_id', '=', 'contractors.id')
                ->select([
                    'repair_orders.id',
                    DB::raw('COALESCE(products.pro_name, "N/A") as product_name'),
                    DB::raw('COALESCE(products.sku, "N/A") as sku'),
                    'repair_orders.pro_qty',
                    DB::raw('"N/A" as batch_no'),
                    'repair_orders.repair_date',
                    DB::raw('"Repair Order" as source'),
                    DB::raw('COALESCE(contractors.name, "N/A") as related_person')
                ]);

            if ($filter_type == 'product_repair' && !empty($product_id)) {
                $query->where('repair_orders.pro_id', $product_id);
            }

            if ($filter_type == 'contractor_repair' && !empty($contractor_id)) {
                $query->where('repair_orders.contractor_id', $contractor_id);
            }

            $repairs = $query->get();

            foreach ($repairs as $index => $item) {
                $data[] = [
                    'sno' => count($data) + 1,
                    'product_name' => $item->product_name ?? 'N/A',
                    'sku' => $item->sku,
                    'batch_no' => $item->batch_no,
                    'quantity' => number_format($item->pro_qty ?? 0, 0),
                    'date' => !empty($item->repair_date) ? Carbon::parse($item->repair_date)->format('d-m-Y') : 'N/A',
                    'source' => $item->source,
                    'related_person' => $item->related_person
                ];
            }
        }

        return response()->json([
            'status' => 200,
            'data' => $data
        ]);
    }

    /**
     * Repairing Order Report - View
     */
    public function repairing_order_report()
    {
        // Get all contractors for filter dropdown
        $contractors = DB::table('contractors')->select('id', 'name')->orderBy('name')->get();

        return view('admin.reports.repairing_order_report', compact('contractors'));
    }

    /**
     * Repairing Order Report - AJAX Data
     */
    public function repairing_order_report_data(Request $request)
    {
        $contractor_id = $request->input('contractor_id');
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query
        $query = DB::table('repair_orders')
            ->leftJoin('products', 'repair_orders.pro_id', '=', 'products.id')
            ->leftJoin('contractors', 'repair_orders.contractor_id', '=', 'contractors.id')
            ->select([
                'repair_orders.id',
                DB::raw('COALESCE(contractors.name, "N/A") as contractor_name'),
                DB::raw('COALESCE(products.pro_name, "N/A") as product_name'),
                'repair_orders.price',
                'repair_orders.labour_cost',
                'repair_orders.material_cost',
                'repair_orders.total_price',
                'repair_orders.status'
            ]);

        // Contractor filter
        if (!empty($contractor_id) && $contractor_id != 'all') {
            $query->where('repair_orders.contractor_id', $contractor_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('repair_orders.repair_date', [$start_date, $end_date]);
        }

        // Get all data
        $repairs = $query->orderBy('repair_orders.id', 'desc')->get();

        // Calculate totals
        $total_price = 0;
        $total_labour_cost = 0;
        $total_material_cost = 0;
        $total_total_price = 0;

        $data = [];
        foreach ($repairs as $index => $repair) {
            $total_price += $repair->price ?? 0;
            $total_labour_cost += $repair->labour_cost ?? 0;
            $total_material_cost += $repair->material_cost ?? 0;
            $total_total_price += $repair->total_price ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'id' => $repair->id,
                'contractor_name' => $repair->contractor_name,
                'product_name' => $repair->product_name ?? 'N/A',
                'price' => number_format($repair->price ?? 0, 2),
                'labour_cost' => number_format($repair->labour_cost ?? 0, 2),
                'material_cost' => number_format($repair->material_cost ?? 0, 2),
                'total_price' => number_format($repair->total_price ?? 0, 2),
                'status' => ucfirst($repair->status)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_price' => number_format($total_price, 2),
                'total_labour_cost' => number_format($total_labour_cost, 2),
                'total_material_cost' => number_format($total_material_cost, 2),
                'total_total_price' => number_format($total_total_price, 2)
            ]
        ]);
    }

    /**
     * Salaries Report - View
     */
    public function salaries_report()
    {
        // Get all employees for filter dropdown (only for Admin/Cashier)
        $employees = DB::table('employees')->select('id', 'emp_name')->orderBy('emp_name')->get();

        // Check if user is an Employee - get their employee record
        $user_employee_id = null;
        if (Auth::user()->hasRole('Employee')) {
            $user = Auth::user();
            $employee = null;

            // First try to find by email (most reliable match)
            if ($user->email) {
                $employee = DB::table('employees')->where('email', $user->email)->first();
            }

            // If not found by email, try created_by
            if (!$employee) {
                $employee = DB::table('employees')->where('created_by', $user->id)->first();
            }

            $user_employee_id = $employee ? $employee->id : null;
        }

        return view('admin.reports.salaries_report', compact('employees', 'user_employee_id'));
    }

    /**
     * Salaries Report - AJAX Data
     */
    public function salaries_report_data(Request $request)
    {
        $employee_id = $request->input('employee_id');
        $month_year = $request->input('month_year'); // Format: YYYY-MM
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m') : null;

        // Build query
        $query = DB::table('salaries')
            ->join('employees', 'salaries.employee_id', '=', 'employees.id')
            ->select([
                'salaries.id',
                'employees.emp_name',
                'employees.id as employee_id',
                'salaries.month',
                'salaries.salary_per_day',
                'salaries.working_days',
                'salaries.advance',
                'salaries.total_salary',
                'salaries.status'
            ]);

        // If user is Employee, show only their own salaries
        if (Auth::user()->hasRole('Employee')) {
            $user = Auth::user();
            $employee = null;

            // First try to find by email (most reliable match)
            if ($user->email) {
                $employee = DB::table('employees')->where('email', $user->email)->first();
            }

            // If not found by email, try created_by
            if (!$employee) {
                $employee = DB::table('employees')->where('created_by', $user->id)->first();
            }

            if ($employee) {
                $query->where('salaries.employee_id', $employee->id);
            }
        } elseif (!empty($employee_id) && $employee_id != 'all') {
            // Admin/Cashier can filter by employee
            $query->where('salaries.employee_id', $employee_id);
        }

        // Month/Year filter (exact month)
        if (!empty($month_year)) {
            $query->where('salaries.month', $month_year);
        }

        // Date range filter (month range)
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('salaries.month', [$start_date, $end_date]);
        }

        // Get all data
        $salaries = $query->orderBy('salaries.month', 'desc')->get();

        // Calculate totals
        $total_advance = 0;
        $total_salary = 0;

        $data = [];
        foreach ($salaries as $index => $salary) {
            $total_advance += $salary->advance ?? 0;
            $total_salary += $salary->total_salary ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'emp_name' => $salary->emp_name ?? 'N/A',
                'employee_id' => $salary->employee_id,
                'month' => !empty($salary->month) ? Carbon::parse($salary->month . '-01')->format('M Y') : 'N/A',
                'salary_per_day' => number_format($salary->salary_per_day ?? 0, 2),
                'working_days' => $salary->working_days ?? 0,
                'advance' => number_format($salary->advance ?? 0, 2),
                'total_salary' => number_format($salary->total_salary ?? 0, 2),
                'status' => ucfirst($salary->status ?? 'unpaid'),
                'signature' => '' // Empty for printing
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_advance' => number_format($total_advance, 2),
                'total_salary' => number_format($total_salary, 2)
            ]
        ]);
    }

    /**
     * Purchase Report - View
     */
    public function purchase_report()
    {
        // Get all suppliers for filter dropdown
        $suppliers = DB::table('suppliers')->select('id', 'sup_name')->orderBy('sup_name')->get();

        // Get all products for filter dropdown
        $products = DB::table('products')->select('id', 'pro_name')->orderBy('pro_name')->get();

        return view('admin.reports.purchase_report', compact('suppliers', 'products'));
    }

    /**
     * Purchase Report - AJAX Data
     */
    public function purchase_report_data(Request $request)
    {
        $filter_type = $request->input('filter_type', 'all'); // all, supplier, product, assets
        $supplier_id = $request->input('supplier_id');
        $product_id = $request->input('product_id');
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query
        $query = DB::table('purchase_details')
            ->join('purchases', 'purchase_details.purchase_id', '=', 'purchases.id')
            ->join('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
            ->join('products', 'purchase_details.pro_id', '=', 'products.id')
            ->select([
                'purchase_details.id',
                'suppliers.sup_name as supplier_name',
                'products.pro_name',
                'purchase_details.purchase_price',
                'purchase_details.pro_qty',
                'purchase_details.total_amount',
                'purchases.paid_amount',
                'purchases.remain_amount',
                'purchases.date',
                'purchases.id as purchase_id'
            ]);

        // Filter by type
        if ($filter_type == 'supplier' && !empty($supplier_id)) {
            $query->where('purchases.supplier_id', $supplier_id);
        } elseif ($filter_type == 'product' && !empty($product_id)) {
            $query->where('purchase_details.pro_id', $product_id);
        } elseif ($filter_type == 'assets') {
            // Filter for assets category - assuming category_id 1 is Assets
            $query->where('products.category_id', 1);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('purchases.date', [$start_date, $end_date]);
        }

        // Get all data
        $purchases = $query->orderBy('purchases.date', 'desc')->get();

        // Calculate totals
        $total_amount = 0;
        $total_paid = 0;
        $total_remaining = 0;

        $data = [];
        foreach ($purchases as $index => $purchase) {
            $total_amount += $purchase->total_amount ?? 0;
            $total_paid += $purchase->paid_amount ?? 0;
            $total_remaining += $purchase->remain_amount ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'id' => $purchase->purchase_id,
                'supplier_name' => $purchase->supplier_name ?? 'N/A',
                'product_name' => $purchase->pro_name ?? 'N/A',
                'price' => number_format($purchase->purchase_price ?? 0, 2),
                'qty' => number_format($purchase->pro_qty ?? 0, 0),
                'total_amount' => number_format($purchase->total_amount ?? 0, 2),
                'paid_amount' => number_format($purchase->paid_amount ?? 0, 2),
                'remain_amount' => number_format($purchase->remain_amount ?? 0, 2),
                'date' => !empty($purchase->date) ? Carbon::parse($purchase->date)->format('d-m-Y') : 'N/A'
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_amount' => number_format($total_amount, 2),
                'total_paid' => number_format($total_paid, 2),
                'total_remaining' => number_format($total_remaining, 2)
            ]
        ]);
    }

    /**
     * Purchase Return Report - View
     */
    public function purchase_return_report()
    {
        // Get all suppliers for filter dropdown
        $suppliers = DB::table('suppliers')->select('id', 'sup_name')->orderBy('sup_name')->get();

        return view('admin.reports.purchase_return_report', compact('suppliers'));
    }

    /**
     * Purchase Return Report - AJAX Data
     */
    public function purchase_return_report_data(Request $request)
    {
        $supplier_id = $request->input('supplier_id');
        $status = $request->input('status'); // 0=pending, 1=approved
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query
        $query = DB::table('purchase_return_details')
            ->join('purchase_returns', 'purchase_return_details.purchase_return_id', '=', 'purchase_returns.id')
            ->join('suppliers', 'purchase_returns.supplier_id', '=', 'suppliers.id')
            ->join('products', 'purchase_return_details.pro_id', '=', 'products.id')
            ->select([
                'purchase_return_details.id',
                'suppliers.sup_name as supplier_name',
                'products.pro_name',
                'purchase_return_details.batch_no',
                'purchase_return_details.pro_qty as return_qty',
                'purchase_return_details.total as return_amount',
                'purchase_returns.refund_amount',
                'purchase_returns.adjustment_amount',
                'purchase_returns.return_date',
                'purchase_returns.status',
                'purchase_returns.id as return_id'
            ]);

        // Supplier filter
        if (!empty($supplier_id) && $supplier_id != 'all') {
            $query->where('purchase_returns.supplier_id', $supplier_id);
        }

        // Status filter
        if ($status !== null && $status !== '' && $status != 'all') {
            $query->where('purchase_returns.status', $status);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('purchase_returns.return_date', [$start_date, $end_date]);
        }

        // Get all data
        $returns = $query->orderBy('purchase_returns.return_date', 'desc')->get();

        // Calculate totals
        $total_return_amount = 0;
        $total_refund = 0;
        $total_adjustment = 0;

        $data = [];
        foreach ($returns as $index => $return) {
            $total_return_amount += $return->return_amount ?? 0;
            $total_refund += $return->refund_amount ?? 0;
            $total_adjustment += $return->adjustment_amount ?? 0;

            $statusText = 'Pending';
            if ($return->status == 1) {
                $statusText = 'Approved';
            } elseif ($return->status == 2) {
                $statusText = 'Rejected';
            }

            $data[] = [
                'sno' => $index + 1,
                'id' => $return->return_id,
                'supplier_name' => $return->supplier_name ?? 'N/A',
                'product_name' => $return->pro_name ?? 'N/A',
                'batch_no' => $return->batch_no ?? 'N/A',
                'return_qty' => number_format($return->return_qty ?? 0, 0),
                'return_amount' => number_format($return->return_amount ?? 0, 2),
                'refund_amount' => number_format($return->refund_amount ?? 0, 2),
                'adjustment_amount' => number_format($return->adjustment_amount ?? 0, 2),
                'date' => !empty($return->return_date) ? Carbon::parse($return->return_date)->format('d-m-Y') : 'N/A',
                'status' => $statusText
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'total_return_amount' => number_format($total_return_amount, 2),
                'total_refund' => number_format($total_refund, 2),
                'total_adjustment' => number_format($total_adjustment, 2)
            ]
        ]);
    }

    /**
     * Purchase Qty Report - View
     */
    public function purchase_qty_report()
    {
        // Get all suppliers for filter dropdown
        $suppliers = DB::table('suppliers')->select('id', 'sup_name')->orderBy('sup_name')->get();

        // Get all products for filter dropdown
        $products = DB::table('products')->select('id', 'pro_name')->orderBy('pro_name')->get();

        return view('admin.reports.purchase_qty_report', compact('suppliers', 'products'));
    }

    /**
     * Purchase Qty Report - AJAX Data (Summary Report)
     */
    public function purchase_qty_report_data(Request $request)
    {
        $supplier_id = $request->input('supplier_id');
        $product_id = $request->input('product_id');
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query - GROUP BY product
        $query = DB::table('purchase_details')
            ->join('purchases', 'purchase_details.purchase_id', '=', 'purchases.id')
            ->join('products', 'purchase_details.pro_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                'products.id as product_id',
                DB::raw('SUM(purchase_details.pro_qty) as total_qty')
            ])
            ->groupBy('products.id', 'products.pro_name');

        // Supplier filter
        if (!empty($supplier_id) && $supplier_id != 'all') {
            $query->where('purchases.supplier_id', $supplier_id);
        }

        // Product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('purchase_details.pro_id', $product_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('purchases.date', [$start_date, $end_date]);
        }

        // Get all data
        $products_summary = $query->orderBy('products.pro_name')->get();

        // Calculate total
        $grand_total_qty = 0;

        $data = [];
        foreach ($products_summary as $index => $product) {
            $grand_total_qty += $product->total_qty ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'product_name' => $product->pro_name ?? 'N/A',
                'total_qty' => number_format($product->total_qty ?? 0, 0)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'grand_total_qty' => number_format($grand_total_qty, 0)
            ]
        ]);
    }

    /**
     * Purchase Return Qty Report - View
     */
    public function purchase_return_qty_report()
    {
        // Get all suppliers for filter dropdown
        $suppliers = DB::table('suppliers')->select('id', 'sup_name')->orderBy('sup_name')->get();

        // Get all products for filter dropdown
        $products = DB::table('products')->select('id', 'pro_name')->orderBy('pro_name')->get();

        return view('admin.reports.purchase_return_qty_report', compact('suppliers', 'products'));
    }

    /**
     * Purchase Return Qty Report - AJAX Data (Summary Report)
     */
    public function purchase_return_qty_report_data(Request $request)
    {
        $supplier_id = $request->input('supplier_id');
        $product_id = $request->input('product_id');
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // Build query - GROUP BY product
        $query = DB::table('purchase_return_details')
            ->join('purchase_returns', 'purchase_return_details.purchase_return_id', '=', 'purchase_returns.id')
            ->join('products', 'purchase_return_details.pro_id', '=', 'products.id')
            ->select([
                'products.pro_name',
                'products.id as product_id',
                DB::raw('SUM(purchase_return_details.pro_qty) as total_return_qty')
            ])
            ->groupBy('products.id', 'products.pro_name');

        // Supplier filter
        if (!empty($supplier_id) && $supplier_id != 'all') {
            $query->where('purchase_returns.supplier_id', $supplier_id);
        }

        // Product filter
        if (!empty($product_id) && $product_id != 'all') {
            $query->where('purchase_return_details.pro_id', $product_id);
        }

        // Date range filter
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('purchase_returns.return_date', [$start_date, $end_date]);
        }

        // Get all data
        $products_summary = $query->orderBy('products.pro_name')->get();

        // Calculate total
        $grand_total_return_qty = 0;

        $data = [];
        foreach ($products_summary as $index => $product) {
            $grand_total_return_qty += $product->total_return_qty ?? 0;

            $data[] = [
                'sno' => $index + 1,
                'product_name' => $product->pro_name ?? 'N/A',
                'total_return_qty' => number_format($product->total_return_qty ?? 0, 0)
            ];
        }

        return response()->json([
            'status' => 200,
            'data' => $data,
            'totals' => [
                'grand_total_return_qty' => number_format($grand_total_return_qty, 0)
            ]
        ]);
    }

    /**
     * Cash Flow Report - View
     */
    public function cash_flow_report()
    {
        return view('admin.reports.cash_flow_report');
    }

    /**
     * Cash Flow / Profit & Loss Report - AJAX Data
     */
    public function cash_flow_data(Request $request)
    {
        $start_date = !empty($request->input('start_date')) ? Carbon::parse($request->input('start_date'))->format('Y-m-d') : null;
        $end_date = !empty($request->input('end_date')) ? Carbon::parse($request->input('end_date'))->format('Y-m-d') : null;

        // --- 1. INCOME / CASH INFLOW ---

        // Sales
        $sales_query = DB::table('sales')->where('is_deleted', '!=', 1);
        if ($start_date && $end_date) $sales_query->whereBetween('invoice_date', [$start_date, $end_date]);
        $total_sales = $sales_query->sum('grand_total');
        
        // Calculate Cash Sales (where bank_id is 0 or null or Payment_type is Cash)
        $cash_sales_query = clone $sales_query;
        $total_sales_cash = $cash_sales_query->where(function($q) {
            $q->where('Payment_type', 'Cash')->orWhereNull('bank_id')->orWhere('bank_id', 0);
        })->sum('paid');

        // Purchase Returns
        $purchase_return_query = DB::table('purchase_returns')->where('status', 1); // Approved
        if ($start_date && $end_date) $purchase_return_query->whereBetween('return_date', [$start_date, $end_date]);
        $total_purchase_returns = $purchase_return_query->sum('refund_amount');

        // Investment Received
        $investment_query = DB::table('investments')->where('status', 1);
        if ($start_date && $end_date) $investment_query->whereBetween('date', [$start_date, $end_date]);
        $total_investments = $investment_query->sum('invested_amount');
        // Cash Investments
        $cash_investment_query = clone $investment_query;
        $total_investments_cash = $cash_investment_query->where('payment_type', 'Cash')->sum('invested_amount');

        // Other Income (Manual Transactions)
        $other_income_query = DB::table('transactions')
            ->whereIn('transaction_type', ['credit', 'deposit', 'income'])
            ->where('source_type', '!=', 'sales');
        if ($start_date && $end_date) $other_income_query->whereBetween('created_at', [$start_date, $end_date]);
        $total_other_income = $other_income_query->sum('amount');
        // Cash Other Income
        $cash_income_query = clone $other_income_query;
        $total_other_income_cash = $cash_income_query->whereNull('bank_id')->sum('amount');


        // --- 2. EXPENSES / CASH OUTFLOW ---

        // Purchases
        $purchase_query = DB::table('purchases');
        if ($start_date && $end_date) $purchase_query->whereBetween('date', [$start_date, $end_date]);
        $purchases_data = $purchase_query->select(DB::raw('SUM(grand_total) as total_amount'), DB::raw('SUM(paid_amount) as paid_amount'))->first();
        $total_purchases_accrual = $purchases_data->total_amount ?? 0;
        $total_purchases_cash = $purchases_data->paid_amount ?? 0; // Assuming paid_amount is cash/bank outflow mixed, will treat as cash outflow for general calculation, but specific cash report might need separation if column existed.

        // Sale Returns
        $sale_return_query = DB::table('sale_returns');
        if ($start_date && $end_date) $sale_return_query->whereBetween('invoice_date', [$start_date, $end_date]);
        $total_sale_returns = $sale_return_query->sum('grand_total');

        // Production Costs
        $production_query = DB::table('productions')->where('status', '!=', 'cancelled');
        if ($start_date && $end_date) $production_query->whereBetween('start_date', [$start_date, $end_date]);
        $production_data = $production_query->select(DB::raw('SUM(total_cost) as total_cost'), DB::raw('SUM(labour_cost) as labour_cost'), DB::raw('SUM(total_cost - IFNULL(labour_cost,0)) as material_cost'))->first();
        $total_production_cost = $production_data->total_cost ?? 0;
        $total_labour_cost = $production_data->labour_cost ?? 0;
        $total_material_cost = $production_data->material_cost ?? 0;
        
        // Cash Production (Labour is usually cash)
        $cash_production_query = clone $production_query;
        $total_labour_cash = $cash_production_query->where('payment_type', 'Cash')->sum('labour_cost');

        // Salaries
        $salary_query = DB::table('salaries');
        if ($start_date && $end_date) $salary_query->whereBetween('month', [Carbon::parse($start_date)->format('Y-m'), Carbon::parse($end_date)->format('Y-m')]);
        $total_salaries = $salary_query->sum('total_salary');
        // Cash Salaries
        $cash_salary_query = clone $salary_query;
        $total_salaries_cash = $cash_salary_query->where('payment_type', 'Cash')->sum('total_salary');

        // General Expenses
        $expense_query = DB::table('expenses')->where('is_deleted', 0);
        if ($start_date && $end_date) $expense_query->whereBetween('exp_date', [$start_date, $end_date]);
        $total_expenses = $expense_query->sum('total_amt');

        // Salesman Expenses
        $salesman_expense_query = DB::table('saleman_ledgers')->where('transaction_type', 'expense');
        if ($start_date && $end_date) $salesman_expense_query->whereBetween('invoice_date', [$start_date, $end_date]);
        $total_salesman_expenses = $salesman_expense_query->sum('debit');

        // Investment Withdrawals
        $withdrawal_query = DB::table('investment_withdraws');
        if ($start_date && $end_date) $withdrawal_query->whereBetween('date', [$start_date, $end_date]);
        $total_withdrawals = $withdrawal_query->sum('amount');
        // Cash Withdrawals
        $cash_withdraw_query = clone $withdrawal_query;
        $total_withdrawals_cash = $cash_withdraw_query->where('payment_type', 'Cash')->sum('amount');


        // --- 3. BREAKDOWNS & DETAILS (For New Tabs) ---

        // A. Cash In Hand Details (Constructed)
        $cash_in_hand_details = [
            ['name' => 'Cash Sales', 'in' => $total_sales_cash, 'out' => 0],
            ['name' => 'Investments Received (Cash)', 'in' => $total_investments_cash, 'out' => 0],
            ['name' => 'Manual Deposits/Income', 'in' => $total_other_income_cash, 'out' => 0],
            ['name' => 'Cash Purchases (Paid)', 'in' => 0, 'out' => $total_purchases_cash], // Assuming purchases paid is outflow
            ['name' => 'Expenses (General)', 'in' => 0, 'out' => $total_expenses], // Assuming general expenses are cash
            ['name' => 'Expenses (Salesman)', 'in' => 0, 'out' => $total_salesman_expenses],
            ['name' => 'Employee Salaries (Cash)', 'in' => 0, 'out' => $total_salaries_cash],
            ['name' => 'Production Labour (Cash)', 'in' => 0, 'out' => $total_labour_cash],
            ['name' => 'Investment Withdrawals (Cash)', 'in' => 0, 'out' => $total_withdrawals_cash],
        ];

        // B. Contractor Labour Details
        $contractor_details = DB::table('productions')
            ->join('contractors', 'productions.contractor_id', '=', 'contractors.id')
            ->select('contractors.name', 'productions.order_number', 'productions.labour_cost', 'productions.start_date')
            ->where('productions.status', '!=', 'cancelled');
        if ($start_date && $end_date) $contractor_details->whereBetween('productions.start_date', [$start_date, $end_date]);
        $contractor_list = $contractor_details->get();

        // C. Employee Salaries Details
        $salary_details = DB::table('salaries')
            ->join('employees', 'salaries.employee_id', '=', 'employees.id')
            ->select('employees.emp_name', 'salaries.month', 'salaries.total_salary', 'salaries.payment_type', 'salaries.status');
        if ($start_date && $end_date) $salary_details->whereBetween('salaries.month', [Carbon::parse($start_date)->format('Y-m'), Carbon::parse($end_date)->format('Y-m')]);
        $salary_list = $salary_details->get();

        // D. Purchase Supplier Products Details
        $purchase_details = DB::table('purchase_details')
            ->join('purchases', 'purchase_details.purchase_id', '=', 'purchases.id')
            ->join('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
            ->join('products', 'purchase_details.pro_id', '=', 'products.id')
            ->select('suppliers.sup_name', 'products.pro_name', 'purchase_details.pro_qty', 'purchase_details.total_amount', 'purchases.date');
        if ($start_date && $end_date) $purchase_details->whereBetween('purchases.date', [$start_date, $end_date]);
        $purchase_list = $purchase_details->get();

        // E. Purchase Return Details
        $purchase_returns = DB::table('purchase_return_details')
            ->join('purchase_returns', 'purchase_return_details.purchase_return_id', '=', 'purchase_returns.id')
            ->join('suppliers', 'purchase_returns.supplier_id', '=', 'suppliers.id')
            ->join('products', 'purchase_return_details.pro_id', '=', 'products.id')
            ->select('suppliers.sup_name', 'products.pro_name', 'purchase_return_details.pro_qty', 'purchase_return_details.total as return_amount', 'purchase_returns.return_date')
            ->where('purchase_returns.status', 1);
        if ($start_date && $end_date) $purchase_returns->whereBetween('purchase_returns.return_date', [$start_date, $end_date]);
        $return_list = $purchase_returns->get();

        // F. Investment Details
        $investments = DB::table('investments')
            ->leftJoin('investors', 'investments.investor_id', '=', 'investors.id')
            ->select('investments.date', 'investments.invested_amount', 'investments.payment_type', DB::raw('COALESCE(investors.investor_name, "Company") as name'))
            ->where('investments.status', 1);
        if ($start_date && $end_date) $investments->whereBetween('investments.date', [$start_date, $end_date]);
        $investment_list = $investments->get();

        // G. Investment Withdrawals Details
        $withdrawals = DB::table('investment_withdraws')
            ->leftJoin('investors', 'investment_withdraws.investor_id', '=', 'investors.id')
            ->select('investment_withdraws.date', 'investment_withdraws.amount', 'investment_withdraws.payment_type', DB::raw('COALESCE(investors.investor_name, "Company") as name'));
        if ($start_date && $end_date) $withdrawals->whereBetween('investment_withdraws.date', [$start_date, $end_date]);
        $withdrawal_list = $withdrawals->get();

        // H. Sale Details
        $sale_details = DB::table('sales')
            ->join('customers', 'sales.cus_id', '=', 'customers.id')
            ->where('sales.is_deleted', '!=', 1)
            ->select('sales.invoice_date', 'sales.grand_total', 'sales.paid', 'sales.Payment_type', DB::raw('COALESCE(customers.shop_name, customers.name) as customer_name'));
        if ($start_date && $end_date) $sale_details->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        $sale_list = $sale_details->get();


        // --- 4. CALCULATIONS ---

        $total_inflow = $total_sales + $total_investments + $total_other_income + $total_purchase_returns;
        $gross_profit = ($total_sales - $total_sale_returns) - ($total_purchases_accrual - $total_purchase_returns);
        $total_operating_expenses = $total_salaries + $total_expenses + $total_salesman_expenses + $total_labour_cost;
        $net_profit = $gross_profit - $total_operating_expenses;
        
        // Net Cash Flow (In - Out)
        // Out: Purchase Paid + Expenses + Salaries + Salesman Exp + Sale Returns + Withdrawals + Labour
        $total_outflow_cash = $total_purchases_cash + $total_expenses + $total_salaries + $total_salesman_expenses + $total_sale_returns + $total_withdrawals + $total_labour_cost;
        $net_cash_flow = $total_inflow - $total_outflow_cash;


        // Bank Breakdown (Existing)
        $banks = DB::table('banks')
            ->leftJoin('transactions', function($join) use ($start_date, $end_date) {
                $join->on('banks.id', '=', 'transactions.bank_id');
                if ($start_date && $end_date) $join->whereBetween('transactions.created_at', [$start_date, $end_date]);
            })
            ->select('banks.bank_name', DB::raw('SUM(CASE WHEN transactions.transaction_type IN ("deposit","credit") THEN transactions.amount ELSE 0 END) as total_in'), DB::raw('SUM(CASE WHEN transactions.transaction_type IN ("withdraw","debit") THEN transactions.amount ELSE 0 END) as total_out'))
            ->groupBy('banks.id', 'banks.bank_name')
            ->get();

        // Expense Breakdown (Existing)
        $expense_categories = DB::table('expense_details')
            ->join('expenses', 'expense_details.exp_id', '=', 'expenses.id')
            ->join('expense_categories', 'expense_details.exp_cat_id', '=', 'expense_categories.id')
            ->select('expense_categories.exp_cat_name as name', DB::raw('SUM(expense_details.total) as total'))
            ->where('expenses.is_deleted', 0)
            ->when($start_date && $end_date, function($q) use ($start_date, $end_date) {
                return $q->whereBetween('expenses.exp_date', [$start_date, $end_date]);
            })
            ->groupBy('expense_categories.id', 'expense_categories.exp_cat_name')
            ->get();

        return response()->json([
            'status' => 200,
            'summary' => [
                'total_sales' => number_format($total_sales, 2),
                'total_sale_returns' => number_format($total_sale_returns, 2),
                'total_purchases' => number_format($total_purchases_accrual, 2),
                'total_purchase_paid' => number_format($total_purchases_cash, 2),
                'total_production_cost' => number_format($total_production_cost, 2),
                'total_labour_cost' => number_format($total_labour_cost, 2),
                'total_material_cost' => number_format($total_material_cost, 2),
                'total_salaries' => number_format($total_salaries, 2),
                'total_expenses' => number_format($total_expenses, 2),
                'total_salesman_expenses' => number_format($total_salesman_expenses, 2),
                'total_investments' => number_format($total_investments, 2),
                'total_withdrawals' => number_format($total_withdrawals, 2),
                'net_profit' => number_format($net_profit, 2),
                'net_cash_flow' => number_format($net_cash_flow, 2)
            ],
            'breakdowns' => [
                'banks' => $banks,
                'expenses' => $expense_categories,
                'cash_in_hand' => $cash_in_hand_details
            ],
            'details' => [
                'contractor_labour' => $contractor_list,
                'salaries' => $salary_list,
                'purchase_products' => $purchase_list,
                'purchase_returns' => $return_list,
                'investments' => $investment_list,
                'withdrawals' => $withdrawal_list,
                'sales' => $sale_list
            ]
        ]);
    }
}