<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerLedger;
use App\Models\Investment;
use App\Models\PurchaseDetail;
use App\Models\Sale;
use App\Models\SaleDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\Stock;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SaleController extends Controller
{
    

    public function load_sales(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        // FIXED: Properly join and select all sales including converted from Order Booker
        $query = DB::table('sales')
        ->join('customers', 'sales.cus_id', '=', 'customers.id')
        ->join('users', 'sales.saleman_id', '=', 'users.id')
        ->select([
            'sales.*',
            'customers.name as cus_name',
            DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),
            'sales.id as id',
            'users.name as saleman_name',
        ])
        ->where('sales.is_deleted', '!=', 1)
        ->where('sales.is_active', '=', 1); // FIXED: Only show active sales

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin'])) {
            $query->where('sales.saleman_id', Auth::id());
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where(function($q) use ($search) {
                $q->where('customers.name', 'like', '%' . $search . '%')
                  ->orWhere('customers.shop_name', 'like', '%' . $search . '%')
                  ->orWhere('sales.detail', 'like', '%' . $search . '%');
            });
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('sales.is_active', '=', $type);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $sale = $query->get();

        $all_data = [];
        foreach ($sale as $data) {


            $btn = '<td class="text-right">';
            if (Auth::user()->can('view_sales')) {
                $btn .= '<button id="view_btn" data-vid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="View" style="margin-right: 3px"><i class="fa-duotone fa-arrows-to-eye"></i></button>';
            }
            if (Auth::user()->can('update_sales')) {
                $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_sales')) {
                $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            // FIXED: Show if converted from Order Booker
            $source = '';
            if (strpos($data->detail, 'Order Booker') !== false || strpos($data->detail, 'OB-') !== false) {
                $source = '<span class="label label-purple" style="font-size: 10px;">From Order Booker</span> ';
            }

            $all_data[] = [
                'id'            => $data->id,
                'trn_date'      => Carbon::parse($data->invoice_date)->format('d-m-Y'),
                'shop_name'     => $data->shop_name,
                'cus_name'      => $source . $data->cus_name,
                'saleman_name'  => $data->saleman_name,
                'subtotal'      => number_format($data->subtotal, 0),
                'dis_percent'   => number_format($data->dis_percent, 0),
                'dis_amount'    => number_format($data->dis_amount, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'paid'          => number_format($data->paid, 0),
                'due'           => number_format($data->due, 0),
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }

    public function getSaleProduct(Request $request)
    {
        $prodid = $request->prodid;

        $result = PurchaseDetail::where('pro_id', $prodid)
            ->whereRaw('pro_qty - COALESCE(sale_qty, 0) > 0')
            ->orderBy('id', 'asc')
            ->get();

        if ($result->count() > 0) {
            $data = $result->map(function ($row) {
                return [
                    'pur_id' => $row->pur_id,
                    'batch' => $row->batch_no,
                    'stock' => $row->pro_qty - ($row->sale_qty ?? 0),
                    'purchase_price' => $row->single_price,
                    'sale_price' => $row->sale_price,
                ];
            });
            return response()->json($data);
        } else {
            return response()->json(['message' => 'No Record Found'], 404);
        }
    }

    public function getBatchData(Request $request)
    {
        $batch = $request->batch;
        $prodid = $request->prodid;

        // DB se record nikalo
        $row = DB::table('purchase_details')
            ->where('batch_no', $batch)
            ->where('pro_id', $prodid)
            ->first();

        if ($row) {
            $data = [
                "purchase_price" => $row->single_price,
                "sale_price" => $row->sale_price,
                "pro_qty" => $row->pro_qty,
                "stock" => $row->pro_qty - ($row->sale_qty ?? 0)
            ];

            return response()->json($data); // JSON response
        } else {
            return response()->json(["error" => "No Record Found"], 404);
        }
    }


    public function insert_sale(Request $request)
    {

        $sale = new Sale();
        $sale->invoice_date             = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $sale->cus_id                   = $request->input('cus_id');
        $sale->bank_id                  = $request->input('bank_id');
        $sale->Payment_type             = $request->input('payment_type');
        $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $sale->detail                   = $request->input('detail');
        $sale->subtotal                 = $request->input('subtotal');
        $sale->dis_percent              = $request->input('total_dis_per');
        $sale->dis_amount               = $request->input('total_dis_amt');
        $sale->grand_total              = $request->input('grand_total');
        $sale->paid                     = $request->input('paid');
        $sale->due                      = $request->input('due');
        $sale->is_active                = 1;
        $sale->is_deleted               = 0;
        $sale->created_by               = Auth::id();
        $sale->save();

        // jab sale ho gi bank balance increase ho ga or hand cash 

        if ($request->payment_type === "Cash") {
            // hand_cash column
            DB::table('hand_cashes')->increment('cash_amount', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Cash',
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Sale',
                'source_id'         => $sale->id,
                'description'       => 'Sale Payment Received',
                'created_at'        => now(),
            ]);

        } elseif ($request->payment_type === "Bank") {
            
            DB::table('banks')->where('id', $request->bank_id)
            ->increment('bank_balance', $request->paid);

            DB::table('transactions')->insert([
                'payment_type'      => 'Bank',
                'bank_id'           => $request->input('bank_id'),
                'amount'            => $request->input('paid'),
                'transaction_type'  => 'deposit',
                'source_type'       => 'Sale',
                'source_id'         => $sale->id,
                'description'       => 'Sale Payment Received',
                'created_at'        => now(),
            ]);

        }


        // Get the last inserted sale ID
        $sale_id = $sale->id;
        // Extract data from the request
        $pro_ids    = $request->input('pro_id');
        $pro_sku    = $request->input('pro_sku');
        $batch_no   = $request->input('batch_no');

        $purchase_price   = $request->input('pro_cost');
        $pro_sale   = $request->input('pro_sale');
        $pro_qty    = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt    = $request->input('dis_amt');
        $total      = $request->input('total');

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $detail                     = new SaleDetail();
            $detail->sale_id            = $sale_id;
            $detail->pro_id             = $pro_id;
            $detail->pro_sku            = $pro_sku[$index] ?? null;
            $detail->batch_no           = $batch_no[$index] ?? null;
            $detail->purchase_price     = $purchase_price[$index] ?? null;
            $detail->date               = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
            $detail->sale_price         = $pro_sale[$index];
            $detail->dis_per            = $dis_percent[$index];
            $detail->dis_amount         = $dis_amt[$index];
            $detail->pro_qty            = $pro_qty[$index];
            $detail->total              = $total[$index];
            $detail->is_active          = 1;
            $detail->is_deleted         = 0;
            $detail->created_by         = Auth::id();
            $detail->updated_by         = Auth::id();
            $detail->save();

            // PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
            ->where('batch_no', $batch_no[$index])
            ->first()?->increment('sale_qty', $pro_qty[$index]);

            // Update stock
            $salemanstock                  = new SalemanStock();
            $salemanstock->pro_id          = $pro_id;
            $salemanstock->sale_id         = $sale_id;
            $salemanstock->sale_price      = $pro_sale[$index];
            $salemanstock->purchase_price  = $purchase_price[$index];
            $salemanstock->stock_out_qty   = $pro_qty[$index];
            $salemanstock->invoice_date    = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
            $salemanstock->salesman_id     = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $salemanstock->adjustment      = 'Sale';
            $salemanstock->save();

            // Sale & Purchase
            $saleAmount = $total[$index]; // qty * sale price
            $purchaseAmount = $purchase_price[$index] * $pro_qty[$index]; // qty * purchase price

            // Actual profit
            $actualProfit = $saleAmount - $purchaseAmount;

            // Investments fetch karo
            $investments = Investment::where('pro_id', $pro_id)
                ->where('remaining_amount', '>', 0)
                ->get();

            foreach ($investments as $investment) {

                // Investor ka share jitne profit se adjust ho sakta hai
                $adjustableProfit = min($actualProfit, $investment->remaining_amount);

                if ($adjustableProfit > 0) {
                    // Investor ka profit = adjustableProfit ka % according to investor's share
                    $profit = ($adjustableProfit * $investment->profit_percentage) / 100;

                    DB::table('investment_profits')->insert([
                        'investment_id' => $investment->id,
                        'investor_id'   => $investment->investor_id,
                        'sale_id'       => $sale->id,
                        'pro_id'        => $pro_id,
                        'profit_amount' => $profit,
                        'created_at'    => now(),
                    ]);

                    // Investment table update karo (remaining_amount minus karo by adjustableProfit)
                    $investment->remaining_amount -= $adjustableProfit;
                    if ($investment->remaining_amount < 0) {
                        $investment->remaining_amount = 0; // negative na ho
                    }
                    $investment->save();

                    // Actual profit bhi kam karna zaroori hai
                    $actualProfit -= $adjustableProfit;
                }
            }



        }

        // Extract customer  ledger data
        $products    = implode(",", $request->input('pro_id'));
        $pro_qty1     = implode(",", $request->input('pro_qty'));
        
        // Insert into CustomerLedger table
        $supplier_ledger = new CustomerLedger();
        $supplier_ledger->cus_id                    = $request->input('cus_id');
        $supplier_ledger->sale_id                   = $sale->id;
        $supplier_ledger->invoice_date              = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $supplier_ledger->detail                    = $request->input('detail');
        $supplier_ledger->pro_id                    = $products;
        $supplier_ledger->pro_qty                   = $pro_qty1;
        $supplier_ledger->transaction_type          = $request->input('payment_type'); // FIXED: Use payment_type (Cash/Bank) instead of 'Sale'
        $supplier_ledger->bank_id                   = $request->input('bank_id');
        $supplier_ledger->total                     = $request->input('grand_total');
        $supplier_ledger->adjustment                = 'Sale';
        // FIXED: For sales, debit is 0, credit is paid amount
        $supplier_ledger->debit                     = 0;

        // Handle payment type for credit/bank columns
        if ($request->input('payment_type') === "Cash") {
            $supplier_ledger->credit                = $request->input('paid');
            $supplier_ledger->bank                  = 0;
        } elseif ($request->input('payment_type') === "Bank") {
            $supplier_ledger->credit                = 0;
            $supplier_ledger->bank                  = $request->input('paid');
        } else {
            $supplier_ledger->credit                = $request->input('paid');
            $supplier_ledger->bank                  = 0;
        }

        $supplier_ledger->created_by                = Auth::id();
        $supplier_ledger->updated_by                = Auth::id();
        $supplier_ledger->save();

        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id                    = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $saleman_ledger->cus_id                   = $request->input('cus_id');
        $saleman_ledger->sale_id                   = $sale->id;
        $saleman_ledger->invoice_date              = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $saleman_ledger->detail                    = $request->input('detail');
        $saleman_ledger->pro_id                    = $products;
        $saleman_ledger->pro_qty                   = $pro_qty1;
        $saleman_ledger->transaction_type          = $request->input('payment_type');
        $saleman_ledger->bank_id                    = $request->input('bank_id');
        $saleman_ledger->total                     = $request->input('grand_total');
        // $saleman_ledger->credit                    = $request->input('paid');
        $saleman_ledger->debit                     = $request->input('due');
        $saleman_ledger->adjustment                = 'Sale';

        // Handle payment type
        if ($request->input('payment_type') === "Cash") {
            $saleman_ledger->credit   = $request->input('paid');
            $saleman_ledger->bank  = 0;

        } elseif ($request->input('payment_type') === "Bank") {
            $saleman_ledger->credit   = 0;
            $saleman_ledger->bank  = $request->input('paid');
        }


        $saleman_ledger->save();

       

        return response()->json(['status' => 200, 'message' => 'Sale added successfully.']);
    }

    public function view_sale($id)
    {
        $sale = DB::table('sales')
            ->leftJoin('customers', 'sales.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'sales.saleman_id', '=', 'users.id')
            ->select(
                'sales.*',
                'customers.name as customer_name',
                'customers.shop_name',
                'customers.phone',
                'customers.address',
                'users.name as salesman_name',
                'sales.id as id'
            )
            ->where('sales.id', $id)
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found'
            ], 404);
        }

        $sale_details = DB::table('sale_details')
            ->join('products', 'sale_details.pro_id', '=', 'products.id')
            ->select(
                'sale_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('sale_details.sale_id', '=', $id)
            ->get();

        $products = [];
        foreach ($sale_details as $index => $detail) {
            $products[] = [
                'no' => $index + 1,
                'pro_id' => $detail->pro_id,
                'pro_name' => $detail->pro_name,
                'sale_price' => $detail->sale_price,
                'pro_qty' => $detail->pro_qty,
                'dis_per' => $detail->dis_per,
                'dis_amount' => $detail->dis_amount,
                'total' => $detail->total,
            ];
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $products,
        ]);
    }

    public function edit_sale($id, Request $request)
    {
        $salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();

        // Retrieve the sale with customer details
        $sale = DB::table('sales')
            ->leftJoin('customers', 'sales.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'sales.saleman_id', '=', 'users.id')
            ->leftJoin('banks', 'sales.bank_id', '=', 'banks.id')
            ->select(
                'sales.*',
                'customers.name as customer_name',
                'banks.bank_name',
                'users.name as salesman_name',
                'sales.id as id'
            )
            ->where('sales.id', $id)
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found'
            ], 404);
        }

        // Retrieve sale details
        $sale_details = DB::table('sale_details')
            ->join('products', 'sale_details.pro_id', '=', 'products.id')
            ->select(
                'sale_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('sale_details.sale_id', '=', $id)
            ->get();

        // Calculate stock for each product
        foreach ($sale_details as $detail) {
            $stock = DB::table('saleman_stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->where('salesman_id', '=', $salesman_id)
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $sale_details
        ]);
    }

    public function update_sale(Request $request)
    {
        // Find the sale record by ID
        $sale = Sale::find($request->input('sale_id'));

        if (!$sale) {
            return response()->json(['status' => 404, 'message' => 'Sale not found.']);
        }

        // Reverse previous payment effect
        if ($sale->Payment_type === "Cash") {
            DB::table('hand_cashes')->decrement('cash_amount', $sale->paid);
        } elseif ($sale->Payment_type === "Bank" && $sale->bank_id) {
            DB::table('banks')->where('id', $sale->bank_id)
                ->decrement('bank_balance', $sale->paid);
        }

        // Update the main sale record
        $sale->invoice_date             = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $sale->cus_id                   = $request->input('cus_id');
        $sale->Payment_type             = $request->input('payment_type');
        $sale->bank_id                  = $request->input('bank_id');
        $sale->detail                   = $request->input('detail');
        $sale->subtotal                 = $request->input('subtotal');
        $sale->dis_percent              = $request->input('total_dis_per');
        $sale->dis_amount               = $request->input('total_dis_amt');
        $sale->grand_total              = $request->input('grand_total');
        $sale->paid                     = $request->input('paid');
        $sale->due                      = $request->input('due');
        $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $sale->updated_by = Auth::id();
        $sale->save();

        // Apply new payment
        if ($request->payment_type === "Cash") {
            DB::table('hand_cashes')->increment('cash_amount', $request->paid);
            Transaction::where('source_type', 'Sale')
                ->where('source_id', $sale->id)
                ->delete();
            DB::table('transactions')->insert([
                'payment_type' => 'Cash',
                'amount' => $request->input('paid'),
                'transaction_type' => 'deposit',
                'source_type' => 'Sale',
                'source_id' => $sale->id,
                'description' => 'Sale Payment Updated',
                'created_at' => now(),
            ]);
        } elseif ($request->payment_type === "Bank") {
            DB::table('banks')->where('id', $request->bank_id)
                ->increment('bank_balance', $request->paid);
            Transaction::where('source_type', 'Sale')
                ->where('source_id', $sale->id)
                ->delete();
            DB::table('transactions')->insert([
                'payment_type' => 'Bank',
                'bank_id' => $request->input('bank_id'),
                'amount' => $request->input('paid'),
                'transaction_type' => 'deposit',
                'source_type' => 'Sale',
                'source_id' => $sale->id,
                'description' => 'Sale Payment Updated',
                'created_at' => now(),
            ]);
        }

        // Delete old sale details
        SaleDetail::where('sale_id', $sale->id)->delete();
        SalemanStock::where('sale_id', $sale->id)->delete();
        CustomerLedger::where('sale_id', $sale->id)->delete();
        SalemanLedger::where('sale_id', $sale->id)->delete();
        DB::table('investment_profits')->where('sale_id', $sale->id)->delete();

        // Reinsert the sale details
        $pro_ids = $request->input('pro_id');
        $pro_sku = $request->input('pro_sku');
        $batch_no = $request->input('batch_no');
        $purchase_price = $request->input('pro_cost');
        $pro_sale = $request->input('pro_sale');
        $pro_qty = $request->input('pro_qty');
        $dis_percent = $request->input('dis_percent');
        $dis_amt = $request->input('dis_amt');
        $total = $request->input('total');

        foreach ($pro_ids as $index => $pro_id) {
            $detail = new SaleDetail();
            $detail->sale_id = $sale->id;
            $detail->pro_id = $pro_id;
            $detail->pro_sku = $pro_sku[$index] ?? null;
            $detail->batch_no = $batch_no[$index] ?? null;
            $detail->purchase_price = $purchase_price[$index] ?? null;
            $detail->date = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
            $detail->sale_price = $pro_sale[$index];
            $detail->dis_per = $dis_percent[$index];
            $detail->dis_amount = $dis_amt[$index];
            $detail->pro_qty = $pro_qty[$index];
            $detail->total = $total[$index];
            $detail->is_active = 1;
            $detail->is_deleted = 0;
            $detail->created_by = Auth::id();
            $detail->updated_by = Auth::id();
            $detail->save();

            // PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
                ->where('batch_no', $batch_no[$index])
                ->first()?->increment('sale_qty', $pro_qty[$index]);

            // Update stock
            $salemanstock = new SalemanStock();
            $salemanstock->pro_id = $pro_id;
            $salemanstock->sale_id = $sale->id;
            $salemanstock->sale_price = $pro_sale[$index];
            $salemanstock->purchase_price = $purchase_price[$index];
            $salemanstock->stock_out_qty = $pro_qty[$index];
            $salemanstock->invoice_date = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
            $salemanstock->salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $salemanstock->adjustment = 'Sale';
            $salemanstock->save();

            // Handle investments
            $saleAmount = $total[$index];
            $purchaseAmount = $purchase_price[$index] * $pro_qty[$index];
            $actualProfit = $saleAmount - $purchaseAmount;

            $investments = Investment::where('pro_id', $pro_id)
                ->where('remaining_amount', '>', 0)
                ->get();

            foreach ($investments as $investment) {
                $adjustableProfit = min($actualProfit, $investment->remaining_amount);

                if ($adjustableProfit > 0) {
                    $profit = ($adjustableProfit * $investment->profit_percentage) / 100;

                    DB::table('investment_profits')->insert([
                        'investment_id' => $investment->id,
                        'investor_id' => $investment->investor_id,
                        'sale_id' => $sale->id,
                        'pro_id' => $pro_id,
                        'profit_amount' => $profit,
                        'created_at' => now(),
                    ]);

                    $investment->remaining_amount -= $adjustableProfit;
                    if ($investment->remaining_amount < 0) {
                        $investment->remaining_amount = 0;
                    }
                    $investment->save();

                    $actualProfit -= $adjustableProfit;
                }
            }
        }

        // Extract customer ledger data
        $products = implode(",", $request->input('pro_id'));
        $pro_qty1 = implode(",", $request->input('pro_qty'));

        // Insert into CustomerLedger table
        $supplier_ledger = new CustomerLedger();
        $supplier_ledger->cus_id = $request->input('cus_id');
        $supplier_ledger->sale_id = $sale->id;
        $supplier_ledger->invoice_date = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $supplier_ledger->detail = $request->input('detail');
        $supplier_ledger->pro_id = $products;
        $supplier_ledger->pro_qty = $pro_qty1;
        $supplier_ledger->transaction_type = $request->input('payment_type');
        $supplier_ledger->bank_id = $request->input('bank_id');
        $supplier_ledger->total = $request->input('grand_total');
        $supplier_ledger->adjustment = 'Sale';
        $supplier_ledger->debit = 0;

        if ($request->input('payment_type') === "Cash") {
            $supplier_ledger->credit = $request->input('paid');
            $supplier_ledger->bank = 0;
        } elseif ($request->input('payment_type') === "Bank") {
            $supplier_ledger->credit = 0;
            $supplier_ledger->bank = $request->input('paid');
        } else {
            $supplier_ledger->credit = $request->input('paid');
            $supplier_ledger->bank = 0;
        }

        $supplier_ledger->created_by = Auth::id();
        $supplier_ledger->updated_by = Auth::id();
        $supplier_ledger->save();

        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $saleman_ledger->cus_id = $request->input('cus_id');
        $saleman_ledger->sale_id = $sale->id;
        $saleman_ledger->invoice_date = Carbon::parse($request->input('sale_date'))->format('Y-m-d');
        $saleman_ledger->detail = $request->input('detail');
        $saleman_ledger->pro_id = $products;
        $saleman_ledger->pro_qty = $pro_qty1;
        $saleman_ledger->transaction_type = $request->input('payment_type');
        $saleman_ledger->bank_id = $request->input('bank_id');
        $saleman_ledger->total = $request->input('grand_total');
        $saleman_ledger->debit = $request->input('due');
        $saleman_ledger->adjustment = 'Sale';

        if ($request->input('payment_type') === "Cash") {
            $saleman_ledger->credit = $request->input('paid');
            $saleman_ledger->bank = 0;
        } elseif ($request->input('payment_type') === "Bank") {
            $saleman_ledger->credit = 0;
            $saleman_ledger->bank = $request->input('paid');
        }

        $saleman_ledger->save();

        return response()->json(['status' => 200, 'message' => 'Sale updated successfully.']);
    }



    public function delete_sale($id)
    {
        $data = Sale::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found.'
            ]);
        }

        // Reverse entry using saved data
        if ($data->Payment_type === "Cash") {
            DB::table('hand_cashes')->decrement('cash_amount', $data->paid);

        } elseif ($data->Payment_type === "Bank" && $data->bank_id) {
            DB::table('banks')->where('id', $data->bank_id)
                ->decrement('bank_balance', $data->paid);
        }

        // Delete sale details first
        $data->saleDetails()->delete();
        $data->salemanStock()->delete();
        $data->customerLedger()->delete();
        $data->salemanLedger()->delete();
        $data->investorLedger()->delete();
        // $data->stock()->delete();

        // Related Transactions delete
        Transaction::where('source_type', 'Sale')
            ->where('source_id', $data->id)
            ->delete();


        // Delete the sale
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Sale and its details deleted successfully.'
        ]);
    }


    public function todaysale(Request $request)
    {
        $today = now()->toDateString();

        $today_sales = DB::table('sales')
            ->whereDate('invoice_date', $today)
            ->where('is_deleted', '!=', 1)
            ->whereNotNull('grand_total')
            ->sum('grand_total');

        return response()->json(['today_sales' => 'Rs. ' . number_format($today_sales, 2)]);
    }


    public function totalsales()
    {
        $totalsales = DB::table('sales')
                        ->where('is_deleted', '!=', 1)
                        ->whereNotNull('grand_total')
                        ->sum('grand_total');

        return response()->json([
            'total_sales' => 'Rs. ' . number_format($totalsales, 2)
        ]);
    }

    public function weeklySales()
    {
        $startDate = Carbon::now()->subDays(7)->toDateString();
        $endDate = Carbon::now()->toDateString();

        $weeklySales = Sale::whereBetween('invoice_date', [$startDate, $endDate])
                        ->where('is_deleted', '!=', 1)
                        ->sum('grand_total');

        return response()->json(['weekly_sales' => 'Rs. ' . number_format($weeklySales, 2)]);
    }


    public function monthlySales()
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $monthlySales = DB::table('sales')
            ->whereYear('invoice_date', $currentYear)
            ->whereMonth('invoice_date', $currentMonth)
            ->where('is_deleted', '!=', 1)
            ->sum('grand_total');

        return response()->json([
            'monthly_sales' => 'Rs. ' .  number_format($monthlySales, 2)
        ]);
    }


}