<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\NonServiceStock;
use App\Models\PurchaseDetail;
use App\Models\SalemanLedger;
use App\Models\SalemanStock;
use App\Models\SaleReturn;
use App\Models\SaleReturnDetail;
use App\Models\Stock;
use App\Models\SupplierLedger;
use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SaleReturnController extends Controller
{
    

    public function load_sale_return(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        // Build the query with potential filters
        $query = DB::table('sale_returns')
        ->join('customers', 'sale_returns.cus_id', '=', 'customers.id')
        ->join('users', 'sale_returns.saleman_id', '=', 'users.id')
        ->leftJoin('users as updater', 'sale_returns.updated_by', '=', 'updater.id') // 👈 extra join for updated_by
        ->select([
            'sale_returns.*',
            'customers.name as cus_name',
            DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),

            'sale_returns.id as id',
            'sale_returns.created_at',
            'users.name as saleman_name',
            'updater.name as updated_by_name', // 👈 new column
            DB::raw('(SELECT SUM(CAST(pro_qty AS DECIMAL(10,2))) FROM sale_return_details WHERE sale_return_details.sale_return_id = sale_returns.id) as total_qty')
        ]);

        // Role-based filter
        if (!Auth::user()->hasRole(['Admin', 'Super Admin', 'Store man'])) {
            // FIXED: Salesmen should see all their returns (pending, approved, rejected)
            $query->where('sale_returns.saleman_id', Auth::id());
        }

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('trn_date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('customers.name', 'like', '%' . $search . '%');
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $sale = $query->get();
 
        $all_data = [];
        $sr_no = 1; // 👈 counter start
        foreach ($sale as $data) {


            $btn = '<td class="text-right">';
           
                if (Auth::user()->can('update_sale_return')) {
                    $btn .= '<button id="edit_btn" data-eid="'.$data->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
                }
                if (Auth::user()->can('delete_sale_return')) {
                    $btn .= '<button id="delete_btn" data-did="'.$data->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
                }
            $btn .= '</td>';

            // Format return_type for display
            $return_type_display = '----';
            if (!empty($data->return_type)) {
                $return_type_display = ucwords(str_replace('_', ' ', $data->return_type));
            }

            $all_data[] = [
                // 'sr_no'         => $sr_no++,
                'id'            => $data->id,
                'date'          => Carbon::parse($data->created_at)->format('d-m-Y'),
                'shop_name'      => $data->shop_name,
                'saleman_name'  => $data->saleman_name,
                'total_qty'     => number_format($data->total_qty ?? 0, 0),
                'return_type'   => $return_type_display,
                'updated_by'  => $data->updated_by_name ?? '----',
                'subtotal'      => number_format($data->subtotal, 0),
                'grand_total'   => number_format($data->grand_total, 0),
                'red_percent'   => number_format($data->red_percent, 0),
                'red_amount'    => number_format($data->red_amount, 0),
                'paid'          => number_format($data->paid, 0),
                'status'              => !empty($data->status) ? $data->status : '---',
                // 'due'           => number_format($data->due, 0),
                'btn'           => $btn
            ];
            
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data
        ];

        return response()->json($data);
    }


    public function insert_sale_return(Request $request)
    {

        $sale = new SaleReturn();
        $sale->invoice_date             = Carbon::parse($request->input('return_date'))->format('Y-m-d');
        $sale->cus_id                   = $request->input('cus_id');
        $sale->bank_id                  = $request->input('bank_id');
        $sale->Payment_type             = $request->input('payment_type');
        $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $sale->detail                   = $request->input('detail');
        $sale->return_type              = $request->input('return_type');
        $sale->status                   = 'pending'; // Set default status to pending
        $sale->subtotal                 = $request->input('subtotal');
        $sale->red_percent              = $request->input('total_dis_per');
        $sale->red_amount               = $request->input('total_dis_amt');
        $sale->grand_total              = $request->input('grand_total');
        $sale->paid                     = $request->input('paid');
        $sale->due                      = $request->input('due');
        $sale->created_by               = Auth::id();
        $sale->save();


        // Extract customer  ledger data
        $products    = implode(",", $request->input('pro_id'));
        $pro_qty1     = implode(",", $request->input('pro_qty'));

        // Get the last inserted Book Test ID
        $sale_return_id = $sale->id;
        // Extract data from the request
        $pro_ids    = $request->input('pro_id');
        $pro_sku    = $request->input('pro_sku');
        $batch_no    = $request->input('batch_no');
        $qty_sale    = $request->input('qty_sale');


        $purchase_price   = $request->input('pro_cost');
        $pro_sale   = $request->input('pro_sale');
        $pro_qty    = $request->input('pro_qty');
        $red_percent = $request->input('dis_percent');
        $red_amt    = $request->input('dis_amt');
        $total      = $request->input('total');

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $product_stock                          = new SaleReturnDetail();
            $product_stock->sale_return_id          = $sale_return_id;
            $product_stock->pro_id                  = $pro_id;
            $product_stock->pro_sku       = $pro_sku[$index];
            $product_stock->batch_no       = $batch_no[$index];
            $product_stock->qty_sale       = $qty_sale[$index];


            $product_stock->sale_return_price       = $pro_sale[$index];
            $product_stock->red_per                 = $red_percent[$index];
            $product_stock->red_amount              = $red_amt[$index];
            $product_stock->pro_qty                 = $pro_qty[$index];
            $product_stock->total                   = $total[$index];
            $product_stock->save();

        }

        // Insert into Customer Ledger table
        $ledger = new CustomerLedger();
        $ledger->cus_id                 = $request->input('cus_id');
        $ledger->sale_return_id         = $sale_return_id;
        $ledger->invoice_date           = Carbon::parse($request->input('return_date'))->format('Y-m-d');
        $ledger->detail                 = $request->input('detail') ?: 'Sale Return';
        $ledger->pro_id                 = $products;
        $ledger->pro_qty                = $pro_qty1;
        $ledger->transaction_type       = 'Sale_return'; // FIXED: Changed from 'Sale Return' to match ENUM
        $ledger->adjustment             = 'Sale Return';
        $ledger->bank_id                = $request->input('bank_id');
        // CRITICAL FIX: Set total as NEGATIVE to reverse the original sale
        $ledger->total                  = -$request->input('grand_total');
        $ledger->return_amount          = $request->input('grand_total');

        // CRITICAL FIX: "Paid" applies to customer balance, not cash refund
        // debit = 0 means no cash is refunded, return just reduces balance
        $ledger->debit                  = 0;
        $ledger->credit                 = 0;
        $ledger->bank                   = 0;
        $ledger->save();

        // Insert into Salesman Ledger table
        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $saleman_ledger->cus_id                    = $request->input('cus_id');
        $saleman_ledger->sale_return_id            = $sale_return_id;
        $saleman_ledger->invoice_date              = Carbon::parse($request->input('return_date'))->format('Y-m-d');
        $saleman_ledger->detail                    = $request->input('detail') ?: 'Sale Return';
        $saleman_ledger->pro_id                    = $products;
        $saleman_ledger->pro_qty                   = $pro_qty1;
        $saleman_ledger->transaction_type          = 'Sale_return'; // FIXED: Match ENUM value
        $saleman_ledger->bank_id                   = $request->input('bank_id');
        $saleman_ledger->adjustment                = 'Sale Return';
        $saleman_ledger->return_paid               = $request->input('paid');
        $saleman_ledger->debit                     = 0;
        $saleman_ledger->credit                    = 0;
        $saleman_ledger->bank                      = 0;
        $saleman_ledger->save();

        return response()->json([
            'status'=>200
        ]);


    }


    public function edit_sale_return($id, Request $request)
    {
        $salesman_id = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();

        // Retrieve the sale with customer details
        $sale = DB::table('sale_returns')
            ->leftJoin('customers', 'sale_returns.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'sale_returns.saleman_id', '=', 'users.id')
            ->leftJoin('banks', 'sale_returns.bank_id', '=', 'banks.id')
            ->select(
                'sale_returns.*',
                'customers.name as customer_name',
                DB::raw('COALESCE(customers.shop_name, customers.company_name, customers.name) as customer_business_name'),
                'banks.bank_name',
                'users.name as salesman_name',
                'sale_returns.id as id'
            )
            ->where('sale_returns.id', $id)
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found'
            ], 404);
        }

        // Retrieve sale details for the specific sale
        $sale_details = DB::table('sale_return_details')
            ->join('products', 'sale_return_details.pro_id', '=', 'products.id')
            ->select(
                'sale_return_details.*',
                'products.pro_name',
                'products.id as pro_id'
            )
            ->where('sale_return_details.sale_return_id', '=', $id)
            ->get();

        // Calculate stock for each product (filtered by selected salesman)
        foreach ($sale_details as $detail) {
            $stock = DB::table('saleman_stocks')
                ->selectRaw('SUM(stock_in_qty) as stock_in_qty, SUM(stock_out_qty) as stock_out_qty')
                ->where('pro_id', '=', $detail->pro_id)
                ->where('salesman_id', '=', $salesman_id) // <-- Filter by selected salesman
                ->first();

            $detail->pro_stock = ($stock->stock_in_qty ?? 0) - ($stock->stock_out_qty ?? 0);
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $sale_details
        ]);
    }
    


    public function view_sale_return($id)
    {
        $sale_return = DB::table('sale_returns')
            ->leftJoin('customers', 'sale_returns.cus_id', '=', 'customers.id')
            ->select(
                'sale_returns.*',
                'customers.name',
                'customers.phone',
                'customers.address',
                'customers.shop_name',
                DB::raw('COALESCE(customers.shop_name, customers.company_name, customers.name) as customer_business_name'),
                'sale_returns.id as id'
            )
            ->where('sale_returns.id', $id)
            ->first();

        if (!$sale_return) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale Return not found'
            ], 404);
        }

        $all_customers = DB::table('customers')->select('id', 'name')->get();

        $sale_details = DB::table('sale_return_details')
        ->select('*')->where('sale_return_id', '=', $id)
        ->get();

        $products = [];

        foreach($sale_details AS $index => $pros){

            $pro = DB::table('products')->select('*')
            ->where('id', '=', $pros->pro_id)->first();

            if ($pro) {
                $products[] = [
                    'no'         => $index+1,
                    'pro_id'     => $pro->id,
                    'pro_name'   => $pro->pro_name,
                    'sale_price' => $pros->sale_return_price,
                    'pro_qty'    => $pros->pro_qty,
                    'dis_per'    => $pros->red_per,
                    'dis_amount' => $pros->red_amount,
                    'total'      => $pros->total,
                ];
            }
        }

        return response()->json([
            'status' => 200,
            'sale_return' => $sale_return,
            'products' => $products,
            'all_customers' => $all_customers,
        ]);
    }


    public function invoice_print()
    {
        // Fetch the last inserted sale
        $sale = DB::table('sale_returns')
            ->leftJoin('customers', 'sale_returns.cus_id', '=', 'customers.id')
            ->leftJoin('users', 'sale_returns.saleman_id', '=', 'users.id')
            ->select(
                'sale_returns.*',
                'customers.name as customer_name',
                'users.name as salesman_name',
                'sale_returns.id as sale_id'
            )
            ->orderBy('sale_returns.id', 'desc') // Get the most recent sale
            ->first();

        if (!$sale) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale not found'
            ], 404);
        }

        // Fetch the details for the sale
        $sale_details = DB::table('sale_return_details')
            ->where('sale_return_id', $sale->id)
            ->get();

        $products = [];

        foreach ($sale_details as $index => $detail) {
            $product = DB::table('products')
                ->where('id', $detail->pro_id)
                ->first();

            if ($product) {
                $products[] = [
                    'no'        => $index + 1,
                    'pro_id'    => $product->id,
                    'pro_name'  => $product->pro_name,
                    'sale_return_price'=> $detail->sale_return_price,
                    'pro_qty'   => $detail->pro_qty,
                    'red_per'   => $detail->red_per,
                    'red_amount'=> $detail->red_amount,
                    'total'     => $detail->total,
                ];
            }
        }

        return response()->json([
            'status' => 200,
            'sale' => $sale,
            'products' => $products,
        ]);
    }

    

    public function update_sale_return(Request $request)
    {
        // PERFORMANCE FIX: Wrap everything in a database transaction for speed and data integrity
        DB::beginTransaction();

        try {
            // Find the sale record by ID
            $sale = SaleReturn::find($request->input('sale_return_id'));

            if (!$sale) {
                DB::rollBack();
                return response()->json(['status' => 404, 'message' => 'Sale not found.']);
            }

            // PERFORMANCE FIX: Only process payment changes if status is being approved
            $oldStatus = $sale->status;
            $newStatus = $request->input('order_status');
            $isBeingApproved = ($oldStatus !== 'approved' && $newStatus === 'approved');

            // Update the main sale record
            $sale->invoice_date             = Carbon::parse($request->input('return_date'))->format('Y-m-d');
            $sale->cus_id                   = $request->input('cus_id');
            $sale->bank_id                  = $request->input('bank_id');
            $sale->Payment_type             = $request->input('payment_type');
            $sale->saleman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
            $sale->detail                   = $request->input('detail');
            $sale->return_type              = $request->input('return_type');
            $sale->subtotal                 = $request->input('subtotal');
            $sale->red_percent              = $request->input('total_dis_per');
            $sale->red_amount               = $request->input('total_dis_amt');
            $sale->grand_total              = $request->input('grand_total');
            $sale->paid                     = $request->input('paid');
            $sale->due                      = $request->input('due');
            $sale->status                   = $newStatus;
            $sale->updated_by               = Auth::id();
            $sale->save();

            // Delete old sale details for this sale (using whereIn for better performance)
            SaleReturnDetail::where('sale_return_id', $sale->id)->delete();
            SalemanStock::where('sale_return_id', $sale->id)->delete();
            CustomerLedger::where('sale_return_id', $sale->id)->delete();
            SalemanLedger::where('sale_return_id', $sale->id)->delete();
            Stock::where('sale_return_id', $sale->id)->delete();
            NonServiceStock::where('sale_return_id', $sale->id)->delete();
            Transaction::where('source_type', 'Sale Return')
                ->where('source_id', $sale->id)
                ->delete();

            // DISABLED: Sale returns apply to customer balance only, no cash/bank refunds
            // The "Paid" field represents amount applied to customer's outstanding balance
            // No actual cash or bank funds are refunded to the customer
            //
            // if ($isBeingApproved && $request->paid > 0) {
            //     if ($request->payment_type === "Cash") {
            //         DB::table('hand_cashes')->decrement('cash_amount', $request->paid);
            //
            //         DB::table('transactions')->insert([
            //             'payment_type'      => 'Cash',
            //             'amount'            => $request->paid,
            //             'transaction_type'  => 'withdraw',
            //             'source_type'       => 'Sale Return',
            //             'source_id'         => $sale->id,
            //             'description'       => 'Sale Return Approved',
            //             'created_at'        => now(),
            //         ]);
            //
            //     } elseif ($request->payment_type === "Bank") {
            //         DB::table('banks')->where('id', $request->bank_id)
            //             ->decrement('bank_balance', $request->paid);
            //
            //         DB::table('transactions')->insert([
            //             'payment_type'      => 'Bank',
            //             'bank_id'           => $request->bank_id,
            //             'amount'            => $request->paid,
            //             'transaction_type'  => 'withdraw',
            //             'source_type'       => 'Sale Return',
            //             'source_id'         => $sale->id,
            //             'description'       => 'Sale Return Approved',
            //             'created_at'        => now(),
            //         ]);
            //     }
            // }


        // Extract customer  ledger data
        $products    = implode(",", $request->input('pro_id'));
        $pro_qty1     = implode(",", $request->input('pro_qty'));

        // Get the last inserted Book Test ID
        $sale_return_id = $sale->id;
        // Extract data from the request
        $pro_ids    = $request->input('pro_id');
        $pro_sku   = $request->input('pro_sku');
        $batch_no  = $request->input('batch_no');
        $qty_sale  = $request->input('qty_sale');


        $purchase_price   = $request->input('pro_cost');
        $pro_sale   = $request->input('pro_sale');
        $pro_qty    = $request->input('pro_qty');
        $red_percent = $request->input('dis_percent');
        $red_amt    = $request->input('dis_amt');
        $total      = $request->input('total');

        // Loop through the data and save each entry
        foreach ($pro_ids as $index => $pro_id) {
            $product_stock                          = new SaleReturnDetail();
            $product_stock->sale_return_id          = $sale_return_id;
            $product_stock->pro_id                  = $pro_id;
            $product_stock->pro_sku                 = $pro_sku[$index];
            $product_stock->batch_no                 = $batch_no[$index];
            $product_stock->qty_sale                 = $qty_sale[$index];

            $product_stock->sale_return_price       = $pro_sale[$index];
            $product_stock->red_per                 = $red_percent[$index];
            $product_stock->red_amount              = $red_amt[$index];
            $product_stock->pro_qty                 = $pro_qty[$index];
            $product_stock->total                   = $total[$index];
            $product_stock->save();

            // 🟢 2. PurchaseDetail (stock) update
            PurchaseDetail::where('pro_id', $pro_id)
            ->where('batch_no', $batch_no[$index])
            ->first()?->decrement('sale_qty', $pro_qty[$index]);

            // ✅ Condition check: serviceable / nonserviceable
            if ($request->input('return_type') === 'serviceable') {
                // Serviceable → stock me wapas add karein
                $stock                   = new Stock();
                $stock->sale_return_id   = $sale_return_id;
                $stock->pro_id           = $pro_id;
                $stock->stock_in_qty     = $pro_qty[$index]; 
                $stock->salesman_id      = $request->input('salesman_id');
                $stock->adjustment       = 'Sale Return';
                $stock->invoice_date     = Carbon::parse($request->input('return_date'))->format('Y-m-d');
                $stock->save();
            } else {
                // Non-Serviceable → non_serviceable_stocks me entry
                $nonService                     = new NonServiceStock();
                $nonService->sale_return_id     = $sale_return_id;
                $nonService->pro_id             = $pro_id;
                $nonService->stock_in_qty       = $pro_qty[$index];
                $nonService->invoice_date       = Carbon::parse($request->input('return_date'))->format('Y-m-d');
                $nonService->adjustment         = 'Sale Return'; // ya request me reason bhejna
                $nonService->save();
            }

        }

        // Insert into Customer Ledger table
        $ledger = new CustomerLedger();
        $ledger->cus_id                 = $request->input('cus_id');
        $ledger->sale_return_id         = $sale_return_id;
        $ledger->invoice_date           = Carbon::parse($request->input('return_date'))->format('Y-m-d');
        $ledger->detail                 = $request->input('detail') ?: 'Sale Return';
        $ledger->pro_id                 = $products;
        $ledger->pro_qty                = $pro_qty1;
        $ledger->transaction_type       = 'Sale_return'; // FIXED: Changed from 'Sale Return' to match ENUM
        $ledger->adjustment             = 'Sale Return';
        $ledger->bank_id                = $request->input('bank_id');
        // CRITICAL FIX: Set total as NEGATIVE to reverse the original sale
        $ledger->total                  = -$request->input('grand_total');
        $ledger->return_amount          = $request->input('grand_total');

        // CRITICAL FIX: "Paid" applies to customer balance, not cash refund
        // debit = 0 means no cash is refunded, return just reduces balance
        $ledger->debit                  = 0;
        $ledger->credit                 = 0;
        $ledger->bank                   = 0;
        $ledger->save();

       // Insert into Salesman Ledger table
        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->salesman_id               = $request->input('salesman_id') ? $request->input('salesman_id') : Auth::id();
        $saleman_ledger->cus_id                    = $request->input('cus_id');
        $saleman_ledger->sale_return_id            = $sale_return_id;
        $saleman_ledger->invoice_date              = Carbon::parse($request->input('return_date'))->format('Y-m-d');
        $saleman_ledger->detail                    = $request->input('detail') ?: 'Sale Return';
        $saleman_ledger->pro_id                    = $products;
        $saleman_ledger->pro_qty                   = $pro_qty1;
        $saleman_ledger->transaction_type          = 'Sale_return'; // FIXED: Match ENUM value
        $saleman_ledger->bank_id                   = $request->input('bank_id');
        $saleman_ledger->adjustment                = 'Sale Return';
        $saleman_ledger->return_paid               = $request->input('paid');
        $saleman_ledger->debit                     = 0;
        $saleman_ledger->credit                    = 0;
        $saleman_ledger->bank                      = 0;
        $saleman_ledger->save();

            // PERFORMANCE FIX: Commit the transaction
            DB::commit();

            return response()->json(['status' => 200, 'message' => 'Sale return updated successfully.']);

        } catch (\Exception $e) {
            // PERFORMANCE FIX: Rollback on error
            DB::rollBack();
            \Log::error('Error updating sale return: ' . $e->getMessage());
            return response()->json(['status' => 500, 'message' => 'Error updating sale return: ' . $e->getMessage()]);
        }
    }





    public function delete_sale_return($id)
    {
        $data = SaleReturn::find($id);

        if (!$data) {
            return response()->json([
                'status' => 404,
                'message' => 'Sale return not found.'
            ]);
        }



        // DISABLED: Since sale returns don't affect cash/bank (only customer balance),
        // no need to reverse cash/bank on deletion
        //
        // if ($data->Payment_type === "Cash") {
        //     DB::table('hand_cashes')->increment('cash_amount', $data->paid);
        //
        // } elseif ($data->Payment_type === "Bank" && $data->bank_id) {
        //     DB::table('banks')->where('id', $data->bank_id)
        //         ->increment('bank_balance', $data->paid);
        // }

        // Delete sale details first
        $data->saleReturnDetails()->delete();
        $data->stock()->delete();
        $data->nonServiceStock()->delete();
        $data->customerLedger()->delete();
        $data->salemanLedger()->delete();

        // 3️⃣ Related Transactions delete
        Transaction::where('source_type', 'Sale Return')
            ->where('source_id', $data->id)
            ->delete();
        // Delete the sale
        $data->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Sale and its details deleted successfully.'
        ]);
    }


    public function todaysale(Request $request)
    {
        $today = now()->toDateString();

        $today_sales = DB::table('sales')
            ->whereDate('date', $today)
            ->whereNotNull('grand_total')
            ->sum('grand_total');

        return response()->json(['today_sales' => 'Rs. ' . number_format($today_sales, 2)]);
    }


    public function totalsales()
    {
        $totalsales = DB::table('sales')
                        ->whereNotNull('grand_total')
                        ->sum('grand_total');

        return response()->json([
            'total_sales' => 'Rs. ' . number_format($totalsales, 2)
        ]);
    }

    public function weeklySales()
    {
        $startDate = Carbon::now()->subDays(7)->toDateString();
        $endDate = Carbon::now()->toDateString();

        $weeklySales = Sale::whereBetween('date', [$startDate, $endDate])
                        ->sum('grand_total');

        return response()->json(['weekly_sales' => 'Rs. ' . number_format($weeklySales, 2)]);
    }


    public function monthlySales()
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $monthlySales = DB::table('sales')
            ->whereYear('date', $currentYear)
            ->whereMonth('date', $currentMonth)
            ->sum('grand_total');

        return response()->json([
            'monthly_sales' => 'Rs. ' .  number_format($monthlySales, 2)
        ]);
    }


}
