<?php

namespace App\Http\Controllers;

use App\Models\CustomerLedger;
use App\Models\OrderBookerLedger;
use App\Models\SalemanLedger;
use App\Models\SalemanRoute;
use App\Models\SalesMan;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SalesmanController extends Controller
{
    

    public function load_salesman(Request $request)
    {
        // $users = User::all();

        // return view('admin.users', compact('users'));

        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        $query = DB::table('sales_man');

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('created_at', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('sales_man.name', 'like', '%' . $search . '%');
        }



        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $users = $query->get();

   

        $all_data = [];
        foreach ($users as $user) {

           

            $btn = '<td class="text-right">';
            if (Auth::user()->can('update_salesman')) {
                $btn .= '<button id="edit_btn" data-eid="'.$user->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_salesman')) {
                $btn .= '<button id="delete_btn" data-did="'.$user->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            if($user->saleman_image != ""){

                $image = '<img   height="50px" src="uploads/saleman_image/'.$user->saleman_image.'">';

            }else{

                $image = '<img  " height="50px" src="uploads/no_image.png">';

            }

          

            

            $all_data[] = [
                'id'            => $user->id,
                'name'          => $user->name,
                'phone'         => $user->phone,
                'email'          => $user->email,
                'cnic'          => $user->cnic,
                'saleman_image'           => $image,
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
        ];

        return response()->json($data);


    }

    public function insert_salesman(Request $request)
    {
        $user_cnic = SalesMan::where('cnic', $request->input('cnic'))->exists();

        if($user_cnic){

            return response()->json([
                'status'=> "cnic_Exist",
            ]);
            
        }
        else{

            if ($request->hasFile('saleman_image')) {
                $image1 = $request->file('saleman_image');
                $saleman_image = hexdec(uniqid()).'.'.$image1->getClientOriginalExtension();
                $image1->move(public_path('/uploads/saleman_image/'), $saleman_image);
            } else {
                $saleman_image = null; // Set to null if no front image is provided
            }

            $data = new SalesMan();
            $data->name         = $request->input('name');
            $data->phone        = $request->input('phone');
            $data->email        = $request->input('email');
            $data->cnic        = $request->input('cnic');
            $data->address        = $request->input('address');
            $data->saleman_image        = $saleman_image;
            $data->created_by   = Auth::id();
            $data->save();

            $saleman_id     = $data->id;
            $city_ids       = $request->input('city_id');
            $day            = $request->input('day');
            $area_id        = $request->input('area_id');

            foreach ($city_ids as $index => $city_id) {
                if (isset($day[$index]) && isset($area_id[$index])) {
                    $route = new SalemanRoute();
                    $route->saleman_id = $saleman_id;
                    $route->city_id    = $city_id;
                    $route->day        = $day[$index];
                    $route->area_id    = $area_id[$index];
                    $route->save();
                }
            }


            return response()->json([
                'status'=>200
            ]);

        }
    }

    public function edit_salesman($id)
    {
        $data = SalesMan::with(['routes.city', 'routes.area'])->findOrFail($id);

        return response()->json([
            'status' => 200,
            'data'   => $data,
        ]);
    }

    public function update_salesman(Request $request)
    {
        // Validate inputs first
    

        // Check if CNIC already exists for another salesman
        $user_cnic = SalesMan::where('cnic', $request->input('cnic'))
            ->where('id', '!=', $request->input('salesman_id'))
            ->exists();

        if ($user_cnic) {
            return response()->json([
                'status'=> "cnic_Exist",
            ]);
        }

        $user = SalesMan::findOrFail($request->input('salesman_id'));

        // Handle image update
        if ($request->hasFile('saleman_image')) {
            $oldPath = public_path('/uploads/saleman_image/' . $user->saleman_image);
            if ($user->saleman_image && file_exists($oldPath)) {
                unlink($oldPath);
            }

            $image = $request->file('saleman_image');
            $filename = hexdec(uniqid()) . '.' . $image->getClientOriginalExtension();
            $image->move(public_path('/uploads/saleman_image/'), $filename);
            $user->saleman_image = $filename;
        }

        // Update main salesman details
        $user->name       = $request->input('salesman_name');
        $user->phone      = $request->input('phone');
        $user->cnic       = $request->input('cnic');
        $user->email      = $request->input('email');
        $user->address    = $request->input('address');
        $user->updated_by = Auth::id();
        $user->save();

        // Update routes (delete & reinsert)
        SalemanRoute::where('saleman_id', $user->id)->delete();

        $city_ids = $request->input('city_id');
        $days     = $request->input('day');
        $area_ids = $request->input('area_id');

        foreach ($city_ids as $index => $city_id) {
            if (!empty($days[$index]) && !empty($area_ids[$index])) {
                SalemanRoute::create([
                    'saleman_id' => $user->id,
                    'city_id'    => $city_id,
                    'day'        => $days[$index],
                    'area_id'    => $area_ids[$index],
                ]);
            }
        }

        return response()->json([
            'status'=>200,
        ]);
    }


    public function delete_salesman($id)
    {
        $data = SalesMan::findOrFail($id);

        // Delete salesman routes first
        SalemanRoute::where('saleman_id', $data->id)->delete();

        // Then delete salesman
        $data->delete();

        return response()->json([
            'status' => 200
        ]);
    }


    
    public function get_salesman()
    {
        $sales_man = SalesMan::all();

        return response()->json($sales_man);
    }


    // ledger all salesman area only show saleman by roles so its ledger show
    public function load_all_salesman_ledger(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

    
        $query = DB::table('users')
            ->leftJoin('model_has_roles', 'users.id', '=', 'model_has_roles.model_id')
            ->leftJoin('roles', 'model_has_roles.role_id', '=', 'roles.id')

            // Sales aggregation subquery
            ->leftJoin(DB::raw('(
                SELECT saleman_id, 
                    SUM(grand_total) as total_sales,
                    SUM(paid) as total_paid
                FROM sales
                GROUP BY saleman_id
            ) as s'), 'users.id', '=', 's.saleman_id')

            // Expenses aggregation subquery
            ->leftJoin(DB::raw('(
                SELECT created_by, 
                    SUM(total_amt) as total_expenses
                FROM expenses
                GROUP BY created_by
            ) as e'), 'users.id', '=', 'e.created_by')

            // Sale Returns aggregation subquery
            ->leftJoin(DB::raw('(
                SELECT saleman_id, 
                    SUM(paid) as total_return_paid
                FROM sale_returns
                GROUP BY saleman_id
            ) as r'), 'users.id', '=', 'r.saleman_id')

            ->select([
                'users.id',
                'users.name',
                'users.email',
                'users.created_at',
                'roles.name as role_name',

                DB::raw('COALESCE(s.total_sales,0) as total_sales'),
                DB::raw('COALESCE(s.total_paid,0) as total_paid'),
                DB::raw('COALESCE(e.total_expenses,0) as total_expenses'),
                DB::raw('COALESCE(r.total_return_paid,0) as total_return_paid'),

                DB::raw('COALESCE(s.total_paid,0) - (COALESCE(e.total_expenses,0) + COALESCE(r.total_return_paid,0)) as balance')
            ])

            ->where('users.id', '!=', 1)
            ->where('users.is_deleted', '!=', 1)
        ->where('roles.name', '=', 'Sales Man');



        // agar super admin hai Ã¢â€ â€™ sab show karo
        if (!Auth::user()->hasRole(['Cashier', 'Super Admin'])) {
            // salesman ya koi aur Ã¢â€ â€™ sirf apna hi show hoga
            $query->where('users.id', Auth::id());
        }


        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('created_at', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('users.name', 'like', '%' . $search . '%');
        }

        $type = $request->input('type');
        if (isset($type)) {
            $query->where('users.is_active', '=', $type);
        }

        $role = $request->input('search_by_role');
        if (isset($role)) {
            $query->where('roles.id', '=', $role);
        }

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $users = $query->get();

        // Calculate totals by roles
        $roleTotals = DB::table('model_has_roles')
        ->join('roles', 'model_has_roles.role_id', '=', 'roles.id')
        ->select('roles.name', DB::raw('count(model_has_roles.model_id) as total'))
        ->groupBy('roles.name')
        ->get()
        ->pluck('total', 'name');

        $all_data = [];
        foreach ($users as $user) {

            $btn = '<td class="text-right">';
                // View button
                $btn .= '<a href="' . route("salesman_ledger", $user->id) . '" 
                class="btn btn-success btn-icon" data-toggle="tooltip" title="SaleMan Ledger" style="padding: 6px 10px;"><i class="fa-duotone fa-memo" style="margin-right: 3px"></i></a>';

            $btn .= '</td>';

            if ($user->role_name == "Teacher"){
                                    
                    $role_name = '<div class="label label-table label-warning">'.$user->role_name.'</div>';   
                }else if ($user->role_name == "Student"){
                    $role_name = '<div class="label label-table label-success">'.$user->role_name.'</div>';   
                }else if ($user->role_name == "Parent"){
                    $role_name = '<div class="label label-table label-info">'.$user->role_name.'</div>';
                }else {
                    $role_name = '<div class="label label-table label-primary">'.$user->role_name.'</div>';
            }

            

            $all_data[] = [
                'id'            => $user->id,
                'date'          => $user->created_at  ? Carbon::parse($user->created_at)->format('d-m-Y')  : "",
                'name'          => $user->name,
                'total_sales'         => $user->total_sales, 
                'total_paid'          => $user->total_paid,
                'total_expenses'    => $user->total_expenses,
                'total_return_paid' => $user->total_return_paid,
                'balance'        => $user->balance,
                'btn'           => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
            "roleTotals"      => $roleTotals
        ];

        return response()->json($data);


    }



    /**
     * COMPLETELY REVAMPED LEDGER FUNCTION
     * Handles:
     * 1. Sale (Paid) - Credit = paid amount, Balance = due amount
     * 2. Sale (Unpaid) - Credit = 0, Balance = total amount  
     * 3. Sale (Partial) - Credit = paid amount, Balance = running balance
     * 4. Sale Return - Different handling for paid from credit vs paid from balance
     * 5. Expenses - Tracked separately
     */
    public function load_salesman_ledger(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $saleman_id = $request->input('saleman_id');
        
        // =====================================================
        // STEP 1: Get Sales Data for the selected date range
        // =====================================================
        $salesQuery = DB::table('sales')
            ->where('sales.saleman_id', '=', $saleman_id)
            ->where('sales.is_deleted', '!=', 1)
            ->leftJoin('customers', 'customers.id', '=', 'sales.cus_id')
            ->select(
                'sales.id',
                'sales.invoice_date',
                'sales.cus_id',
                'sales.grand_total as total',
                'sales.paid as credit',
                'sales.due as debit',
                'sales.Payment_type as payment_type',
                'sales.bank_id',
                DB::raw("'Sale' as entry_type"),
                DB::raw("'Sale' as transaction_type"),
                'customers.name as cus_name',
                DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),
                'sales.created_at'
            );
            
        if (!empty($start_date) && !empty($end_date)) {
            $salesQuery->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }
        
        // =====================================================
        // STEP 2: Get Sale Returns Data
        // =====================================================
        $returnsQuery = DB::table('sale_returns')
            ->where('sale_returns.saleman_id', '=', $saleman_id)
            ->leftJoin('customers', 'customers.id', '=', 'sale_returns.cus_id')
            ->select(
                'sale_returns.id',
                'sale_returns.invoice_date',
                'sale_returns.cus_id',
                'sale_returns.grand_total as total',
                'sale_returns.paid as credit',
                DB::raw('0 as debit'),
                'sale_returns.Payment_type as payment_type',
                'sale_returns.bank_id',
                DB::raw("'Sale_return' as entry_type"),
                DB::raw("'Sale_return' as transaction_type"),
                'customers.name as cus_name',
                DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),
                'sale_returns.created_at'
            );
            
        if (!empty($start_date) && !empty($end_date)) {
            $returnsQuery->whereBetween('sale_returns.invoice_date', [$start_date, $end_date]);
        }
        
        // =====================================================
        // STEP 3: Get Expenses Data
        // =====================================================
        $expensesQuery = DB::table('expenses')
            ->where('expenses.created_by', '=', $saleman_id)
            ->where('expenses.is_deleted', '!=', 1)
            ->select(
                'expenses.id',
                DB::raw('expenses.exp_date as invoice_date'),
                DB::raw('NULL as cus_id'),
                'expenses.total_amt as total',
                DB::raw('0 as credit'),
                DB::raw('0 as debit'),
                DB::raw("'Cash' as payment_type"),
                DB::raw('NULL as bank_id'),
                DB::raw("'Expense' as entry_type"),
                DB::raw("'expense' as transaction_type"),
                DB::raw('"---" as cus_name'),
                DB::raw('"Expense" as shop_name'),
                'expenses.created_at'
            );
            
        if (!empty($start_date) && !empty($end_date)) {
            $expensesQuery->whereBetween('expenses.exp_date', [$start_date, $end_date]);
        }
        
        // =====================================================
        // STEP 4: Get Payment Received entries from saleman_ledgers
        // =====================================================
        $paymentsQuery = DB::table('saleman_ledgers')
            ->where('saleman_ledgers.salesman_id', '=', $saleman_id)
            ->whereNull('saleman_ledgers.sale_id')
            ->whereNull('saleman_ledgers.sale_return_id')
            ->whereNull('saleman_ledgers.deleted_at')
            ->where('saleman_ledgers.transaction_type', '!=', 'expense')
            ->leftJoin('customers', 'customers.id', '=', 'saleman_ledgers.cus_id')
            ->leftJoin('banks', 'banks.id', '=', 'saleman_ledgers.bank_id')
            ->select(
                'saleman_ledgers.id',
                'saleman_ledgers.invoice_date',
                'saleman_ledgers.cus_id',
                DB::raw('0 as total'),
                DB::raw('(COALESCE(saleman_ledgers.credit, 0) + COALESCE(saleman_ledgers.bank, 0)) as credit'),
                'saleman_ledgers.debit',
                'saleman_ledgers.transaction_type as payment_type',
                'saleman_ledgers.bank_id',
                DB::raw("'Payment' as entry_type"),
                'saleman_ledgers.transaction_type',
                'customers.name as cus_name',
                DB::raw('COALESCE(customers.shop_name, "---") as shop_name'),
                'saleman_ledgers.created_at'
            );
            
        if (!empty($start_date) && !empty($end_date)) {
            $paymentsQuery->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
        }
        
        // =====================================================
        // STEP 5: Combine all queries and process
        // =====================================================
        $sales = $salesQuery->get();
        $returns = $returnsQuery->get();
        $expenses = $expensesQuery->get();
        $payments = $paymentsQuery->get();
        
        // Merge all records
        $allRecords = collect()
            ->merge($sales)
            ->merge($returns)
            ->merge($expenses)
            ->merge($payments)
            ->sortBy('invoice_date')
            ->values();
        
        // =====================================================
        // STEP 6: Calculate running balances and prepare data
        // =====================================================
        $all_data = [];
        $currentBalance = 0;
        
        // Totals for summary
        $total_sale_amount = 0;
        $total_cash = 0;
        $total_bank = 0;
        $total_easypaisa = 0;
        $total_jazzcash = 0;
        $total_meezan = 0;
        $total_upaisa = 0;
        $total_expenses = 0;
        $total_return_paid_from_credit = 0;
        $total_return_paid_from_balance = 0;
        $total_return_amount = 0;
        
        foreach ($allRecords as $record) {
            $displayCredit = 0;
            $displayDebit = 0;
            $displayBank = 0;
            $displayReturnPaid = 0;
            $transactionType = '';
            $btn = '';
            $displayId = '---';
            
            switch ($record->entry_type) {
                case 'Sale':
                    // SALE LOGIC
                    // Total is the grand total of the sale
                    // Credit is the paid amount (cash or bank)
                    // Balance increases by the unpaid amount (due)
                    
                    $total_sale_amount += $record->total;
                    
                    if ($record->payment_type === 'Cash') {
                        $displayCredit = $record->credit;
                        $total_cash += $record->credit;
                        $transactionType = '<div class="label label-table label-success" style="width:100%; max-width:130px;">Sale (Cash)</div>';
                    } elseif ($record->payment_type === 'Bank') {
                        $displayBank = $record->credit;
                        // Track by bank
                        if ($record->bank_id == 1) {
                            $total_easypaisa += $record->credit;
                        } elseif ($record->bank_id == 2) {
                            $total_jazzcash += $record->credit;
                        } elseif ($record->bank_id == 3) {
                            $total_meezan += $record->credit;
                        } elseif ($record->bank_id == 4) {
                            $total_upaisa += $record->credit;
                        } else {
                            $total_bank += $record->credit;
                        }
                        $transactionType = '<div class="label label-table label-info" style="width:100%; max-width:130px;">Sale (Bank)</div>';
                    } else {
                        $displayCredit = $record->credit;
                        $total_cash += $record->credit;
                        $transactionType = '<div class="label label-table label-warning" style="width:100%; max-width:130px;">Sale</div>';
                    }
                    
                    // Balance increases by the due amount
                    $currentBalance += $record->debit;
                    $displayId = 'S-' . $record->id;
                    break;
                    
                case 'Sale_return':
                    // SALE RETURN LOGIC
                    // Check if the return was paid from credit (customer had paid fully) 
                    // or from balance (customer had outstanding balance)
                    
                    $total_return_amount += $record->total;
                    
                    if ($record->credit > 0) {
                        // Return Paid from Credit - customer gets money back
                        $displayReturnPaid = $record->credit;
                        $displayDebit = $record->total;
                        $total_return_paid_from_credit += $record->credit;
                        $transactionType = '<div class="label label-table label-danger" style="width:100%; max-width:130px;">Return (Paid)</div>';
                    } else {
                        // Return from Balance - reduces customer's outstanding balance
                        $displayDebit = $record->total;
                        $currentBalance -= $record->total;
                        $total_return_paid_from_balance += $record->total;
                        $transactionType = '<div class="label label-table label-purple" style="width:100%; max-width:130px;">Return (Balance)</div>';
                    }
                    
                    $displayId = 'SR-' . $record->id;
                    break;
                    
                case 'Expense':
                    // EXPENSE LOGIC
                    $total_expenses += $record->total;
                    $transactionType = '<div class="label label-table label-default" style="width:100%; max-width:130px;">Expense</div>';
                    $displayId = 'EXP-' . $record->id;
                    break;
                    
                case 'Payment':
                    // PAYMENT RECEIVED LOGIC
                    $displayCredit = $record->credit;
                    
                    if ($record->payment_type === 'credit' || $record->payment_type === 'Cash') {
                        $total_cash += $record->credit;
                        $transactionType = '<div class="label label-table label-success" style="width:100%; max-width:130px;">Payment (Cash)</div>';
                    } elseif ($record->payment_type === 'bank' || $record->payment_type === 'Bank') {
                        $displayBank = $record->credit;
                        $displayCredit = 0;
                        if ($record->bank_id == 1) {
                            $total_easypaisa += $record->credit;
                        } elseif ($record->bank_id == 2) {
                            $total_jazzcash += $record->credit;
                        } elseif ($record->bank_id == 3) {
                            $total_meezan += $record->credit;
                        } elseif ($record->bank_id == 4) {
                            $total_upaisa += $record->credit;
                        } else {
                            $total_bank += $record->credit;
                        }
                        $transactionType = '<div class="label label-table label-info" style="width:100%; max-width:130px;">Payment (Bank)</div>';
                    } else {
                        $transactionType = '<div class="label label-table label-success" style="width:100%; max-width:130px;">Payment</div>';
                    }
                    
                    // Payment reduces balance
                    $currentBalance -= $record->credit;
                    $displayId = 'PAY-' . $record->id;
                    
                    // Add delete button for payments
                    if (Auth::user()->hasRole(['Super Admin'])) {
                        $btn = '<td class="text-right">
                                    <button id="delete_btn" data-did="' . $record->id . '" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>
                                </td>';
                    }
                    break;
            }
            
            $all_data[] = [
                'id'                    => $record->id,
                'invoice_date'          => Carbon::parse($record->invoice_date)->format('d-m-Y'),
                'shop_name'             => $record->shop_name,
                'sale_id'               => $displayId,
                'total'                 => number_format($record->total, 0),
                'transaction_type'      => $transactionType,
                'credit'                => number_format($displayCredit, 0),
                'bank'                  => number_format($displayBank, 0),
                'return_paid'           => number_format($displayReturnPaid, 0),
                'status'                => $record->entry_type,
                'balance'               => number_format($currentBalance, 0),
                'btn'                   => $btn
            ];
        }
        
        // =====================================================
        // STEP 7: Get approved amounts to calculate pending
        // =====================================================
        $approvedAmounts = DB::table('saleman_ledgers')
            ->where('saleman_ledgers.salesman_id', '=', $saleman_id)
            ->whereNull('saleman_ledgers.deleted_at')
            ->where('saleman_ledgers.status', '=', 'approved')
            ->select([
                DB::raw('SUM(CASE WHEN transaction_type = "Cash" OR transaction_type = "credit" THEN saleman_cash ELSE 0 END) as approved_cash'),
                DB::raw('SUM(CASE WHEN transaction_type = "Bank" AND bank_id = 1 THEN saleman_cash ELSE 0 END) as approved_easypaisa'),
                DB::raw('SUM(CASE WHEN transaction_type = "Bank" AND bank_id = 2 THEN saleman_cash ELSE 0 END) as approved_jazzcash'),
                DB::raw('SUM(CASE WHEN transaction_type = "Bank" AND bank_id = 3 THEN saleman_cash ELSE 0 END) as approved_meezan'),
                DB::raw('SUM(CASE WHEN transaction_type = "Bank" AND bank_id = 4 THEN saleman_cash ELSE 0 END) as approved_upaisa'),
                DB::raw('SUM(CASE WHEN transaction_type = "expense" THEN saleman_cash ELSE 0 END) as approved_expense'),
            ])
            ->first();
        
        $saleman = User::find($saleman_id);
        
        // Calculate pending amounts (total - approved)
        $pending_cash = $total_cash - ($approvedAmounts->approved_cash ?? 0);
        $pending_easypaisa = $total_easypaisa - ($approvedAmounts->approved_easypaisa ?? 0);
        $pending_jazzcash = $total_jazzcash - ($approvedAmounts->approved_jazzcash ?? 0);
        $pending_meezan = $total_meezan - ($approvedAmounts->approved_meezan ?? 0);
        $pending_upaisa = $total_upaisa - ($approvedAmounts->approved_upaisa ?? 0);
        $pending_expenses = $total_expenses - ($approvedAmounts->approved_expense ?? 0);
        
        // Total amount = Cash + Banks - Expenses - Return Paid from Credit
        $total_collected = $pending_cash + $pending_easypaisa + $pending_jazzcash + $pending_meezan + $pending_upaisa;
        $total_amount_to_send = $total_collected - $pending_expenses - $total_return_paid_from_credit;
        
        $total_results = array(
            "total_amount"      => number_format($total_sale_amount, 0),
            "total_debit"       => number_format(0, 0),
            "return_paid"       => number_format($total_return_paid_from_credit + $total_return_paid_from_balance, 0),
            "total_credit"      => number_format($total_cash + $total_easypaisa + $total_jazzcash + $total_meezan + $total_upaisa, 0),
            "total_balance"     => number_format($currentBalance, 0),
            "saleman_name"      => $saleman->name . " Ledger (ID: " . $saleman->id . ")",
            
            // Pending amounts for "Send to Cashier"
            'cash'              => $pending_cash,
            'easypaisa'         => $pending_easypaisa,
            'jazzcash'          => $pending_jazzcash,
            'meezan'            => $pending_meezan,
            'u_paisa'           => $pending_upaisa,
            'total_expense'     => $pending_expenses,
            
            // Additional info for Send to Cashier modal
            'return_paid_credit'    => $total_return_paid_from_credit,
            'return_paid_balance'   => $total_return_paid_from_balance,
            'total_amount_to_send'  => $total_amount_to_send,
        );
    
        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => count($all_data),
            "recordsFiltered" => count($all_data),
            "data"            => $all_data,
            "totals"          => $total_results,
        ];
    
        return response()->json($data);
    }
    


    public function insert_salesman_payment(Request $request)
    {
        try {
            // Validate the request
            $validated = $request->validate([
                'salesman_id' => 'required|exists:users,id',
                'cus_id' => 'required|exists:customers,id',
                'amount' => 'required|numeric|min:0.01',
                'transaction_type' => 'required|in:credit,bank',
                'transaction_date' => 'required|date_format:d-m-Y',
                'bank_id' => 'required_if:transaction_type,bank|nullable|exists:banks,id',
                'transaction_details' => 'nullable|string|max:500'
            ], [
                'salesman_id.required' => 'Salesman ID is required',
                'salesman_id.exists' => 'Selected salesman does not exist',
                'cus_id.required' => 'Please select a customer',
                'cus_id.exists' => 'Selected customer does not exist',
                'amount.required' => 'Payment amount is required',
                'amount.numeric' => 'Payment amount must be a number',
                'amount.min' => 'Payment amount must be greater than zero',
                'transaction_type.required' => 'Payment type is required',
                'transaction_type.in' => 'Invalid payment type',
                'transaction_date.required' => 'Payment date is required',
                'bank_id.required_if' => 'Bank selection is required for bank payments',
                'bank_id.exists' => 'Selected bank does not exist'
            ]);

            // Get the payment amount from request
            $paymentAmount = floatval($request->input('amount', 0));
            $customerId = $request->input('cus_id');

            // Log for debugging
            \Log::info('Payment Recording', [
                'customer_id' => $customerId,
                'payment_amount' => $paymentAmount,
                'transaction_type' => $request->input('transaction_type'),
                'user_id' => Auth::id()
            ]);

        // Set credit, bank, and debit amounts based on transaction type
        // Payment is always a CREDIT (reduces customer's debt)
        $credit = $request->input('transaction_type') == "credit" ? $paymentAmount : 0;
        $bank   = $request->input('transaction_type') == "bank"   ? $paymentAmount : 0;
        $debit  = 0; // Payment never increases debt

        // Create Salesman Ledger Entry
        $saleman_ledger = new SalemanLedger();
        $saleman_ledger->cus_id = $customerId;
        $saleman_ledger->invoice_date = Carbon::parse($request->input('transaction_date'))->format('Y-m-d');

        // Build detail message
        $detail = $request->input('transaction_details') ?: "Payment Received from Customer";
        $saleman_ledger->detail = $detail;

        $saleman_ledger->transaction_type = $request->input('transaction_type');
        $saleman_ledger->credit = $credit;
        $saleman_ledger->debit  = $debit;
        $saleman_ledger->bank = $bank;
        $saleman_ledger->bank_id  = $request->input('bank_id');
        $saleman_ledger->salesman_id = $request->input('salesman_id');
        $saleman_ledger->created_by = Auth::id();
        $saleman_ledger->save();

        // Update Hand Cash or Bank Balance
        if ($request->transaction_type === "credit") {
            // Add to hand cash
            DB::table('hand_cashes')->increment('cash_amount', $paymentAmount);

            // Record transaction
            DB::table('transactions')->insert([
                'payment_type'      => 'Cash',
                'amount'            => $credit,
                'transaction_type'  => 'deposit',
                'source_type'       => 'Ledger Payment',
                'source_id'         => $saleman_ledger->id,
                'description'       => 'Customer Payment Received (Cash)',
                'created_at'        => now(),
            ]);

        } elseif ($request->transaction_type === "bank") {
            // Add to bank balance
            DB::table('banks')->where('id', $request->bank_id)
                ->increment('bank_balance', $paymentAmount);

            // Record transaction
            DB::table('transactions')->insert([
                'payment_type'      => 'Bank',
                'bank_id'           => $request->input('bank_id'),
                'amount'            => $bank,
                'transaction_type'  => 'deposit',
                'source_type'       => 'Ledger Payment',
                'source_id'         => $saleman_ledger->id,
                'description'       => 'Customer Payment Received (Bank)',
                'created_at'        => now(),
            ]);
        }

        // Create Customer Ledger Entry (Payment reduces customer's outstanding balance)
        $customer_ledger = new CustomerLedger();
        $customer_ledger->cus_id = $customerId;
        $customer_ledger->invoice_date = Carbon::parse($request->input('transaction_date'))->format('Y-m-d');
        $customer_ledger->detail = $detail;
        $customer_ledger->transaction_type = $request->input('transaction_type'); // 'credit' or 'bank'

        // Set bank_id only if transaction type is bank
        $customer_ledger->bank_id = $request->input('transaction_type') == "bank" ? $request->input('bank_id') : null;

        // Record payment based on type (consistent with CustomerController pattern)
        // For cash payments: amount goes in credit column
        // For bank payments: amount goes in bank column
        $customer_ledger->credit = $credit;
        $customer_ledger->debit = 0; // No debit for payment
        $customer_ledger->bank = $bank;
        $customer_ledger->total = $credit + $bank; // Total payment amount

        $customer_ledger->created_by = Auth::id();
        $customer_ledger->updated_by = Auth::id();
        $customer_ledger->save();

            \Log::info('Payment Recorded Successfully', [
                'salesman_ledger_id' => $saleman_ledger->id,
                'customer_ledger_id' => $customer_ledger->id,
                'payment_amount' => $paymentAmount
            ]);

            return response()->json([
                'status' => 200,
                'message' => 'Payment recorded successfully',
                'data' => [
                    'payment_amount' => $paymentAmount,
                    'transaction_type' => $request->input('transaction_type')
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            \Log::error('Validation Error in insert_salesman_payment', [
                'errors' => $e->errors(),
                'request_data' => $request->all()
            ]);

            return response()->json([
                'status' => 422,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);

        } catch (\Exception $e) {
            \Log::error('Error in insert_salesman_payment', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'request_data' => $request->all()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'An error occurred while saving the payment: ' . $e->getMessage()
            ], 500);
        }
    }


    /**
     * REVAMPED: Send to Cashier - Properly calculates all pending amounts
     * Updated to handle dynamic bank inputs from frontend
     */
    public function send_to_cashier(Request $request)
    {
        try {
            // Check if user has already sent to cashier today
            $user_id = Auth::id();
            $today = Carbon::today()->format('Y-m-d');

            $todaySubmission = SalemanLedger::where('salesman_id', $user_id)
                ->where('detail', 'Send to cashier')
                ->whereDate('invoice_date', $today)
                ->first();

            if ($todaySubmission) {
                return response()->json([
                    'status' => 400,
                    'message' => 'You have already sent to cashier today. Please try again tomorrow.'
                ], 400);
            }

            $data = [];

            // Add cash entry
            $data['cash'] = [
                'amount' => $request->input('pending_cash', 0),
                'transaction_type' => 'Cash',
                'bank_id' => null
            ];

            // Dynamically process bank inputs (pending_bank_1, pending_bank_2, etc.)
            $allInputs = $request->all();
            foreach ($allInputs as $key => $value) {
                if (strpos($key, 'pending_bank_') === 0) {
                    // Extract bank_id from key (e.g., pending_bank_1 -> 1)
                    $bank_id = str_replace('pending_bank_', '', $key);
                    if (is_numeric($bank_id) && $value > 0) {
                        $data['bank_' . $bank_id] = [
                            'amount' => $value,
                            'transaction_type' => 'Bank',
                            'bank_id' => (int) $bank_id
                        ];
                    }
                }
            }

            // Add expenses entry
            $data['expenses'] = [
                'amount' => $request->input('pending_expenses', 0),
                'transaction_type' => 'expense',
                'bank_id' => null
            ];

            // Also record return paid amounts if provided
            if ($request->input('pending_return_credit') > 0) {
                $data['return_credit'] = [
                    'amount' => $request->input('pending_return_credit'),
                    'transaction_type' => 'return_credit',
                    'bank_id' => null
                ];
            }

            if ($request->input('pending_return_balance') > 0) {
                $data['return_balance'] = [
                    'amount' => $request->input('pending_return_balance'),
                    'transaction_type' => 'return_balance',
                    'bank_id' => null
                ];
            }

            $recordsInserted = 0;
            foreach ($data as $item) {
                if ($item['amount'] > 0) { // sirf non-zero records insert karo
                    $ledger                     = new SalemanLedger();
                    $ledger->cus_id             = null;
                    $ledger->invoice_date       = now();
                    $ledger->detail             = 'Send to cashier';
                    $ledger->saleman_cash       = $item['amount'];
                    $ledger->transaction_type   = $item['transaction_type'];
                    $ledger->bank_id            = $item['bank_id'];
                    $ledger->status             = 'pending';
                    $ledger->salesman_id        = Auth::id();
                    $ledger->created_by         = Auth::id();
                    $ledger->save();
                    $recordsInserted++;
                }
            }

            if ($recordsInserted === 0) {
                return response()->json([
                    'status' => 400,
                    'message' => 'No pending amounts to send to cashier!'
                ], 400);
            }

            return response()->json([
                'status' => 200,
                'message' => 'All pending payments sent to cashier!'
            ]);

        } catch (\Exception $e) {
            \Log::error('Error in send_to_cashier: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error sending to cashier: ' . $e->getMessage()
            ], 500);
        }
    }


    /**
     * Check if salesman can send to cashier today
     */
    public function check_can_send_to_cashier()
    {
        try {
            $user_id = Auth::id();
            $today = Carbon::today()->format('Y-m-d');

            // Check if user has already submitted today
            $todaySubmission = SalemanLedger::where('salesman_id', $user_id)
                ->where('detail', 'Send to cashier')
                ->whereDate('invoice_date', $today)
                ->first();

            return response()->json([
                'status' => 200,
                'can_send' => $todaySubmission ? false : true,
                'message' => $todaySubmission ? 'You have already sent to cashier today' : 'You can send to cashier'
            ]);
        } catch (\Exception $e) {
            \Log::error('Check Can Send to Cashier Error: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'can_send' => true,
                'message' => 'Error checking submission status'
            ], 500);
        }
    }





    public function edit_salesman_payment($id)
    {
        $Data = SalemanLedger::find($id);

        $Data->invoice_date = $Data->invoice_date ? Carbon::parse($Data->invoice_date)->format('d-m-Y') : null;

        return response()->json([
            'status'=>200,
            'data' => $Data,
        ]);
    }


    public function update_salesman_payment(Request $request)
    {

 
        $ledger = SalemanLedger::find($request->input('pay_id')); 
        $ledger->saleman_cash = $request->input('amount');
        $ledger->status    = $request->input('status'); 
        $ledger->transaction_type = $request->input('transaction_type');
        $ledger->bank_id = $request->input('bank_id');
        $ledger->save();

        return response()->json([
            'status'=>200
        ]);

    }

 

    public function delete_salesman_payment($id)
    {
        // Find the salesman ledger entry by ID
        $salesmanLedger = SalemanLedger::find($id);

        if (!$salesmanLedger) {
            return response()->json([
                'status' => 404,
                'message' => 'Record not found'
            ]);
        }

        // Delete related customer ledger
        CustomerLedger::where('cus_id', $salesmanLedger->cus_id)
            ->where('invoice_date', $salesmanLedger->invoice_date)
            ->where('transaction_type', $salesmanLedger->transaction_type)
            ->where('credit', $salesmanLedger->credit)
            ->where('debit', $salesmanLedger->debit)
            ->where('bank', $salesmanLedger->bank)
            ->delete();

        // Delete related transaction(s)
        DB::table('transactions')
            ->where('source_type', 'Ledger Payment')
            ->where('source_id', $salesmanLedger->id)
            ->delete();

        // Delete salesman ledger
        $salesmanLedger->delete();

        return response()->json([
            'status' => 200,
            'message' => 'Record deleted successfully'
        ]);
    }

 
    public function getCustomerBalance($customerId)
    {
        try {
            // Fetch customer ledger entries (matching CustomerController logic)
            $ledgerEntries = DB::table('customer_ledgers')
                ->leftJoin('sale_returns', 'sale_returns.id', '=', 'customer_ledgers.sale_return_id')
                ->where('customer_ledgers.cus_id', $customerId)
                ->whereNull('customer_ledgers.deleted_at')
                ->where(function($q) {
                    $q->whereNull('customer_ledgers.sale_return_id')
                      ->orWhere(function($q2) {
                          $q2->whereNotNull('customer_ledgers.sale_return_id')
                             ->where('sale_returns.status', '=', 'approved');
                      });
                })
                ->select(
                    'customer_ledgers.detail',
                    'customer_ledgers.adjustment',
                    'customer_ledgers.transaction_type',
                    'customer_ledgers.debit',
                    'customer_ledgers.credit',
                    'customer_ledgers.bank',
                    'customer_ledgers.total',
                    'customer_ledgers.return_amount'
                )
                ->orderBy('customer_ledgers.invoice_date', 'asc')
                ->orderBy('customer_ledgers.id', 'asc')
                ->get();

            // Calculate balance using the same logic as CustomerController
            $currentBalance = 0;

            foreach ($ledgerEntries as $data) {
                if ($data->detail == "Opening Balance") {
                    // Opening Balance: Customer owes this amount from the start
                    $currentBalance += $data->debit;
                } else if ($data->adjustment == "Sale") {
                    // Sale: Customer owes more (total minus what they paid)
                    $currentBalance += $data->total - $data->credit - $data->bank;
                } else if ($data->transaction_type == "Sale_return" || $data->adjustment == "Sale Return") {
                    // Sale Return reduces customer debt by the return_amount
                    $returnAmount = $data->return_amount ?? $data->total ?? 0;
                    $reduction = min($returnAmount, max(0, $currentBalance));
                    $currentBalance -= $reduction;
                } else if ($data->transaction_type == "Payment Receive" || $data->transaction_type == "credit" || $data->transaction_type == "bank") {
                    // Payment: Customer owes less (payment reduces debt)
                    $currentBalance -= ($data->credit + $data->bank);
                } else if ($data->debit > 0) {
                    // Any other debit reduces balance
                    $currentBalance -= $data->debit;
                } else if ($data->credit > 0 || $data->bank > 0) {
                    // Any other credit/bank payment reduces balance
                    $currentBalance -= ($data->credit + $data->bank);
                }
            }

            return response()->json([
                'status' => 200,
                'balance' => round($currentBalance, 2)
            ]);
        } catch (\Exception $e) {
            \Log::error('Error fetching customer balance', [
                'customer_id' => $customerId,
                'error' => $e->getMessage()
            ]);

            return response()->json([
                'status' => 500,
                'message' => 'Error fetching customer balance',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * REVAMPED: Get Pending Balances for Send to Cashier Modal
     * Calculates:
     * - Cash (from sales paid in cash)
     * - Bank amounts by bank type (DYNAMICALLY FETCHED)
     * - Expenses (to be subtracted)
     * - Return Paid from Credit
     * - Return Paid from Balance
     * - Total Amount (Cash + Banks - Expenses - Return Paid from Credit)
     *
     * NOW ACCEPTS DATE RANGE PARAMETERS
     */
    public function getPendingBalances(Request $request)
    {
        $saleman_id = Auth::id();
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";

        // Get total cash from sales
        $salesCashQuery = DB::table('sales')
            ->where('saleman_id', $saleman_id)
            ->where('is_deleted', '!=', 1)
            ->where('Payment_type', 'Cash');

        if (!empty($start_date) && !empty($end_date)) {
            $salesCashQuery->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        $total_sales_cash = $salesCashQuery->sum('paid');

        // Get bank payments from sales grouped by bank_id with bank names
        $salesBankQuery = DB::table('sales')
            ->leftJoin('banks', 'sales.bank_id', '=', 'banks.id')
            ->where('sales.saleman_id', $saleman_id)
            ->where('sales.is_deleted', '!=', 1)
            ->where('sales.Payment_type', 'Bank')
            ->whereNotNull('sales.bank_id');

        if (!empty($start_date) && !empty($end_date)) {
            $salesBankQuery->whereBetween('sales.invoice_date', [$start_date, $end_date]);
        }

        $salesBanks = $salesBankQuery->select([
                'sales.bank_id',
                'banks.bank_name',
                DB::raw('SUM(sales.paid) as total')
            ])
            ->groupBy('sales.bank_id', 'banks.bank_name')
            ->get();

        // Get cash payments from ledger
        $paymentCashQuery = DB::table('saleman_ledgers')
            ->where('salesman_id', $saleman_id)
            ->whereNull('sale_id')
            ->whereNull('sale_return_id')
            ->whereNull('deleted_at')
            ->where(function($q) {
                $q->where('transaction_type', 'credit')
                  ->orWhere('transaction_type', 'Cash');
            });

        if (!empty($start_date) && !empty($end_date)) {
            $paymentCashQuery->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        $total_payment_cash = $paymentCashQuery->sum('credit');

        // Get bank payments from ledger grouped by bank_id with bank names
        $paymentBankQuery = DB::table('saleman_ledgers')
            ->leftJoin('banks', 'saleman_ledgers.bank_id', '=', 'banks.id')
            ->where('saleman_ledgers.salesman_id', $saleman_id)
            ->whereNull('saleman_ledgers.sale_id')
            ->whereNull('saleman_ledgers.sale_return_id')
            ->whereNull('saleman_ledgers.deleted_at')
            ->where(function($q) {
                $q->where('saleman_ledgers.transaction_type', 'bank')
                  ->orWhere('saleman_ledgers.transaction_type', 'Bank');
            })
            ->whereNotNull('saleman_ledgers.bank_id');

        if (!empty($start_date) && !empty($end_date)) {
            $paymentBankQuery->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
        }

        $paymentBanks = $paymentBankQuery->select([
                'saleman_ledgers.bank_id',
                'banks.bank_name',
                DB::raw('SUM(saleman_ledgers.bank) as total')
            ])
            ->groupBy('saleman_ledgers.bank_id', 'banks.bank_name')
            ->get();

        // Get total expenses
        $expenseQuery = DB::table('expenses')
            ->where('created_by', $saleman_id)
            ->where('is_deleted', '!=', 1);

        // Apply date filter if provided
        if (!empty($start_date) && !empty($end_date)) {
            $expenseQuery->whereBetween('exp_date', [$start_date, $end_date]);
        }

        $totalExpense = $expenseQuery->sum('total_amt');

        // Get return amounts
        $returnQuery = DB::table('sale_returns')
            ->where('saleman_id', $saleman_id);

        // Apply date filter if provided
        if (!empty($start_date) && !empty($end_date)) {
            $returnQuery->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        $returnTotals = $returnQuery->select([
                DB::raw('SUM(CASE WHEN paid > 0 THEN paid ELSE 0 END) as return_paid_credit'),
                DB::raw('SUM(CASE WHEN paid = 0 OR paid IS NULL THEN grand_total ELSE 0 END) as return_paid_balance'),
            ])
            ->first();
        
        // Merge bank amounts from sales and payments
        $allBanks = collect();

        // Add sales banks
        foreach ($salesBanks as $bank) {
            $allBanks->push([
                'bank_id' => $bank->bank_id,
                'bank_name' => $bank->bank_name,
                'total' => $bank->total
            ]);
        }

        // Add/merge payment banks
        foreach ($paymentBanks as $bank) {
            $existing = $allBanks->firstWhere('bank_id', $bank->bank_id);
            if ($existing) {
                $index = $allBanks->search(function($item) use ($bank) {
                    return $item['bank_id'] == $bank->bank_id;
                });
                $allBanks[$index]['total'] += $bank->total;
            } else {
                $allBanks->push([
                    'bank_id' => $bank->bank_id,
                    'bank_name' => $bank->bank_name,
                    'total' => $bank->total
                ]);
            }
        }

        // Get approved amounts (cash)
        $approvedCashQuery = DB::table('saleman_ledgers')
            ->where('salesman_id', $saleman_id)
            ->whereNull('deleted_at')
            ->where('status', 'approved')
            ->where(function($q) {
                $q->where('transaction_type', 'Cash')
                  ->orWhere('transaction_type', 'credit');
            });

        if (!empty($start_date) && !empty($end_date)) {
            $approvedCashQuery->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        $approved_cash = $approvedCashQuery->sum('saleman_cash');

        // Get approved amounts (banks) - grouped by bank_id
        $approvedBanksQuery = DB::table('saleman_ledgers')
            ->leftJoin('banks', 'saleman_ledgers.bank_id', '=', 'banks.id')
            ->where('saleman_ledgers.salesman_id', $saleman_id)
            ->whereNull('saleman_ledgers.deleted_at')
            ->where('saleman_ledgers.status', 'approved')
            ->where('saleman_ledgers.transaction_type', 'Bank')
            ->whereNotNull('saleman_ledgers.bank_id');

        if (!empty($start_date) && !empty($end_date)) {
            $approvedBanksQuery->whereBetween('saleman_ledgers.invoice_date', [$start_date, $end_date]);
        }

        $approvedBanks = $approvedBanksQuery->select([
                'saleman_ledgers.bank_id',
                DB::raw('SUM(saleman_ledgers.saleman_cash) as total')
            ])
            ->groupBy('saleman_ledgers.bank_id')
            ->get();

        // Get approved expenses and returns
        $approvedOtherQuery = DB::table('saleman_ledgers')
            ->where('salesman_id', $saleman_id)
            ->whereNull('deleted_at')
            ->where('status', 'approved');

        if (!empty($start_date) && !empty($end_date)) {
            $approvedOtherQuery->whereBetween('invoice_date', [$start_date, $end_date]);
        }

        $approvedOther = $approvedOtherQuery->select([
                DB::raw('SUM(CASE WHEN transaction_type = "expense" THEN saleman_cash ELSE 0 END) as approved_expense'),
                DB::raw('SUM(CASE WHEN transaction_type = "return_credit" THEN saleman_cash ELSE 0 END) as approved_return_credit'),
                DB::raw('SUM(CASE WHEN transaction_type = "return_balance" THEN saleman_cash ELSE 0 END) as approved_return_balance'),
            ])
            ->first();

        // Calculate pending cash
        $total_cash = $total_sales_cash + $total_payment_cash;
        $pending_cash = $total_cash - $approved_cash;

        // Calculate pending for each bank
        $banks_data = [];
        $total_banks_pending = 0;

        foreach ($allBanks as $bank) {
            $approved_for_bank = $approvedBanks->firstWhere('bank_id', $bank['bank_id']);
            $approved_amount = $approved_for_bank ? $approved_for_bank->total : 0;
            $pending_amount = $bank['total'] - $approved_amount;

            $banks_data[] = [
                'bank_id' => $bank['bank_id'],
                'bank_name' => $bank['bank_name'],
                'total' => $bank['total'],
                'approved' => $approved_amount,
                'pending' => $pending_amount
            ];

            $total_banks_pending += $pending_amount;
        }

        // Calculate other pending amounts
        $pending_expenses = $totalExpense - ($approvedOther->approved_expense ?? 0);
        $pending_return_credit = ($returnTotals->return_paid_credit ?? 0) - ($approvedOther->approved_return_credit ?? 0);
        $pending_return_balance = ($returnTotals->return_paid_balance ?? 0) - ($approvedOther->approved_return_balance ?? 0);

        // Total amount = Cash + All Banks - Expenses - Return Paid from Credit
        $total_amounts = $pending_cash + $total_banks_pending - $pending_expenses - $pending_return_credit;

        return response()->json([
            'cash' => $pending_cash,
            'banks' => $banks_data,
            'total_banks' => $total_banks_pending,
            'expenses' => $pending_expenses,
            'return_credit' => $pending_return_credit,
            'return_balance' => $pending_return_balance,
            'total_amounts' => $total_amounts
        ]);
    }

    /**
     * Get all received payments from salesmen for Cashier/Admin
     * Cashier sees: pending (sent by salesman)
     * Admin sees: received (received by cashier)
     * Groups by salesman and submission batch
     */
    public function getReceivedPayments(Request $request)
    {
        // Determine status filter based on user role
        $statusFilter = [];
        if (Auth::user()->hasRole('Cashier')) {
            // Cashier sees pending submissions
            $statusFilter = ['pending'];
        } else {
            // Admin sees received submissions
            $statusFilter = ['received', 'approved'];
        }

        // Get Salesman submissions
        $salesmanSubmissions = DB::table('saleman_ledgers as sl')
            ->join('users', 'sl.salesman_id', '=', 'users.id')
            ->whereNull('sl.sale_id')
            ->whereNull('sl.sale_return_id')
            ->whereNull('sl.deleted_at')
            ->whereIn('sl.status', $statusFilter)
            ->where('sl.detail', '=', 'Send to cashier')
            ->select([
                'sl.salesman_id as source_id',
                DB::raw('"salesman" as source_type'),
                'users.name as source_name',
                DB::raw('DATE(sl.created_at) as submission_date'),
                DB::raw('MIN(sl.id) as batch_id'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "Cash" OR sl.transaction_type = "credit" THEN sl.saleman_cash ELSE 0 END) as cash'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "Bank" AND sl.bank_id = 1 THEN sl.saleman_cash ELSE 0 END) as easypaisa'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "Bank" AND sl.bank_id = 2 THEN sl.saleman_cash ELSE 0 END) as jazzcash'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "Bank" AND sl.bank_id = 3 THEN sl.saleman_cash ELSE 0 END) as meezan'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "Bank" AND sl.bank_id = 4 THEN sl.saleman_cash ELSE 0 END) as upaisa'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "expense" THEN sl.saleman_cash ELSE 0 END) as expenses'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "return_credit" THEN sl.saleman_cash ELSE 0 END) as return_credit'),
                DB::raw('SUM(CASE WHEN sl.transaction_type = "return_balance" THEN sl.saleman_cash ELSE 0 END) as return_balance'),
                DB::raw('MAX(sl.status) as status'),
                DB::raw('MAX(sl.created_at) as created_at')
            ])
            ->groupBy('sl.salesman_id', 'users.name', DB::raw('DATE(sl.created_at)'))
            ->get();

        // Get Order Booker submissions
        $orderBookerSubmissions = DB::table('order_booker_ledgers as obl')
            ->join('users', 'obl.created_by_user', '=', 'users.id')
            ->whereNull('obl.deleted_at')
            ->whereIn('obl.status', $statusFilter)
            ->where('obl.detail', '=', 'Send to cashier')
            ->select([
                'obl.created_by_user as source_id',
                DB::raw('"order_booker" as source_type'),
                'users.name as source_name',
                DB::raw('DATE(obl.created_at) as submission_date'),
                DB::raw('MIN(obl.id) as batch_id'),
                DB::raw('obl.paid_amount as cash'),
                DB::raw('0 as easypaisa'),
                DB::raw('0 as jazzcash'),
                DB::raw('0 as meezan'),
                DB::raw('0 as upaisa'),
                DB::raw('obl.expense_amount as expenses'),
                DB::raw('0 as return_credit'),
                DB::raw('0 as return_balance'),
                DB::raw('MAX(obl.status) as status'),
                DB::raw('MAX(obl.created_at) as created_at')
            ])
            ->groupBy('obl.created_by_user', 'users.name', DB::raw('DATE(obl.created_at)'), 'obl.paid_amount', 'obl.expense_amount')
            ->get();

        // Combine both submissions
        $allSubmissions = $salesmanSubmissions->concat($orderBookerSubmissions)->sortByDesc('created_at');

        $all_data = [];
        foreach ($allSubmissions as $submission) {
            // Calculate total
            $total = ($submission->cash + $submission->easypaisa + $submission->jazzcash +
                     $submission->meezan + $submission->upaisa) - $submission->expenses -
                     $submission->return_credit;

            $statusLabel = '';
            if ($submission->status === 'pending') {
                $statusLabel = '<span class="label label-warning">Pending ('.ucfirst(str_replace('_', ' ', $submission->source_type)).')</span>';
            } elseif ($submission->status === 'received') {
                $statusLabel = '<span class="label label-info">Received (Cashier)</span>';
            } else {
                $statusLabel = '<span class="label label-success">Approved (Admin)</span>';
            }

            $btn = '<td class="text-right">';
            $btn .= '<button class="btn btn-info btn-icon view_details_btn" data-batch="'.$submission->batch_id.'" data-source-type="'.$submission->source_type.'" data-source-id="'.$submission->source_id.'" data-date="'.$submission->submission_date.'" data-toggle="tooltip" title="View Details" style="margin-right: 3px"><i class="fa-duotone fa-eye"></i></button>';
            $btn .= '</td>';

            $all_data[] = [
                'batch_id'       => $submission->batch_id,
                'source_type'    => ucfirst(str_replace('_', ' ', $submission->source_type)),
                'source_name'    => $submission->source_name,
                'date'           => Carbon::parse($submission->created_at)->format('d-m-Y H:i'),
                'cash'           => number_format($submission->cash, 0),
                'easypaisa'      => number_format($submission->easypaisa, 0),
                'jazzcash'       => number_format($submission->jazzcash, 0),
                'meezan'         => number_format($submission->meezan, 0),
                'upaisa'         => number_format($submission->upaisa, 0),
                'expenses'       => number_format($submission->expenses, 0),
                'return_credit'  => number_format($submission->return_credit, 0),
                'return_balance' => number_format($submission->return_balance, 0),
                'total'          => number_format($total, 0),
                'status'         => $statusLabel,
                'btn'            => $btn
            ];
        }

        return response()->json([
            'data' => $all_data
        ]);
    }

    /**
     * Get details of a specific submission batch
     */
    public function getSubmissionDetails($source_type, $source_id, $date)
    {
        if ($source_type === 'salesman') {
            $submissions = DB::table('saleman_ledgers')
                ->where('salesman_id', $source_id)
                ->whereDate('created_at', $date)
                ->where('detail', '=', 'Send to cashier')
                ->whereNull('deleted_at')
                ->get();

            if ($submissions->isEmpty()) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Submission not found'
                ], 404);
            }

            $user = User::find($source_id);

            $details = [
                'source_type' => 'Salesman',
                'source_name' => $user->name,
                'date' => Carbon::parse($submissions->first()->created_at)->format('d-m-Y H:i'),
                'status' => $submissions->first()->status,
                'cash' => 0,
                'easypaisa' => 0,
                'jazzcash' => 0,
                'meezan' => 0,
                'upaisa' => 0,
                'expenses' => 0,
                'return_credit' => 0,
                'return_balance' => 0,
            ];

            foreach ($submissions as $sub) {
                if ($sub->transaction_type === 'Cash' || $sub->transaction_type === 'credit') {
                    $details['cash'] += $sub->saleman_cash;
                } elseif ($sub->transaction_type === 'Bank') {
                    if ($sub->bank_id == 1) $details['easypaisa'] += $sub->saleman_cash;
                    elseif ($sub->bank_id == 2) $details['jazzcash'] += $sub->saleman_cash;
                    elseif ($sub->bank_id == 3) $details['meezan'] += $sub->saleman_cash;
                    elseif ($sub->bank_id == 4) $details['upaisa'] += $sub->saleman_cash;
                } elseif ($sub->transaction_type === 'expense') {
                    $details['expenses'] += $sub->saleman_cash;
                } elseif ($sub->transaction_type === 'return_credit') {
                    $details['return_credit'] += $sub->saleman_cash;
                } elseif ($sub->transaction_type === 'return_balance') {
                    $details['return_balance'] += $sub->saleman_cash;
                }
            }

            $details['total'] = ($details['cash'] + $details['easypaisa'] + $details['jazzcash'] +
                                $details['meezan'] + $details['upaisa']) - $details['expenses'] -
                                $details['return_credit'];

            return response()->json([
                'status' => 200,
                'details' => $details,
                'submission_ids' => $submissions->pluck('id')->toArray()
            ]);

        } elseif ($source_type === 'order_booker') {
            $submissions = DB::table('order_booker_ledgers')
                ->where('created_by_user', $source_id)
                ->whereDate('created_at', $date)
                ->where('detail', '=', 'Send to cashier')
                ->whereNull('deleted_at')
                ->get();

            if ($submissions->isEmpty()) {
                return response()->json([
                    'status' => 404,
                    'message' => 'Submission not found'
                ], 404);
            }

            $user = User::find($source_id);
            $submission = $submissions->first();

            $details = [
                'source_type' => 'Order Booker',
                'source_name' => $user->name,
                'date' => Carbon::parse($submission->created_at)->format('d-m-Y H:i'),
                'status' => $submission->status,
                'cash' => $submission->paid_amount,
                'easypaisa' => 0,
                'jazzcash' => 0,
                'meezan' => 0,
                'upaisa' => 0,
                'expenses' => $submission->expense_amount,
                'return_credit' => 0,
                'return_balance' => 0,
            ];

            $details['total'] = $details['cash'] - $details['expenses'];

            return response()->json([
                'status' => 200,
                'details' => $details,
                'submission_ids' => $submissions->pluck('id')->toArray()
            ]);
        }

        return response()->json([
            'status' => 400,
            'message' => 'Invalid source type'
        ], 400);
    }

    /**
     * Receive a submission batch (Cashier action)
     * Changes status from 'pending' to 'received'
     * This sends the submission to Admin for approval
     */
    public function receiveSubmission(Request $request)
    {
        $source_type = $request->input('source_type', 'salesman');
        $source_id = $request->input('source_id') ?? $request->input('salesman_id'); // Support both for backward compatibility
        $date = $request->input('date');

        if ($source_type === 'salesman') {
            // Update all records in this batch to received
            $updated = DB::table('saleman_ledgers')
                ->where('salesman_id', $source_id)
                ->whereDate('created_at', $date)
                ->where('detail', '=', 'Send to cashier')
                ->where('status', '=', 'pending')
                ->whereNull('deleted_at')
                ->update([
                    'status' => 'received',
                    'received_at' => now(),
                    'received_by' => Auth::id()
                ]);
        } elseif ($source_type === 'order_booker') {
            $updated = DB::table('order_booker_ledgers')
                ->where('created_by_user', $source_id)
                ->whereDate('created_at', $date)
                ->where('detail', '=', 'Send to cashier')
                ->where('status', '=', 'pending')
                ->whereNull('deleted_at')
                ->update([
                    'status' => 'received',
                    'received_at' => now(),
                    'received_by' => Auth::id()
                ]);
        } else {
            return response()->json([
                'status' => 400,
                'message' => 'Invalid source type'
            ], 400);
        }

        if ($updated > 0) {
            return response()->json([
                'status' => 200,
                'message' => 'Payment received successfully and sent to Admin for approval!'
            ]);
        } else {
            return response()->json([
                'status' => 404,
                'message' => 'Submission not found or already received'
            ], 404);
        }
    }

    /**
     * Approve a submission batch (Admin action)
     * Changes status from 'received' to 'approved'
     * Also updates cash/bank balances and tracks expenses
     */
    public function approveSubmission(Request $request)
    {
        $source_type = $request->input('source_type', 'salesman');
        $source_id = $request->input('source_id') ?? $request->input('salesman_id'); // Support both for backward compatibility
        $date = $request->input('date');

        DB::beginTransaction();
        try {
            if ($source_type === 'salesman') {
                // Get all records in this batch before updating
                $submissions = DB::table('saleman_ledgers')
                    ->where('salesman_id', $source_id)
                    ->whereDate('created_at', $date)
                    ->where('detail', '=', 'Send to cashier')
                    ->where('status', '=', 'received')
                    ->whereNull('deleted_at')
                    ->get();

                if ($submissions->isEmpty()) {
                    return response()->json([
                        'status' => 404,
                        'message' => 'Submission not found or not yet received by Cashier'
                    ], 404);
                }

                // Process each submission and update cash/bank/expense
                foreach ($submissions as $sub) {
                    // Cash transactions
                    if ($sub->transaction_type === 'Cash' || $sub->transaction_type === 'credit') {
                        // Add to hand cash
                        DB::table('hand_cashes')->increment('cash_amount', $sub->saleman_cash);

                        // Record transaction
                        DB::table('transactions')->insert([
                            'payment_type'      => 'Cash',
                            'amount'            => $sub->saleman_cash,
                            'transaction_type'  => 'deposit',
                            'source_type'       => 'Salesman Submission',
                            'source_id'         => $sub->id,
                            'description'       => 'Salesman Cash Submission Approved',
                            'created_at'        => now(),
                        ]);
                    }
                    // Bank transactions
                    elseif ($sub->transaction_type === 'Bank' && $sub->bank_id) {
                        // Add to bank balance
                        DB::table('banks')->where('id', $sub->bank_id)
                            ->increment('bank_balance', $sub->saleman_cash);

                        // Record transaction
                        DB::table('transactions')->insert([
                            'payment_type'      => 'Bank',
                            'bank_id'           => $sub->bank_id,
                            'amount'            => $sub->saleman_cash,
                            'transaction_type'  => 'deposit',
                            'source_type'       => 'Salesman Submission',
                            'source_id'         => $sub->id,
                            'description'       => 'Salesman Bank Submission Approved',
                            'created_at'        => now(),
                        ]);
                    }
                    // Expense transactions - these are deducted from cash
                    elseif ($sub->transaction_type === 'expense') {
                        // Expense is already deducted by salesman, we just record the transaction
                        DB::table('transactions')->insert([
                            'payment_type'      => 'Cash',
                            'amount'            => $sub->saleman_cash,
                            'transaction_type'  => 'expense',
                            'source_type'       => 'Salesman Expense',
                            'source_id'         => $sub->id,
                            'description'       => 'Salesman Expense Approved',
                            'created_at'        => now(),
                        ]);
                    }
                }

                // Update all records in this batch to approved
                $updated = DB::table('saleman_ledgers')
                    ->where('salesman_id', $source_id)
                    ->whereDate('created_at', $date)
                    ->where('detail', '=', 'Send to cashier')
                    ->where('status', '=', 'received')
                    ->whereNull('deleted_at')
                    ->update([
                        'status' => 'approved',
                        'approved_at' => now(),
                        'approved_by' => Auth::id()
                    ]);

            } elseif ($source_type === 'order_booker') {
                // Get all records in this batch before updating
                $submissions = DB::table('order_booker_ledgers')
                    ->where('created_by_user', $source_id)
                    ->whereDate('created_at', $date)
                    ->where('detail', '=', 'Send to cashier')
                    ->where('status', '=', 'received')
                    ->whereNull('deleted_at')
                    ->get();

                if ($submissions->isEmpty()) {
                    return response()->json([
                        'status' => 404,
                        'message' => 'Submission not found or not yet received by Cashier'
                    ], 404);
                }

                // Process each submission and update cash/bank/expense
                foreach ($submissions as $sub) {
                    // Cash amount (paid_amount goes to cash if no bank_id, otherwise to bank)
                    if ($sub->paid_amount > 0) {
                        if ($sub->bank_id) {
                            // Add to bank balance
                            DB::table('banks')->where('id', $sub->bank_id)
                                ->increment('bank_balance', $sub->paid_amount);

                            // Record transaction
                            DB::table('transactions')->insert([
                                'payment_type'      => 'Bank',
                                'bank_id'           => $sub->bank_id,
                                'amount'            => $sub->paid_amount,
                                'transaction_type'  => 'deposit',
                                'source_type'       => 'Order Booker Submission',
                                'source_id'         => $sub->id,
                                'description'       => 'Order Booker Bank Submission Approved',
                                'created_at'        => now(),
                            ]);
                        } else {
                            // Add to hand cash
                            DB::table('hand_cashes')->increment('cash_amount', $sub->paid_amount);

                            // Record transaction
                            DB::table('transactions')->insert([
                                'payment_type'      => 'Cash',
                                'amount'            => $sub->paid_amount,
                                'transaction_type'  => 'deposit',
                                'source_type'       => 'Order Booker Submission',
                                'source_id'         => $sub->id,
                                'description'       => 'Order Booker Cash Submission Approved',
                                'created_at'        => now(),
                            ]);
                        }
                    }

                    // Expense amount
                    if ($sub->expense_amount > 0) {
                        // Expense is already deducted by order booker, we just record the transaction
                        DB::table('transactions')->insert([
                            'payment_type'      => 'Cash',
                            'amount'            => $sub->expense_amount,
                            'transaction_type'  => 'expense',
                            'source_type'       => 'Order Booker Expense',
                            'source_id'         => $sub->id,
                            'description'       => 'Order Booker Expense Approved',
                            'created_at'        => now(),
                        ]);
                    }
                }

                // Update all records in this batch to approved
                $updated = DB::table('order_booker_ledgers')
                    ->where('created_by_user', $source_id)
                    ->whereDate('created_at', $date)
                    ->where('detail', '=', 'Send to cashier')
                    ->where('status', '=', 'received')
                    ->whereNull('deleted_at')
                    ->update([
                        'status' => 'approved',
                        'approved_at' => now(),
                        'approved_by' => Auth::id()
                    ]);
            } else {
                return response()->json([
                    'status' => 400,
                    'message' => 'Invalid source type'
                ], 400);
            }

            DB::commit();

            return response()->json([
                'status' => 200,
                'message' => 'Payment approved successfully! Cash and Bank balances updated.'
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Error in approveSubmission: ' . $e->getMessage());
            return response()->json([
                'status' => 500,
                'message' => 'Error approving submission: ' . $e->getMessage()
            ], 500);
        }
    }

}