<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class StockController extends Controller
{
    

    public function load_stock(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
    
        // Map DataTable columns to database columns
        $columns = [
            'stocks.pro_id',
            'products.pro_name',
            'total_stock_in_qty',
            'total_stock_out_qty',

            'balance',
        ];
    
        $order_by = isset($columns[$column]) ? $columns[$column] : 'stocks.pro_id';
    
      
        $query = DB::table('stocks')
        ->join('products', 'stocks.pro_id', '=', 'products.id')
        ->select([
            'stocks.pro_id',
            'products.pro_name',
            'products.sku',

            // Sale aur Return ko alag treat karo - with NULL handling
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Purchase' THEN stocks.stock_in_qty
            ELSE 0 END ), 0) as total_purchase_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Issue Order' THEN stocks.stock_out_qty
            ELSE 0 END ), 0) as total_issue_order_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Order Return' THEN stocks.stock_in_qty
            ELSE 0 END ), 0) as total_return_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Sale Return' THEN stocks.stock_in_qty
            ELSE 0 END ), 0) as total_sale_return_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Production Issue' THEN stocks.stock_out_qty
            ELSE 0 END ), 0) as total_production_issue_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Production Completed' THEN stocks.stock_in_qty
            ELSE 0 END ), 0) as total_production_complete_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Repair' THEN stocks.stock_in_qty
            ELSE 0 END ), 0) as total_repair_qty "),
            DB::raw(" COALESCE(SUM( CASE WHEN stocks.adjustment = 'Repair Used' THEN stocks.stock_out_qty
            ELSE 0 END ), 0) as total_repair_used_qty "),
           

            // Normal in/out ka sum (sab combine)
            DB::raw('COALESCE(SUM(stocks.stock_in_qty), 0) as total_stock_in_qty'),
            DB::raw('COALESCE(SUM(stocks.stock_out_qty), 0) as total_stock_out_qty'),

            // Balance nikalne ka smart formula - ensure NULL is handled
            DB::raw('COALESCE(SUM(stocks.stock_in_qty), 0) - COALESCE(SUM(stocks.stock_out_qty), 0) as balance')
        ])
        ->groupBy('stocks.pro_id', 'products.pro_name', 'products.sku');
    
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('stocks.invoice_date', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('stocks.is_active', '=', $type);
        }
    
        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        foreach ($purchases as $data) {
            // Balance is calculated in SQL with COALESCE, but ensure it's numeric
            $balance = is_numeric($data->balance) ? (float)$data->balance : 0;
            
            $all_data[] = [
                'id' => $data->pro_id,
                'pro_name' => $data->pro_name ?? '',
                // 'sku' => $data->sku,
                'total_purchase_qty' => is_numeric($data->total_purchase_qty) ? (float)$data->total_purchase_qty : 0,
                'total_production_complete_qty' => is_numeric($data->total_production_complete_qty) ? (float)$data->total_production_complete_qty : 0,
                'total_repair_qty' => is_numeric($data->total_repair_qty) ? (float)$data->total_repair_qty : 0,
                'total_sale_return_qty' => is_numeric($data->total_sale_return_qty) ? (float)$data->total_sale_return_qty : 0,
                // order return 
                'total_return_qty' => is_numeric($data->total_return_qty) ? (float)$data->total_return_qty : 0,
                'total_issue_order_qty' => is_numeric($data->total_issue_order_qty) ? (float)$data->total_issue_order_qty : 0,
                'total_production_issue_qty' => is_numeric($data->total_production_issue_qty) ? (float)$data->total_production_issue_qty : 0,
                
                'total_repair_used_qty' => is_numeric($data->total_repair_used_qty) ? (float)$data->total_repair_used_qty : 0,

                'balance' => $balance,
            ];
        }
    
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data
        ];
    
        return response()->json($data);
    }



    public function load_saleman_stock(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
    
        // Map DataTable columns to database columns
        $columns = [
            'saleman_stocks.pro_id',
            'products.pro_name',
            'total_stock_in_qty',
            'total_stock_out_qty',

            'balance',
        ];
    
        $order_by = isset($columns[$column]) ? $columns[$column] : 'saleman_stocks.pro_id';
    
        $query = DB::table('saleman_stocks')
        ->join('products', 'saleman_stocks.pro_id', '=', 'products.id')
        ->join('users', 'saleman_stocks.salesman_id', '=', 'users.id')
        ->select([
            'saleman_stocks.pro_id',
            'products.pro_name',
            'products.sku',

            // Sale aur Return ko alag treat karo
            DB::raw("SUM( CASE  WHEN saleman_stocks.adjustment = 'Sale' THEN saleman_stocks.stock_out_qty
            ELSE 0 END ) as total_sales_qty "),
            DB::raw(" SUM( CASE WHEN saleman_stocks.adjustment = 'Order Return' THEN saleman_stocks.stock_out_qty
            ELSE 0 END ) as total_return_qty "),

            // Normal in/out ka sum (sab combine)
            DB::raw('SUM(saleman_stocks.stock_in_qty) as total_stock_in_qty'),
            DB::raw('SUM(saleman_stocks.stock_out_qty) as total_stock_out_qty'),

            // Balance nikalne ka smart formula
            DB::raw('SUM(saleman_stocks.stock_in_qty) - SUM(saleman_stocks.stock_out_qty) as balance')
        ])
        ->groupBy('saleman_stocks.pro_id', 'products.pro_name', 'products.sku');


        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('saleman_stocks.invoice_date', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('stocks.is_active', '=', $type);
        }
    
        $salesman = $request->input('search_by_salesman');
        if (isset($salesman)) {
            $query->where('users.id', '=', $salesman);
        }

        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        foreach ($purchases as $data) {
            $all_data[] = [
                'id'                    => $data->pro_id,
                'pro_name'              => $data->pro_name,
                'sku'                    => $data->sku,
                'total_stock_in_qty'    => $data->total_stock_in_qty,
                'total_sales_qty'   => $data->total_sales_qty,
                'total_return_qty'   => $data->total_return_qty,

                'balance'               => $data->balance,
            ];
        }
     
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data
        ];
    
        return response()->json($data);
    }



    public function getSalesmanStock(Request $request)
    {
        $salesman_id = $request->salesman_id;

        $stock = DB::table('saleman_stocks')
            ->select('pro_id',
                DB::raw('SUM(stock_in_qty) as stock_in_qty'),
                DB::raw('SUM(stock_out_qty) as stock_out_qty')
            )
            ->where('salesman_id', $salesman_id)
            ->groupBy('pro_id')
            ->get()
            ->map(function($row) {
                return [
                    'pro_id' => $row->pro_id,
                    'available_stock' => ($row->stock_in_qty ?? 0) - ($row->stock_out_qty ?? 0)
                ];
            });

        return response()->json($stock);
    }


    public function load_non_serviceable_stock(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
    
        // Map DataTable columns to database columns
        $columns = [
            'non_service_stocks.pro_id',
            'products.pro_name',
            'total_stock_in_qty',
            'total_stock_out_qty',
            'balance',
        ];
    
        $order_by = isset($columns[$column]) ? $columns[$column] : 'non_service_stocks.pro_id';
    
      
        $query = DB::table('non_service_stocks')
        ->join('products', 'non_service_stocks.pro_id', '=', 'products.id')
        ->select([
            'non_service_stocks.pro_id',
            'products.pro_name',
            'products.sku',

            // Sale Return qty with date
            DB::raw(" SUM( CASE WHEN non_service_stocks.adjustment = 'Sale Return' THEN non_service_stocks.stock_in_qty
            ELSE 0 END ) as total_sale_return_qty "),
            DB::raw(" MAX( CASE WHEN non_service_stocks.adjustment = 'Sale Return' THEN non_service_stocks.invoice_date
            ELSE NULL END ) as sale_return_date "),
            
            // Receiving qty from store (replaced Repair qty - includes both new and old adjustments for backward compatibility)
            DB::raw(" SUM( CASE WHEN non_service_stocks.adjustment IN ('Receiving from Store', 'Repair') THEN non_service_stocks.stock_out_qty
            ELSE 0 END ) as receiving_qty_from_store "),
        
            // Normal in/out ka sum (sab combine)
            DB::raw('SUM(non_service_stocks.stock_in_qty) as total_stock_in_qty'),
            DB::raw('SUM(non_service_stocks.stock_out_qty) as total_stock_out_qty'),

            // Balance: Sale Return qty - Receiving qty from store (should be 0 when all items received from store)
            DB::raw(" (SUM( CASE WHEN non_service_stocks.adjustment = 'Sale Return' THEN non_service_stocks.stock_in_qty ELSE 0 END ) 
            - SUM( CASE WHEN non_service_stocks.adjustment IN ('Receiving from Store', 'Repair') THEN non_service_stocks.stock_out_qty ELSE 0 END )) as balance ")
        ])
        ->groupBy('non_service_stocks.pro_id', 'products.pro_name', 'products.sku');

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('non_service_stocks.invoice_date', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('stocks.is_active', '=', $type);
        }
    
        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        foreach ($purchases as $data) {
            // Balance is calculated in SQL: Sale Return qty - Receiving qty from store
            // Ensure balance shows 0 (not null or negative) when items are received from store
            $balance = $data->balance ?? 0;
            $balance = max(0, $balance); // Ensure balance never goes negative
            
            $all_data[] = [
                'id'                    => $data->pro_id,
                'pro_name'              => $data->pro_name,
                'sku'                   => $data->sku,
                'total_sale_return_qty' => $data->total_sale_return_qty ?? 0,
                'sale_return_date'      => $data->sale_return_date ? Carbon::parse($data->sale_return_date)->format('d-m-Y') : '',
                'receiving_qty_from_store' => $data->receiving_qty_from_store ?? 0,

                'balance'               => $balance,
            ];
        }
    
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data
        ];
    
        return response()->json($data);
    }


    // reports 
    public function load_main_stock_report(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
       $order_by   = $request->input("columns.$column.data");
    
      
        // Build the query with potential filters
        $query = DB::table('stocks')
        ->leftJoin('products', 'stocks.pro_id', '=', 'products.id')

        ->select([
            'stocks.*', 
            'products.pro_name',  
            'products.sku',

            'stocks.id as id',
        ]);
    
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('stocks.created_at', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('stocks.is_active', '=', $type);
        }
    
        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        foreach ($purchases as $data) {
            $all_data[] = [
                'id' => $data->id,
                'pro_name' => $data->pro_name,
                'sku' => $data->sku,
                'created_at' => $data->created_at,
                'stock_in_qty' => $data->stock_in_qty,
                'stock_out_qty' => $data->stock_out_qty, 
                'adjustment' => $data->adjustment,
                
            ];
        }
    
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data
        ];
    
        return response()->json($data);
    }

    public function load_non_serviceable_stock_report(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
       $order_by   = $request->input("columns.$column.data");
    
      
        // Build the query with potential filters
        $query = DB::table('non_service_stocks')
        ->leftJoin('products', 'non_service_stocks.pro_id', '=', 'products.id')

        ->select([
            'non_service_stocks.*', 
            'products.pro_name',  
            'products.sku',

            'non_service_stocks.id as id',
        ]);
    
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('non_service_stocks.created_at', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('non_service_stocks.is_active', '=', $type);
        }
    
        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        foreach ($purchases as $data) {
            $all_data[] = [
                'id' => $data->id,
                'pro_name' => $data->pro_name,
                'sku' => $data->sku,
                'created_at' => $data->created_at,
                'stock_in_qty' => $data->stock_in_qty,
                'stock_out_qty' => $data->stock_out_qty, 
                'adjustment' => $data->adjustment,
                
            ];
        }
    
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data
        ];
    
        return response()->json($data);
    }

    public function load_profit_report(Request $request)
    {
        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('enddate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length', 10);
        $offset     = $request->input('start', 0);
        $column     = $request->input('order.0.column', 0);
        $dir        = $request->input('order.0.dir', 'asc');
        $order_by   = $request->input("columns.$column.data");
    
      
        // Build the query with potential filters
        $query = DB::table('saleman_stocks')
        ->leftJoin('products', 'saleman_stocks.pro_id', '=', 'products.id')

        ->select([
            'saleman_stocks.*', 
            'products.pro_name',  
            'products.sku',

            'saleman_stocks.id as id',
        ])->where('saleman_stocks.adjustment', 'Sale'); // ✅ sirf Sale wali rows;
    
        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('saleman_stocks.created_at', [$start_date, $end_date]);
        }
    
        if ($search = $request->input('search')) {
            $query->where(function ($q) use ($search) {
                $q->where('products.pro_name', 'like', '%' . $search . '%');
            });
        }
    
        if ($type = $request->input('type')) {
            $query->where('saleman_stocks.is_active', '=', $type);
        }
    
        $total_count = $query->count();
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }
    
        $purchases = $query->get();
    
        $all_data = [];
        $total_profit_sum = 0; // yahan numeric total collect karo

        foreach ($purchases as $data) {

            $sku = $data->sku;
            $short_sku = (strlen($sku) > 7) ? substr($sku, 0, 7) . '...' : $sku;

            $single_profit = $data->sale_price - $data->purchase_price;
            $total_profit = $single_profit * $data->stock_out_qty;
            // Total profit ko ek variable me sum karo (raw number me)
            $total_profit_sum += $total_profit;

            $all_data[] = [
                'id' => $data->id,
                'pro_name' => $data->pro_name,
                'sku' => '<span title="'.$data->sku.'">'.$short_sku.'</span>',
                'purchase_price' => $data->purchase_price,
                'sale_price' => $data->sale_price,
                'pro_qty' => $data->stock_out_qty, 
                'adjustment' => $data->adjustment,
                'single_profit' => number_format($single_profit, 2),    
                'total_profit'  => number_format($total_profit, 2),    
                'created_at' => $data->created_at,
                
            ];
        } 
    
        $data = [
            "draw" => intval($request->input('draw')),
            "recordsTotal" => $total_count,
            "recordsFiltered" => $total_count,
            "data" => $all_data,
            "total_profit"    => number_format($total_profit_sum, 2), // ✅ properly formatted number
        ];
    
        return response()->json($data);
    }
    




}
