<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use App\Models\SupplierLedger;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class SupplierController extends Controller
{
    

    public function load_suppliers(Request $request)
    {
        // $users = User::all();

        // return view('admin.users', compact('users'));

        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

        $query = DB::table('suppliers')
        ->leftJoin('purchases', 'suppliers.id', '=', 'purchases.supplier_id')
        ->select(
            'suppliers.id',
            'suppliers.sup_name',
            'suppliers.company_name',
            'suppliers.phone',
            'suppliers.sup_image',


            DB::raw('SUM(purchases.grand_total) as total_amount'),
            DB::raw('SUM(purchases.paid_amount) as total_paid_amount'),
            DB::raw('SUM(purchases.remain_amount) as total_remain_amount'),

        )
        ->groupBy('suppliers.id',
         'suppliers.sup_name',
         'suppliers.company_name',
         'suppliers.phone',
         'suppliers.cnic',
         'suppliers.email',
         'suppliers.address',
         'suppliers.sup_image',
         'suppliers.created_by',
         'suppliers.updated_by',



        );


        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('created_at', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('suppliers.sup_name', 'like', '%' . $search . '%');
        }



        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $users = $query->get();

   

        $all_data = [];
        foreach ($users as $user) {

           

            $btn = '<td class="text-right">';

            if (Auth::user()->can('view_supplier_ledger')) {
                $btn .= '<a href="' . route("supplier_ledger", $user->id) . '" class="btn btn-success btn-icon" data-toggle="tooltip" title="View" style="padding: 6px 10px;"><i class="fa-duotone fa-memo"></i></a>';

            }
            if (Auth::user()->can('update_supplier')) {
                $btn .= '<button id="edit_btn" data-eid="'.$user->id.'" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit" style="margin-right: 3px"><i class="fa-duotone fa-pen-to-square"></i></button>';
            }
            if (Auth::user()->can('delete_supplier')) {
                $btn .= '<button id="delete_btn" data-did="'.$user->id.'" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>';
            }
            $btn .= '</td>';

            if($user->sup_image != ""){

                $image = '<img   height="50px" src="uploads/sup_image/'.$user->sup_image.'">';

            }else{

                $image = '<img  " height="50px" src="uploads/no_image.png">';

            }
          
            $all_data[] = [
                'id'                => $user->id,
                'company_name'      => $user->company_name,
                'sup_name'          => $user->sup_name,
                'phone'             => $user->phone,
                'total_amount'             => $user->total_amount,
                'total_paid_amount'              => $user->total_paid_amount,
                'total_remain_amount'              => $user->total_remain_amount,
                'sup_image'         => $image,
                'btn'               => $btn
            ];
        };

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
        ];

        return response()->json($data);


    }

    public function insert_supplier(Request $request)
    {
        $user_cnic = Supplier::where('cnic', $request->input('cnic'))->exists();

        if($user_cnic){

            return response()->json([
                'status'=> "cnic_Exist",
            ]);
            
        }else{

            if ($request->hasFile('sup_image')) {
                $image1 = $request->file('sup_image');
                $sup_image = hexdec(uniqid()).'.'.$image1->getClientOriginalExtension();
                $image1->move(public_path('/uploads/sup_image/'), $sup_image);
            } else {
                $sup_image = null; // Set to null if no front image is provided
            }


            $data = new Supplier();
            $data->sup_name         = $request->input('sup_name');
            $data->phone        = $request->input('phone');
            $data->email        = $request->input('email');
            $data->cnic        = $request->input('cnic');
            $data->address        = $request->input('address');
            $data->company_name        = $request->input('company_name');

            $data->sup_image        = $sup_image;


            $data->created_by   = Auth::id();

            $data->save();

            return response()->json([
                'status'=>200
            ]);

        }
    }

    public function edit_supplier($id)
    {
        // Retrieve the sale with customer details
        $data = Supplier::findOrFail($id);

         
        return response()->json([
            'status' => 200,
            'data' => $data, 
        ]);
    }

    public function update_supplier(Request $request)
    {
        $user_cnic =  Supplier::where('cnic', $request->input('cnic'))
        ->where('id', '!=', $request->input('sup_id'))->exists();

        if($user_cnic){

            return response()->json([
                'status'=> "cnic_Exist",
            ]);
            
        }else{
            $user = Supplier::findOrFail($request->input('sup_id'));
            if ($request->hasFile('sup_image')) {
                // Check and delete old CNIC front image
                if ($user->sup_image && file_exists(public_path('/uploads/sup_image/' . $user->sup_image))) {
                    unlink(public_path('/uploads/sup_image/' . $user->sup_image));
                }
            
                // Upload the new CNIC front image
                $image1 = $request->file('sup_image');
                $sup_image = hexdec(uniqid()) . '.' . $image1->getClientOriginalExtension();
                $image1->move(public_path('/uploads/sup_image/'), $sup_image);
                $user->sup_image = $sup_image;
            }
            
            $user->sup_name         = $request->input('sup_name');
            $user->phone        = $request->input('phone');
            $user->cnic        = $request->input('cnic');
            $user->email        = $request->input('email');
            $user->address        = $request->input('address');
            $user->company_name        = $request->input('company_name');

            $user->updated_by   = Auth::id();


            $user->save();
            

            return response()->json([
                'status'=>200
            ]);

        }
    }

    public function delete_supplier($id)
    {
        $data = Supplier::find($id);

        $data->delete();

        return response()->json([
            'status'=>200
        ]);
    }

    public function get_suppliers()
    {
        $supplier = Supplier::all();

        return response()->json($supplier);
    }

    public function get_suppliers_with_purchases()
    {
        // Get suppliers who have purchases
        $suppliers = DB::table('suppliers')
            ->join('purchases', 'suppliers.id', '=', 'purchases.supplier_id')
            ->select('suppliers.id', 'suppliers.sup_name', 'suppliers.company_name')
            ->distinct()
            ->orderBy('suppliers.sup_name', 'asc')
            ->get();

        return response()->json($suppliers);
    }

    public function supplier()
    {
        try {
            $supplier_id = null;
            
            // If user has Supplier role, find their supplier record and show ledger
            if (Auth::user()->hasRole('Supplier')) {
                $user = Auth::user();
                $supplier = null;
                
                // First try to find by created_by (most reliable match)
                $supplier = Supplier::where('created_by', $user->id)->first();
                
                // If not found, try to find by email match
                if (!$supplier && $user->email) {
                    $supplier = Supplier::where('email', $user->email)->first();
                }
                
                // If not found, try to extract username from email and match
                if (!$supplier && $user->email) {
                    $emailUsername = strtolower(explode('@', $user->email)[0]);
                    $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                    
                    if (!$supplier) {
                        $suppliers = Supplier::all();
                        foreach ($suppliers as $sup) {
                            $supNameClean = strtolower(preg_replace('/\s+/', '', $sup->sup_name ?? ''));
                            $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                            if ($supNameClean === $emailUsernameClean || 
                                strpos($supNameClean, $emailUsernameClean) !== false ||
                                strpos($emailUsernameClean, $supNameClean) !== false) {
                                $supplier = $sup;
                                break;
                            }
                        }
                    }
                }
                
                // If not found by email, try to find by matching name (case-insensitive)
                if (!$supplier && $user->name) {
                    $supplier = Supplier::whereRaw('LOWER(sup_name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
                }
                
                // If still not found, try exact match ignoring case
                if (!$supplier && $user->name) {
                    $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) = ?', [strtolower(trim($user->name))])->first();
                }
                
                // If still not found, try partial match (remove spaces and compare)
                if (!$supplier && $user->name) {
                    $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                    $suppliers = Supplier::all();
                    foreach ($suppliers as $sup) {
                        $supNameClean = preg_replace('/\s+/', '', strtolower($sup->sup_name ?? ''));
                        
                        // Exact match
                        if ($userNameClean === $supNameClean) {
                            $supplier = $sup;
                            break;
                        }
                        // Check if one contains the other
                        if (strpos($supNameClean, $userNameClean) !== false || 
                            strpos($userNameClean, $supNameClean) !== false) {
                            $supplier = $sup;
                            break;
                        }
                        // Similar match (using similar_text for fuzzy matching)
                        $similarity = 0;
                        similar_text($userNameClean, $supNameClean, $similarity);
                        if ($similarity > 80) { // 80% similarity threshold
                            $supplier = $sup;
                            break;
                        }
                    }
                }
                
                if ($supplier) {
                    $supplier_id = $supplier->id;
                    // Redirect Supplier role users directly to their ledger
                    return redirect()->route('supplier_ledger', $supplier_id);
                } else {
                    // If supplier record not found, show error message
                    Log::warning('Supplier record not found for user: ' . $user->email . ' (ID: ' . $user->id . ')');
                    return redirect()->route('dashboard')->with('error', 'Supplier record not found. Please contact administrator.');
                }
            }
            
            // For non-Supplier roles: show supplier list
            return view('admin.supplier');
        } catch (\Exception $e) {
            // Log error and redirect to dashboard
            Log::error('Error in supplier() method: ' . $e->getMessage());
            return redirect()->route('dashboard')->with('error', 'An error occurred. Please try again.');
        }
    }

    public function show_supplier_ledger($id)
    {
        // If user has Supplier role, validate they can only access their own ledger
        if (Auth::user()->hasRole('Supplier')) {
            $user = Auth::user();
            $supplier = null;
            
            // First try to find by created_by (most reliable match)
            $supplier = Supplier::where('created_by', $user->id)->first();
            
            // If not found, try to find by email match
            if (!$supplier && $user->email) {
                $supplier = Supplier::where('email', $user->email)->first();
            }
            
            // If not found, try to extract username from email and match
            if (!$supplier && $user->email) {
                $emailUsername = strtolower(explode('@', $user->email)[0]);
                $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                
                if (!$supplier) {
                    $suppliers = Supplier::all();
                    foreach ($suppliers as $sup) {
                        $supNameClean = strtolower(preg_replace('/\s+/', '', $sup->sup_name ?? ''));
                        $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                        if ($supNameClean === $emailUsernameClean || 
                            strpos($supNameClean, $emailUsernameClean) !== false ||
                            strpos($emailUsernameClean, $supNameClean) !== false) {
                            $supplier = $sup;
                            break;
                        }
                    }
                }
            }
            
            // If not found by email, try to find by matching name
            if (!$supplier && $user->name) {
                $supplier = Supplier::whereRaw('LOWER(sup_name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
            }
            
            // If still not found, try exact match ignoring case
            if (!$supplier && $user->name) {
                $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) = ?', [strtolower(trim($user->name))])->first();
            }
            
            // If still not found, try partial match
            if (!$supplier && $user->name) {
                $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                $suppliers = Supplier::all();
                foreach ($suppliers as $sup) {
                    $supNameClean = preg_replace('/\s+/', '', strtolower($sup->sup_name ?? ''));
                    if ($userNameClean === $supNameClean ||
                        strpos($supNameClean, $userNameClean) !== false ||
                        strpos($userNameClean, $supNameClean) !== false) {
                        $supplier = $sup;
                        break;
                    }
                    $similarity = 0;
                    similar_text($userNameClean, $supNameClean, $similarity);
                    if ($similarity > 80) {
                        $supplier = $sup;
                        break;
                    }
                }
            }
            
            // Redirect to their own ledger if trying to access another supplier's ledger
            if ($supplier && $supplier->id != $id) {
                return redirect()->route('supplier_ledger', $supplier->id);
            } elseif (!$supplier) {
                return redirect()->route('dashboard')->with('error', 'Supplier record not found. Please contact administrator.');
            }
        }
        
        // For other roles or valid access, show the supplier ledger page
        return view('admin.supplier_ledger');
    }






    // ledger area 
    public function load_supplier_ledger(Request $request)
    {
        try {
            $sup_id = $request->input('sup_id');
            
            // Validate sup_id
            if (empty($sup_id)) {
                return response()->json([
                    'draw' => intval($request->input('draw', 0)),
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Supplier ID is required'
                ], 400);
            }

        $start_date = !empty($request->input('startdate')) ? Carbon::parse($request->input('startdate'))->format('Y-m-d') : "";
        $end_date   = !empty($request->input('startdate')) ? Carbon::parse($request->input('enddate'))->format('Y-m-d') : "";
        $limit      = $request->input('length');
        $offset     = $request->input('start');
        $column     = $request->input('order.0.column');
        $dir        = $request->input('order.0.dir');
        $order_by   = $request->input("columns.$column.data");

            // CRITICAL: If user has Supplier role, FORCE them to only see their own data
            if (Auth::user()->hasRole('Supplier')) {
                // Use the same matching logic as in supplier() method to find current supplier
                $user = Auth::user();
                $supplier = null;
                
                // First try to find by created_by (most reliable match)
                $supplier = Supplier::where('created_by', $user->id)->first();
                
                // If not found, try to find by email match
                if (!$supplier && $user->email) {
                    $supplier = Supplier::where('email', $user->email)->first();
                }
                
                // If not found, try to extract username from email and match
                if (!$supplier && $user->email) {
                    $emailUsername = strtolower(explode('@', $user->email)[0]);
                    $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) LIKE ?', ['%' . $emailUsername . '%'])->first();
                    
                    if (!$supplier) {
                        $suppliers = Supplier::all();
                        foreach ($suppliers as $sup) {
                            $supNameClean = strtolower(preg_replace('/\s+/', '', $sup->sup_name ?? ''));
                            $emailUsernameClean = preg_replace('/\s+/', '', $emailUsername);
                            if ($supNameClean === $emailUsernameClean || 
                                strpos($supNameClean, $emailUsernameClean) !== false ||
                                strpos($emailUsernameClean, $supNameClean) !== false) {
                                $supplier = $sup;
                                break;
                            }
                        }
                    }
                }
                
                // If not found by email, try to find by matching name (case-insensitive)
                if (!$supplier && $user->name) {
                    $supplier = Supplier::whereRaw('LOWER(sup_name) LIKE ?', ['%' . strtolower($user->name) . '%'])->first();
                }
                
                // If still not found, try exact match ignoring case
                if (!$supplier && $user->name) {
                    $supplier = Supplier::whereRaw('LOWER(TRIM(sup_name)) = ?', [strtolower(trim($user->name))])->first();
                }
                
                // If still not found, try partial match (remove spaces and compare with fuzzy matching)
                if (!$supplier && $user->name) {
                    $userNameClean = preg_replace('/\s+/', '', strtolower($user->name));
                    $suppliers = Supplier::all();
                    foreach ($suppliers as $sup) {
                        $supNameClean = preg_replace('/\s+/', '', strtolower($sup->sup_name ?? ''));
                        
                        // Exact match
                        if ($userNameClean === $supNameClean ||
                            strpos($supNameClean, $userNameClean) !== false ||
                            strpos($userNameClean, $supNameClean) !== false) {
                            $supplier = $sup;
                            break;
                        }
                        // Similar match (using similar_text for fuzzy matching)
                        $similarity = 0;
                        similar_text($userNameClean, $supNameClean, $similarity);
                        if ($similarity > 80) { // 80% similarity threshold
                            $supplier = $sup;
                            break;
                        }
                    }
                }
                
                // CRITICAL: Force sup_id to be ONLY the current supplier's ID - ignore any other sup_id sent
                if ($supplier) {
                    $sup_id = $supplier->id; // Always use the supplier's own ID
                    Log::info('Supplier role user ' . $user->email . ' accessing their own ledger (Supplier ID: ' . $sup_id . ')');
                } else {
                    Log::warning('Supplier role user ' . $user->email . ' - supplier record not found');
                    return response()->json([
                        'draw' => intval($request->input('draw', 0)),
                        'recordsTotal' => 0,
                        'recordsFiltered' => 0,
                        'data' => [],
                        'error' => 'Access denied. Supplier record not found.'
                    ], 403);
                }
            }

        // Build the query with potential filters - ALWAYS filter by the specific supplier_id
        $query = DB::table('supplier_ledgers')
            ->where('supplier_ledgers.supplier_id', '=', $sup_id)
            ->select()->whereNull('deleted_at');

        if (!empty($start_date) && !empty($end_date)) {
            $query->whereBetween('date', [$start_date, $end_date]);
        }

        if ($search = $request->input('search')) { // .value if using DataTables
            $query->where('detail', 'like', '%' . $search . '%');
        }
        $type = $request->input('type');
        if (isset($type)) {
            $query->where('transaction_type', '=', $type);
        }

        $totalsQuery = clone $query;
        $totals = $totalsQuery->select([
            DB::raw('SUM(pay) as total_credit'),
            DB::raw('SUM(pending) as total_debit'),
            DB::raw('SUM(bank) as total_bank'),
        ])->first();

        // Calculate total before pagination
        $total_count = $query->count();

        // Apply sorting and pagination
        $query->orderBy($order_by, $dir);
        if ($limit != -1) {
            $query->offset($offset)->limit($limit);
        }

        // Fetch the data
        $supplier_ledger = $query->get();

        $all_data = [];
        $currentBalance = 0;
        foreach ($supplier_ledger as $data) {
            // Set the initial value of $currentBalance
            if ($data->pay > 0) {
                $currentBalance += $data->pending - $data->pay;
            } else if ($data->pending > 0) {
                $currentBalance += $data->pay + $data->pending;
            }

            if (isset($data->bank) && $data->bank > 0) {
                $currentBalance -= $data->bank; // Bank amount bhi add karein (agar payment receive hui hai)
            }        
            // Initialize $transaction_type with a default value
            $transaction_type = '';  // Default empty value
        
            // Check the transaction type and assign the appropriate value to $transaction_type
            if ($data->transaction_type == "Opening Account") {
                $transaction_type = '<div class="label label-table label-purple" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } else if ($data->transaction_type == "Purchase") {
                $transaction_type = '<div class="label label-table label-warning" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } else if ($data->transaction_type == "Return") {
                $transaction_type = '<div class="label label-table label-danger" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            } else if ($data->transaction_type == "Payment") {
                $transaction_type = '<div class="label label-table label-success" style="width:100%; max-width:130px;">' . $data->transaction_type . '</div>';
            }
        
            // Setup the button based on the purchase_id and ret_id
            $btn = "";
            
            // For Supplier role users, hide all action buttons
            if (Auth::user()->hasRole('Supplier')) {
                $btn = '<td class="text-right"></td>';
            }
            // For other roles, show action buttons
            else {
                if ($data->purchase_id != null || $data->purchase_id > 0) {
                    $btn = '<td class="text-right">
                                <button id="view_pur_btn" data-pur_id="' . $data->purchase_id . '" class="btn btn-success btn-icon" data-toggle="tooltip" title="View" style="padding: 6px 10px;"><i class="fa-duotone fa-memo"></i></button>
                            </td>';
                } 
                else if ($data->pur_return_id != null || $data->pur_return_id > 0) {
                    $btn = '<td class="text-right">
                                <button id="view_ret_btn" data-ret_id="' . $data->pur_return_id . '" class="btn btn-success btn-icon" data-toggle="tooltip" title="View" style="padding: 6px 10px;"><i class="fa-duotone fa-memo"></i></button>
                                <button id="edit_btn" data-eid="' . $data->id . '" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit"><i class="fa-duotone fa-pen-to-square"></i></button>
                            </td>';
                } 
                else if ($data->purchase_id == null && $data->pur_return_id == null && $data->transaction_type != "Opening Account") {
                    $btn = '<td class="text-right">
                                <button id="edit_btn" data-eid="' . $data->id . '" class="btn btn-info btn-icon" data-toggle="tooltip" title="Edit"><i class="fa-duotone fa-pen-to-square"></i></button>
                                <button id="delete_btn" data-did="' . $data->id . '" class="btn btn-danger btn-icon" data-toggle="tooltip" title="Delete" style="padding: 6px 10px;"><i class="fa-duotone fa-trash-xmark"></i></button>
                            </td>';
                }
            }
        
            // Append the data for the response
            $all_data[] = [
                'id'           => $data->id,
                'date'          => Carbon::parse($data->date)->format('d-m-Y'),
                'detail'   => $data->detail,
                'transaction_type'     =>  $data->transaction_type,
                'credit'       => number_format($data->pay, 0),
                'debit'        => number_format($data->pending, 0),
                'bank'         => number_format($data->bank, 0),
                'balance'      => number_format($currentBalance, 0),
                'btn'          => $btn
            ];
        };
        
        $sup_name = Supplier::find($sup_id);
        
        if (!$sup_name) {
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Supplier not found.'
            ], 404);
        }

        $total_results = array(
            "total_debit"     => ($totals->total_debit > 0) ? number_format($totals->total_debit): 0,
            "total_credit"    => ($totals->total_credit > 0) ? number_format($totals->total_credit): 0,
            "total_balance"   => number_format($totals->total_debit - $totals->total_credit - $totals->total_bank),
            "supplier_name"   => $sup_name->sup_name ." Ledger (ID: ". $sup_name->id . ")",
        );

        $data = [
            "draw"            => intval($request->input('draw')),
            "recordsTotal"    => $total_count,
            "recordsFiltered" => $total_count,
            "data"            => $all_data,
            "totals"          => $total_results,
        ];

        return response()->json($data);
        } catch (\Exception $e) {
            Log::error('Error in load_supplier_ledger: ' . $e->getMessage());
            return response()->json([
                'draw' => intval($request->input('draw', 0)),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'An error occurred while loading supplier ledger.'
            ], 500);
        }
    }


    public function insert_supplier_payment(Request $request)
    {
          
        // Set credit and debit amounts based on transaction type
        $credit = $request->input('trn_type') == "credit" ? $request->input('amount') : 0;
        $debit = $request->input('trn_type') == "debit" ? $request->input('amount') : 0;
        $bank = $request->input('trn_type') == "bank" ? $request->input('amount') : 0;

        if ($request->trn_type === "credit") {
            // hand_cash column
            DB::table('hand_cashes')->decrement('cash_amount', $request->amount);

        } elseif ($request->trn_type === "bank") {

            DB::table('banks')->where('id', $request->bank_id)
            ->decrement('bank_balance', $request->amount);
        }

    
        $customer_ledger = new SupplierLedger();
        $customer_ledger->supplier_id = $request->input('sup_pay_id');
        $customer_ledger->date = Carbon::parse($request->input('trn_date'))->format('Y-m-d'); // Correct transaction_date field
        $customer_ledger->detail = $request->input('trn_detail') ?: "Payment Received"; // Default to "Payment Received"
        $customer_ledger->transaction_type = $request->input('trn_type');
        $customer_ledger->pay = $credit;
        $customer_ledger->pending = $debit;
        $customer_ledger->bank = $bank;
        $customer_ledger->bank_id = $request->input('bank_id');
        $customer_ledger->created_by = Auth::id();
        $customer_ledger->updated_by = Auth::id();
    
        // Save the entry
        $customer_ledger->save();
    
        return response()->json([
            'status' => 200
        ]);
    }
    
    
    public function edit_supplier_payment($id)
    {
        $Data = SupplierLedger::find($id);

        $Data->date = $Data->date ? Carbon::parse($Data->date)->format('d-m-Y') : null;

        return response()->json([
            'status'=>200,
            'data' => $Data,
        ]);
    }


    // public function update_supplier_payment(Request $request)
    // {
    //     $credit = 0;
    //     $debit = 0;
    //     $bank = 0;

    //     if($request->input('payment_type') == "credit"){
    //         $credit = $request->input('amount');
    //     } else if($request->input('payment_type') == "debit"){
    //         $debit = $request->input('amount');
    //     } else if($request->input('payment_type') == "bank"){
    //         $bank = $request->input('amount');
    //     }

    //     $supplier_ledger = SupplierLedger::find($request->input('pay_id'));
    //     $supplier_ledger->date = Carbon::parse($request->input('payment_date'))->format('Y-m-d');
    //     $supplier_ledger->detail = ($request->input('detail') != "") ? $request->input('detail') : "Payment";
    //     $supplier_ledger->pay = $credit;
    //     $supplier_ledger->pending = $debit;
    //     $supplier_ledger->bank = $bank;
    //     $supplier_ledger->bank_id = $request->input('bank_id');

    //     $supplier_ledger->updated_by = Auth::id();

    //     $supplier_ledger->save();

    //     return response()->json([
    //         'status'=>200
    //     ]);
    // }
    public function update_supplier_payment(Request $request)
    {
        $supplier_ledger = SupplierLedger::find($request->input('pay_id'));

        if (!$supplier_ledger) {
            return response()->json(['status' => 404, 'message' => 'Payment record not found']);
        }

        // 🔹 Step 1: Reverse old payment effect
        // 🔹 Step 1: Reverse old payment effect (use old transaction_type)
        if ($supplier_ledger->transaction_type === "credit") {
            DB::table('hand_cashes')->increment('cash_amount', $supplier_ledger->pay);

        } elseif ($supplier_ledger->transaction_type === "Bank") {
            DB::table('banks')->where('id', $supplier_ledger->bank_id)
                ->increment('bank_balance', $supplier_ledger->bank);
        }

        // 🔹 Step 2: Prepare new values
        $credit = 0;
        $debit  = 0;
        $bank   = 0;

        if ($request->payment_type === "credit") {
            $credit = $request->amount;
        } elseif ($request->payment_type === "debit") {
            $debit = $request->amount;
        } elseif ($request->payment_type === "bank") {
            $bank = $request->amount;
        }

        // 🔹 Step 3: Update supplier ledger
        $supplier_ledger->date             = Carbon::parse($request->payment_date)->format('Y-m-d');
        $supplier_ledger->detail           = $request->detail ?: "Payment";
        $supplier_ledger->transaction_type = $request->payment_type;
        $supplier_ledger->pay              = $credit;
        $supplier_ledger->pending          = $debit;
        $supplier_ledger->bank             = $bank;
        $supplier_ledger->bank_id          = $request->bank_id;
        $supplier_ledger->updated_by       = Auth::id();
        $supplier_ledger->save();

        // 🔹 Step 4: Apply new effect
        if ($request->payment_type === "credit") {
            DB::table('hand_cashes')->decrement('cash_amount', $request->amount);

        } elseif ($request->payment_type === "bank") {
            DB::table('banks')->where('id', $request->bank_id)
                ->decrement('bank_balance', $request->amount);
        }

        return response()->json(['status' => 200, 'message' => 'Supplier payment updated successfully']);
    }



    
    public function delete_supplier_payment($delete_id)
    {
        $supplier_ledger = SupplierLedger::find($delete_id);

        if (!$supplier_ledger) {
            return response()->json(['status' => 404, 'message' => 'Payment record not found']);
        }

        // 🔹 Step 1: Reverse old payment effect before delete
        if ($supplier_ledger->transaction_type === "credit") {
            DB::table('hand_cashes')->increment('cash_amount', $supplier_ledger->pay);

        } elseif ($supplier_ledger->transaction_type === "Bank") {
            DB::table('banks')->where('id', $supplier_ledger->bank_id)
                ->increment('bank_balance', $supplier_ledger->bank);
        }

        // 🔹 Step 2: Delete the payment row
        $supplier_ledger->delete();

        return response()->json(['status' => 200, 'message' => 'Supplier payment deleted successfully']);
    }



}
