<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class SalemanLedger extends Model
{
    use HasFactory, SoftDeletes;

    protected $table = 'saleman_ledgers';

    protected $fillable = [
        'salesman_id',
        'cus_id',
        'sale_id',
        'sale_return_id',
        'order_id',
        'order_return_id',
        'order_booker_id',
        'pro_id',
        'invoice_date',
        'detail',
        'reason',
        'pro_qty',
        'total',
        'debit',
        'credit',
        'bank',
        'return_paid',
        'saleman_cash',
        'bank_id',
        'status',
        'adjustment',
        'transaction_type',
        'created_by',
        'updated_by',
    ];

    /**
     * Get the salesman associated with this ledger entry
     */
    public function salesman()
    {
        return $this->belongsTo(User::class, 'salesman_id');
    }

    /**
     * Get the customer associated with this ledger entry
     */
    public function customer()
    {
        return $this->belongsTo(Customer::class, 'cus_id');
    }

    /**
     * Get the sale associated with this ledger entry
     */
    public function sale()
    {
        return $this->belongsTo(Sale::class, 'sale_id');
    }

    /**
     * Get the sale return associated with this ledger entry
     */
    public function saleReturn()
    {
        return $this->belongsTo(SaleReturn::class, 'sale_return_id');
    }

    /**
     * Get the bank associated with this ledger entry
     */
    public function bank()
    {
        return $this->belongsTo(Bank::class, 'bank_id');
    }

    /**
     * Get the order booker associated with this ledger entry
     */
    public function orderBooker()
    {
        return $this->belongsTo(OrderBooker::class, 'order_booker_id');
    }

    /**
     * Scope for filtering by salesman
     */
    public function scopeForSalesman($query, $salesmanId)
    {
        return $query->where('salesman_id', $salesmanId);
    }

    /**
     * Scope for filtering by date range
     */
    public function scopeDateRange($query, $startDate, $endDate)
    {
        return $query->whereBetween('invoice_date', [$startDate, $endDate]);
    }

    /**
     * Scope for filtering by status
     */
    public function scopeByStatus($query, $status)
    {
        return $query->where('status', $status);
    }

    /**
     * Scope for approved entries
     */
    public function scopeApproved($query)
    {
        return $query->where('status', 'approved');
    }

    /**
     * Scope for pending entries
     */
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    /**
     * Get the created by user
     */
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    /**
     * Get the updated by user
     */
    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }
}