<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        // 1) Fix ENUM duplication first (remove lowercase 'cash')
        if (Schema::hasTable('saleman_ledgers') && Schema::hasColumn('saleman_ledgers', 'transaction_type')) {
            // Replace the ENUM values with a corrected list (no duplicate 'cash').
            // Note: choose the values you want to keep. This list mirrors your table values
            // but removes the duplicate 'cash' (keeps 'Cash').
            DB::statement("
                ALTER TABLE `saleman_ledgers`
                MODIFY `transaction_type` ENUM(
                    'Open_balance',
                    'Sale',
                    'Sale_return',
                    'Payment',
                    'Pay_return',
                    'Order',
                    'Order_return',
                    'Cash',
                    'Bank',
                    'credit',
                    'debit',
                    'pending',
                    'received',
                    'approved',
                    'expense'
                ) NOT NULL DEFAULT 'Order'
            ");
        }

        // 2) Now add any missing columns (idempotent)
        Schema::table('saleman_ledgers', function (Blueprint $table) {
            // Use unsignedBigInteger for *_id fields for consistency if you use unsigned elsewhere.
            if (! Schema::hasColumn('saleman_ledgers', 'total')) {
                $table->decimal('total', 15, 2)->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'bank')) {
                $table->decimal('bank', 15, 2)->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'return_paid')) {
                $table->decimal('return_paid', 15, 2)->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'saleman_cash')) {
                $table->decimal('saleman_cash', 15, 2)->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'bank_id')) {
                $table->unsignedBigInteger('bank_id')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'order_booker_id')) {
                $table->unsignedBigInteger('order_booker_id')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'status')) {
                // originally 'status' was TEXT in your CREATE TABLE; using string is OK if values are short.
                // If you need long text, change to $table->text('status')->nullable();
                $table->text('status')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'adjustment')) {
                // original was TEXT; keep TEXT for parity
                $table->text('adjustment')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'cus_id')) {
                $table->unsignedBigInteger('cus_id')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'created_by')) {
                $table->unsignedBigInteger('created_by')->nullable();
            }

            if (! Schema::hasColumn('saleman_ledgers', 'updated_by')) {
                $table->unsignedBigInteger('updated_by')->nullable();
            }
        });
    }

    /**
     * Reverse the migrations.
     *
     * NOTE: This down() will drop any columns added by this migration if they exist.
     * It WILL NOT revert the ENUM to the previous duplicate-containing definition,
     * because re-introducing the duplicate 'cash' will cause the same ALTER TABLE errors.
     */
    public function down(): void
    {
        Schema::table('saleman_ledgers', function (Blueprint $table) {
            $columns = [
                'total',
                'bank',
                'return_paid',
                'saleman_cash',
                'bank_id',
                'order_booker_id',
                'status',
                'adjustment',
                'cus_id',
                'created_by',
                'updated_by',
            ];

            foreach ($columns as $column) {
                if (Schema::hasColumn('saleman_ledgers', $column)) {
                    // When dropping multiple columns, some DB drivers require separate statements.
                    $table->dropColumn($column);
                }
            }
        });

        // Intentionally not reverting transaction_type to the previous duplicate-containing ENUM.
    }
};