@extends('admin.main_layout')

@section('css_code')
    <link rel="stylesheet" href="{{ asset('backend/my_plugins/timepicker/timepicker.min.css') }}">
    <style>
        .totals-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .totals-card .total-item {
            text-align: center;
            padding: 10px;
        }
        .totals-card .total-label {
            font-size: 12px;
            opacity: 0.9;
            margin-bottom: 5px;
        }
        .totals-card .total-value {
            font-size: 20px;
            font-weight: bold;
        }
        .filter-card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
    </style>
@endsection

@section('page_head')
    <div class="row">
        <div class="col-md-12 table-toolbar-left" style="padding-bottom: 5px;">
            <div id="page-title">
                <h1 class="page-header text-overflow">Order Booker Cashier Report</h1>
            </div>
            <ol class="breadcrumb">
                <li>Reports</li>
                <li class="active">Order Booker Cashier Report</li>
            </ol>
        </div>
    </div>
@endsection

@section('page_content')
    <div class="panel">
        <div class="panel-body">
            <!-- Filter Card -->
            <div class="filter-card">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>From Date</label>
                            <input type="text" id="start_date" class="form-control datepicker" placeholder="From Date" autocomplete="off">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>To Date</label>
                            <input type="text" id="end_date" class="form-control datepicker" placeholder="To Date" autocomplete="off">
                        </div>
                    </div>
                    @if (Auth::user()->hasRole(['Admin', 'Super Admin']))
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>Order Booker</label>
                            <select id="order_booker_filter" class="form-control">
                                <option value="all">All Order Bookers</option>
                                @foreach($order_bookers as $order_booker)
                                    <option value="{{ $order_booker->id }}">{{ $order_booker->name }}</option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    @endif
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>View Type</label>
                            <select id="view_type" class="form-control">
                                <option value="detailed">Detailed</option>
                                <option value="daily">Daily Summary</option>
                                <option value="monthly">Monthly Summary</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-2">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <div>
                                <button id="filter_btn" class="btn btn-primary btn-block">
                                    <i class="fa fa-filter"></i> Apply Filter
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Totals Card -->
            <div class="totals-card" id="totals_card" style="display: none;">
                <div class="row">
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Records</div>
                        <div class="total-value" id="total_records">0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Cash Received</div>
                        <div class="total-value" id="total_cash_received">Rs. 0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Expenses</div>
                        <div class="total-value" id="total_expenses">Rs. 0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Net Amount</div>
                        <div class="total-value" id="net_amount">Rs. 0</div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="row" style="margin-bottom: 15px;">
                <div class="col-md-12 text-right">
                    <button id="print_btn" class="btn btn-info" title="Print">
                        <i class="fa fa-print"></i> Print
                    </button>
                    <button id="excel_btn" class="btn btn-success" title="Export to Excel">
                        <i class="fa fa-file-excel-o"></i> Excel
                    </button>
                </div>
            </div>

            <!-- Data Table -->
            <div class="table-responsive">
                <table class="table table-striped table-hover table-bordered" id="report_table">
                    <thead id="table_head">
                        <!-- Dynamic thead will be inserted here -->
                    </thead>
                    <tbody id="report_tbody">
                        <tr>
                            <td colspan="12" class="text-center">
                                <p style="padding: 50px 0;"><i class="fa fa-spinner fa-spin"></i> Loading report...</p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

@section('javascript_code')
<script src="{{ asset('backend/my_plugins/timepicker/timepicker.min.js') }}"></script>

<script>
    $(document).ready(function() {
        // Initialize date pickers
        $('#start_date').daterangepicker({
            autoUpdateInput: false,
            autoApply: true,
            singleDatePicker: true,
            locale: {
                format: 'DD-MM-YYYY'
            }
        });
        $('#start_date').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('DD-MM-YYYY'));
        });

        $('#end_date').daterangepicker({
            autoUpdateInput: false,
            autoApply: true,
            singleDatePicker: true,
            locale: {
                format: 'DD-MM-YYYY'
            }
        });
        $('#end_date').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('DD-MM-YYYY'));
        });

        // Filter button click
        $('#filter_btn').on('click', function() {
            loadReport();
        });

        // View type change
        $('#view_type').on('change', function() {
            loadReport();
        });

        // Load report on page load
        loadReport();

        // Function to load report
        function loadReport() {
            var startDate = $('#start_date').val();
            var endDate = $('#end_date').val();
            var orderBookerId = $('#order_booker_filter').val();
            var viewType = $('#view_type').val();

            $.ajax({
                url: '{{ route("order_booker_cashier_data") }}',
                type: 'POST',
                data: {
                    _token: '{{ csrf_token() }}',
                    start_date: startDate,
                    end_date: endDate,
                    order_booker_id: orderBookerId,
                    view_type: viewType
                },
                beforeSend: function() {
                    $('#report_tbody').html('<tr><td colspan="12" class="text-center"><i class="fa fa-spinner fa-spin"></i> Loading...</td></tr>');
                },
                success: function(response) {
                    if (response.status === 200) {
                        if (viewType === 'detailed') {
                            renderDetailedView(response);
                        } else {
                            renderSummaryView(response);
                        }
                    }
                },
                error: function() {
                    swal('Error', 'Failed to load report', 'error');
                }
            });
        }

        function renderDetailedView(response) {
            var data = response.data;
            var totals = response.totals;
            var thead = '<tr style="background-color: #f5f5f5;">';
            thead += '<th width="5%">S.No</th>';
            thead += '<th width="10%">Date</th>';
            thead += '<th width="15%">Order Booker</th>';
            thead += '<th width="15%">Transaction Type</th>';
            thead += '<th width="20%">Detail</th>';
            thead += '<th width="10%">Cash Received</th>';
            thead += '<th width="10%">Expense</th>';
            thead += '<th width="10%">Bank</th>';
            thead += '<th width="10%">Status</th>';
            thead += '</tr>';
            $('#table_head').html(thead);

            var html = '';
            if (data.length === 0) {
                html = '<tr><td colspan="9" class="text-center">No records found</td></tr>';
            } else {
                $.each(data, function(index, item) {
                    html += '<tr>';
                    html += '<td>' + item.sno + '</td>';
                    html += '<td>' + item.invoice_date + '</td>';
                    html += '<td>' + item.order_booker_name + '</td>';
                    html += '<td>' + item.transaction_type + '</td>';
                    html += '<td>' + item.detail + '</td>';
                    html += '<td>Rs. ' + item.cash_received + '</td>';
                    html += '<td>Rs. ' + item.expense + '</td>';
                    html += '<td>' + item.bank_name + '</td>';
                    html += '<td>' + item.status + '</td>';
                    html += '</tr>';
                });
            }

            $('#report_tbody').html(html);

            // Update totals
            $('#total_records').text(totals.total_records);
            $('#total_cash_received').text('Rs. ' + totals.total_cash_received);
            $('#total_expenses').text('Rs. ' + totals.total_expenses);
            $('#net_amount').text('Rs. ' + totals.net_amount);
            $('#totals_card').fadeIn();
        }

        function renderSummaryView(response) {
            var summary = response.summary;
            var thead = '<tr style="background-color: #f5f5f5;">';
            thead += '<th width="30%">Period</th>';
            thead += '<th width="15%">Total Transactions</th>';
            thead += '<th width="20%">Total Cash Received</th>';
            thead += '<th width="20%">Total Expenses</th>';
            thead += '<th width="15%">Net Amount</th>';
            thead += '</tr>';
            $('#table_head').html(thead);

            var html = '';
            var grandTotalTransactions = 0;
            var grandTotalCash = 0;
            var grandTotalExpenses = 0;
            var grandNetAmount = 0;

            if (summary.length === 0) {
                html = '<tr><td colspan="5" class="text-center">No records found</td></tr>';
            } else {
                $.each(summary, function(index, item) {
                    var netVal = parseFloat(item.total_cash_received.replace(/,/g, '')) - parseFloat(item.total_expenses.replace(/,/g, ''));
                    grandTotalTransactions += parseInt(item.total_transactions);
                    grandTotalCash += parseFloat(item.total_cash_received.replace(/,/g, ''));
                    grandTotalExpenses += parseFloat(item.total_expenses.replace(/,/g, ''));
                    grandNetAmount += netVal;

                    html += '<tr>';
                    html += '<td>' + item.period + '</td>';
                    html += '<td>' + item.total_transactions + '</td>';
                    html += '<td>Rs. ' + item.total_cash_received + '</td>';
                    html += '<td>Rs. ' + item.total_expenses + '</td>';
                    html += '<td>Rs. ' + item.net_amount + '</td>';
                    html += '</tr>';
                });
            }

            $('#report_tbody').html(html);

            // Update totals
            $('#total_records').text(grandTotalTransactions);
            $('#total_cash_received').text('Rs. ' + grandTotalCash.toFixed(2));
            $('#total_expenses').text('Rs. ' + grandTotalExpenses.toFixed(2));
            $('#net_amount').text('Rs. ' + grandNetAmount.toFixed(2));
            $('#totals_card').fadeIn();
        }

        // Print functionality
        $('#print_btn').on('click', function() {
            window.print();
        });

        // Excel export
        $('#excel_btn').on('click', function() {
            var table = document.getElementById('report_table');
            var html = table.outerHTML;
            var url = 'data:application/vnd.ms-excel,' + encodeURIComponent(html);
            var downloadLink = document.createElement("a");
            downloadLink.href = url;
            downloadLink.download = 'order_booker_cashier_report_' + new Date().getTime() + '.xls';
            document.body.appendChild(downloadLink);
            downloadLink.click();
            document.body.removeChild(downloadLink);
        });
    });
</script>
@endsection
