@extends('admin.main_layout')

@section('css_code')
    <link rel="stylesheet" href="{{ asset('backend/my_plugins/timepicker/timepicker.min.css') }}">
    <style>
        .totals-card {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        .totals-card .total-item {
            text-align: center;
            padding: 10px;
        }
        .totals-card .total-label {
            font-size: 12px;
            opacity: 0.9;
            margin-bottom: 5px;
        }
        .totals-card .total-value {
            font-size: 20px;
            font-weight: bold;
        }
        .filter-card {
            background: #f8f9fa;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
    </style>
@endsection

@section('page_head')
    <div class="row">
        <div class="col-md-12 table-toolbar-left" style="padding-bottom: 5px;">
            <div id="page-title">
                <h1 class="page-header text-overflow">Salesman Ledger Report</h1>
            </div>
            <ol class="breadcrumb">
                <li>Reports</li>
                <li class="active">Salesman Ledger Report</li>
            </ol>
        </div>
    </div>
@endsection

@section('page_content')
    <div class="panel">
        <div class="panel-body">
            <!-- Filter Card -->
            <div class="filter-card">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>From Date</label>
                            <input type="text" id="start_date" class="form-control datepicker" placeholder="From Date" autocomplete="off">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>To Date</label>
                            <input type="text" id="end_date" class="form-control datepicker" placeholder="To Date" autocomplete="off">
                        </div>
                    </div>
                    @if (Auth::user()->hasRole(['Admin', 'Super Admin']))
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>Salesman <span style="color: red;">*</span></label>
                            <select id="salesman_filter" class="form-control" required>
                                <option value="">Select Salesman</option>
                                @foreach($salesmen as $salesman)
                                    <option value="{{ $salesman->id }}">{{ $salesman->name }}</option>
                                @endforeach
                            </select>
                        </div>
                    </div>
                    @endif
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>Transaction Type</label>
                            <select id="transaction_type" class="form-control">
                                <option value="all">All</option>
                                <option value="sale">Sale</option>
                                <option value="sale_return">Sale Return</option>
                                <option value="payment">Payment</option>
                                <option value="order">Order</option>
                                <option value="order_return">Order Return</option>
                                <option value="cash">Cash</option>
                                <option value="bank">Bank</option>
                                <option value="expense">Expense</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label>&nbsp;</label>
                            <div>
                                <button id="filter_btn" class="btn btn-primary btn-block">
                                    <i class="fa fa-filter"></i> Apply Filter
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Totals Card -->
            <div class="totals-card" id="totals_card" style="display: none;">
                <div class="row">
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Debit</div>
                        <div class="total-value" id="total_debit">Rs. 0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Credit</div>
                        <div class="total-value" id="total_credit">Rs. 0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Final Balance</div>
                        <div class="total-value" id="final_balance">Rs. 0</div>
                    </div>
                    <div class="col-md-3 total-item">
                        <div class="total-label">Total Records</div>
                        <div class="total-value" id="total_records">0</div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="row" style="margin-bottom: 15px;">
                <div class="col-md-12 text-right">
                    <button id="print_btn" class="btn btn-info" title="Print">
                        <i class="fa fa-print"></i> Print
                    </button>
                    <button id="excel_btn" class="btn btn-success" title="Export to Excel">
                        <i class="fa fa-file-excel-o"></i> Excel
                    </button>
                    <button id="pdf_btn" class="btn btn-danger" title="Export to PDF">
                        <i class="fa fa-file-pdf-o"></i> PDF
                    </button>
                </div>
            </div>

            <!-- Data Table -->
            <div class="table-responsive">
                <table class="table table-striped table-hover table-bordered" id="salesman_ledger_report_table">
                    <thead>
                        <tr style="background-color: #f5f5f5;">
                            <th width="4%">S.No</th>
                            <th width="8%">Date</th>
                            <th width="10%">Transaction Type</th>
                            <th width="12%">Detail</th>
                            <th width="12%">Customer</th>
                            <th width="12%">Product</th>
                            <th width="5%">Qty</th>
                            <th width="8%">Debit</th>
                            <th width="8%">Credit</th>
                            <th width="8%">Bank Amount</th>
                            <th width="8%">Cash Amount</th>
                            <th width="10%">Bank Name</th>
                            <th width="7%">Status</th>
                            <th width="8%">Balance</th>
                        </tr>
                    </thead>
                    <tbody id="report_tbody">
                        <tr>
                            <td colspan="14" class="text-center">
                                <p style="padding: 50px 0;"><i class="fa fa-info-circle"></i> Please select a salesman to view ledger</p>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
@endsection

@section('javascript_code')
<script src="{{ asset('backend/my_plugins/timepicker/timepicker.min.js') }}"></script>

<script>
    $(document).ready(function() {
        // Initialize date pickers with daterangepicker
        $('#start_date').daterangepicker({
            autoUpdateInput: false,
            autoApply: true,
            singleDatePicker: true,
            locale: {
                format: 'DD-MM-YYYY'
            }
        });
        $('#start_date').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('DD-MM-YYYY'));
        });

        $('#end_date').daterangepicker({
            autoUpdateInput: false,
            autoApply: true,
            singleDatePicker: true,
            locale: {
                format: 'DD-MM-YYYY'
            }
        });
        $('#end_date').on('apply.daterangepicker', function(ev, picker) {
            $(this).val(picker.startDate.format('DD-MM-YYYY'));
        });

        // Leave date fields empty to load all data by default
        // Users can select dates and apply filter to narrow down results

        @if (!Auth::user()->hasRole(['Admin', 'Super Admin']))
        // Auto-load for Sales Man role
        loadReport();
        @endif

        // Filter button click
        $('#filter_btn').on('click', function() {
            @if (Auth::user()->hasRole(['Admin', 'Super Admin']))
            var salesmanId = $('#salesman_filter').val();
            if (!salesmanId) {
                swal('Warning', 'Please select a salesman', 'warning');
                return;
            }
            @endif
            loadReport();
        });

        // Function to load report
        function loadReport() {
            var startDate = $('#start_date').val();
            var endDate = $('#end_date').val();
            var transactionType = $('#transaction_type').val();

            @if (Auth::user()->hasRole(['Admin', 'Super Admin']))
            var salesmanId = $('#salesman_filter').val();
            @else
            var salesmanId = '{{ Auth::id() }}';
            @endif

            $.ajax({
                url: '{{ route("salesman_ledger_data") }}',
                type: 'POST',
                data: {
                    _token: '{{ csrf_token() }}',
                    start_date: startDate,
                    end_date: endDate,
                    salesman_id: salesmanId,
                    transaction_type: transactionType
                },
                beforeSend: function() {
                    $('#report_tbody').html('<tr><td colspan="14" class="text-center"><i class="fa fa-spinner fa-spin"></i> Loading...</td></tr>');
                },
                success: function(response) {
                    if (response.status === 200) {
                        var data = response.data;
                        var totals = response.totals;
                        var html = '';

                        if (data.length === 0) {
                            html = '<tr><td colspan="14" class="text-center">No records found</td></tr>';
                        } else {
                            $.each(data, function(index, item) {
                                html += '<tr>';
                                html += '<td>' + item.sno + '</td>';
                                html += '<td>' + item.transaction_date + '</td>';
                                html += '<td>' + item.transaction_type + '</td>';
                                html += '<td>' + item.detail + '</td>';
                                html += '<td>' + item.customer + '</td>';
                                html += '<td>' + item.product + '</td>';
                                html += '<td>' + item.qty + '</td>';
                                html += '<td>Rs. ' + item.debit + '</td>';
                                html += '<td>Rs. ' + item.credit + '</td>';
                                html += '<td>Rs. ' + item.bank_amount + '</td>';
                                html += '<td>Rs. ' + item.cash_amount + '</td>';
                                html += '<td>' + item.bank_name + '</td>';
                                html += '<td>' + item.status + '</td>';
                                html += '<td>Rs. ' + item.balance + '</td>';
                                html += '</tr>';
                            });
                        }

                        $('#report_tbody').html(html);

                        // Update totals
                        $('#total_debit').text('Rs. ' + totals.total_debit);
                        $('#total_credit').text('Rs. ' + totals.total_credit);
                        $('#final_balance').text('Rs. ' + totals.final_balance);
                        $('#total_records').text(totals.total_records);
                        $('#totals_card').fadeIn();
                    }
                },
                error: function() {
                    swal('Error', 'Failed to load report', 'error');
                }
            });
        }

        // Print functionality
        $('#print_btn').on('click', function() {
            window.print();
        });

        // Excel export (basic implementation)
        $('#excel_btn').on('click', function() {
            var table = document.getElementById('salesman_ledger_report_table');
            var html = table.outerHTML;
            var url = 'data:application/vnd.ms-excel,' + encodeURIComponent(html);
            var downloadLink = document.createElement("a");
            downloadLink.href = url;
            downloadLink.download = 'salesman_ledger_report_' + new Date().getTime() + '.xls';
            document.body.appendChild(downloadLink);
            downloadLink.click();
            document.body.removeChild(downloadLink);
        });

        // PDF export placeholder
        $('#pdf_btn').on('click', function() {
            swal('Info', 'PDF export feature will be implemented with a PDF library', 'info');
        });
    });
</script>
@endsection
